/*
 * This file is part of fabric-loom, licensed under the MIT License (MIT).
 *
 * Copyright (c) 2024 FabricMC
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package net.fabricmc.loom.api.fabricapi;

import java.io.File;

import org.gradle.api.file.RegularFileProperty;
import org.gradle.api.provider.Property;

import net.fabricmc.loom.util.fmj.FabricModJsonFactory;

/**
 * Represents the settings for data generation.
 */
public interface DataGenerationSettings {
	/**
	 * Contains the output directory where generated data files will be stored.
	 */
	RegularFileProperty getOutputDirectory();

	/**
	 * Contains a boolean indicating whether a run configuration should be created for the data generation process.
	 */
	Property<Boolean> getCreateRunConfiguration();

	/**
	 * Contains a boolean property indicating whether a new source set should be created for the data generation process.
	 */
	Property<Boolean> getCreateSourceSet();

	/**
	 * Contains a string property representing the mod ID associated with the data generation process.
	 *
	 * <p>This must be set when {@link #getCreateRunConfiguration()} is set.
	 */
	Property<String> getModId();

	/**
	 * Contains a boolean property indicating whether strict validation is enabled.
	 */
	Property<Boolean> getStrictValidation();

	/**
	 * Contains a boolean property indicating whether the generated resources will be automatically added to the main sourceset.
	 */
	Property<Boolean> getAddToResources();

	/**
	 * Contains a boolean property indicating whether data generation will be compiled and ran with the client.
	 */
	Property<Boolean> getClient();

	/**
	 * Sets {@link #getModId()} property based on the {@code id} field defined in the provided file.
	 */
	default void modId(File fabricModJsonFile) {
		getModId().set(FabricModJsonFactory.createFromFile(fabricModJsonFile).getId());
	}
}
