/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.impl.transfer.item;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;

import org.apache.commons.lang3.math.Fraction;
import net.fabricmc.fabric.api.transfer.v1.context.ContainerItemContext;
import net.fabricmc.fabric.api.transfer.v1.item.ItemVariant;
import net.fabricmc.fabric.api.transfer.v1.storage.Storage;
import net.fabricmc.fabric.api.transfer.v1.storage.StoragePreconditions;
import net.fabricmc.fabric.api.transfer.v1.storage.StorageView;
import net.fabricmc.fabric.api.transfer.v1.transaction.TransactionContext;
import net.fabricmc.fabric.mixin.transfer.BundleContentsAccessor;
import net.minecraft.class_1792;
import net.minecraft.class_1799;
import net.minecraft.class_9276;
import net.minecraft.class_9326;
import net.minecraft.class_9334;

public class BundleContentsStorage implements Storage<ItemVariant> {
	private final ContainerItemContext ctx;
	private final List<BundleSlotWrapper> slotCache = new ArrayList<>();
	private List<StorageView<ItemVariant>> slots = List.of();
	private final class_1792 originalItem;

	public BundleContentsStorage(ContainerItemContext ctx) {
		this.ctx = ctx;
		this.originalItem = ctx.getItemVariant().getItem();
	}

	private boolean updateStack(class_9326 changes, TransactionContext transaction) {
		ItemVariant newVariant = ctx.getItemVariant().withComponentChanges(changes);
		return ctx.exchange(newVariant, 1, transaction) > 0;
	}

	@Override
	public long insert(ItemVariant resource, long maxAmount, TransactionContext transaction) {
		StoragePreconditions.notBlankNotNegative(resource, maxAmount);

		if (!isStillValid()) return 0;

		if (maxAmount > Integer.MAX_VALUE) maxAmount = Integer.MAX_VALUE;

		class_1799 stack = resource.toStack((int) maxAmount);

		if (!class_9276.method_61667(stack)) return 0;

		var builder = new class_9276.class_9277(bundleContents());

		int inserted = builder.method_57432(stack);

		if (inserted == 0) return 0;

		class_9326 changes = class_9326.method_57841()
				.method_57854(class_9334.field_49650, builder.method_57435())
				.method_57852();

		if (!updateStack(changes, transaction)) return 0;

		return inserted;
	}

	@Override
	public long extract(ItemVariant resource, long maxAmount, TransactionContext transaction) {
		StoragePreconditions.notNegative(maxAmount);

		if (!isStillValid()) return 0;

		updateSlotsIfNeeded();

		long amount = 0;

		for (StorageView<ItemVariant> slot : slots) {
			amount += slot.extract(resource, maxAmount - amount, transaction);
			if (amount == maxAmount) break;
		}

		return amount;
	}

	@Override
	public Iterator<StorageView<ItemVariant>> iterator() {
		updateSlotsIfNeeded();

		return slots.iterator();
	}

	private boolean isStillValid() {
		return ctx.getItemVariant().getItem() == originalItem;
	}

	private void updateSlotsIfNeeded() {
		int bundleSize = bundleContents().method_57426();

		if (slots.size() != bundleSize) {
			while (bundleSize > slotCache.size()) {
				slotCache.add(new BundleSlotWrapper(slotCache.size()));
			}

			slots = Collections.unmodifiableList(slotCache.subList(0, bundleSize));
		}
	}

	class_9276 bundleContents() {
		return ctx.getItemVariant().getComponentMap().method_58695(class_9334.field_49650, class_9276.field_49289);
	}

	private class BundleSlotWrapper implements StorageView<ItemVariant> {
		private final int index;

		private BundleSlotWrapper(int index) {
			this.index = index;
		}

		private class_1799 getStack() {
			if (bundleContents().method_57426() <= index) return class_1799.field_8037;

			return ((List<class_1799>) bundleContents().method_57421()).get(index);
		}

		@Override
		public long extract(ItemVariant resource, long maxAmount, TransactionContext transaction) {
			StoragePreconditions.notNegative(maxAmount);

			if (!BundleContentsStorage.this.isStillValid()) return 0;
			if (bundleContents().method_57426() <= index) return 0;
			if (!resource.matches(getStack())) return 0;

			var stacksCopy = new ArrayList<>((Collection<class_1799>) bundleContents().method_59708());

			int extracted = (int) Math.min(stacksCopy.get(index).method_7947(), maxAmount);

			stacksCopy.get(index).method_7934(extracted);
			if (stacksCopy.get(index).method_7960()) stacksCopy.remove(index);

			class_9326 changes = class_9326.method_57841()
					.method_57854(class_9334.field_49650, new class_9276(stacksCopy))
					.method_57852();

			if (!updateStack(changes, transaction)) return 0;

			return extracted;
		}

		@Override
		public boolean isResourceBlank() {
			return getStack().method_7960();
		}

		@Override
		public ItemVariant getResource() {
			return ItemVariant.of(getStack());
		}

		@Override
		public long getAmount() {
			return getStack().method_7947();
		}

		@Override
		public long getCapacity() {
			Fraction remainingSpace = Fraction.ONE.subtract(bundleContents().method_57428());
			int extraAllowed = Math.max(
					remainingSpace.divideBy(BundleContentsAccessor.getOccupancy(getStack())).intValue(),
					0
			);
			return getAmount() + extraAllowed;
		}
	}
}
