/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.api.transfer.v1.fluid;

import com.mojang.serialization.Codec;
import org.jetbrains.annotations.ApiStatus;
import net.fabricmc.fabric.api.transfer.v1.storage.TransferVariant;
import net.fabricmc.fabric.impl.transfer.VariantCodecs;
import net.fabricmc.fabric.impl.transfer.fluid.FluidVariantImpl;
import net.minecraft.class_3611;
import net.minecraft.class_3612;
import net.minecraft.class_6880;
import net.minecraft.class_9129;
import net.minecraft.class_9139;
import net.minecraft.class_9326;

/**
 * An immutable association of a still fluid and data components.
 *
 * <p>Do not extend this class. Use {@link #of(class_3611)} and {@link #of(class_3611, class_9326)} to create instances.
 *
 * <p>{@link net.fabricmc.fabric.api.transfer.v1.client.fluid.FluidVariantRendering} can be used for client-side rendering of fluid variants.
 *
 * <p><b>Fluid variants must always be compared with {@code equals}, never by reference!</b>
 * {@code hashCode} is guaranteed to be correct and constant time independently of the size of the components.
 */
@ApiStatus.NonExtendable
public interface FluidVariant extends TransferVariant<class_3611> {
	Codec<FluidVariant> CODEC = VariantCodecs.FLUID_CODEC;
	class_9139<class_9129, FluidVariant> PACKET_CODEC = VariantCodecs.FLUID_PACKET_CODEC;

	/**
	 * Retrieve a blank FluidVariant.
	 */
	static FluidVariant blank() {
		return of(class_3612.field_15906);
	}

	/**
	 * Retrieve a FluidVariant with a fluid, and a {@code null} tag.
	 *
	 * <p>The flowing and still variations of {@linkplain net.minecraft.class_3609 flowable fluids}
	 * are normalized to always refer to the still variant. For example,
	 * {@code FluidVariant.of(Fluids.FLOWING_WATER).getFluid() == Fluids.WATER}.
	 */
	static FluidVariant of(class_3611 fluid) {
		return of(fluid, class_9326.field_49588);
	}

	/**
	 * Retrieve a FluidVariant with a fluid, and an optional tag.
	 *
	 * <p>The flowing and still variations of {@linkplain net.minecraft.class_3609 flowable fluids}
	 * are normalized to always refer to the still fluid. For example,
	 * {@code FluidVariant.of(Fluids.FLOWING_WATER, ComponentChanges.EMPTY).getFluid() == Fluids.WATER}.
	 */
	static FluidVariant of(class_3611 fluid, class_9326 components) {
		return FluidVariantImpl.of(fluid, components);
	}

	/**
	 * Return the fluid of this variant.
	 */
	default class_3611 getFluid() {
		return getObject();
	}

	default class_6880<class_3611> getRegistryEntry() {
		return getFluid().method_40178();
	}

	/**
	 * Creates a copy of this FluidVariant with the provided component changes applied.
	 * @param changes the changes to apply
	 * @return the new variant with the changes applied
	 */
	@Override
	FluidVariant withComponentChanges(class_9326 changes);
}
