/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.api.transfer.v1.fluid;

import java.util.Optional;

import org.jetbrains.annotations.Nullable;
import net.fabricmc.fabric.api.lookup.v1.custom.ApiProviderMap;
import net.fabricmc.fabric.impl.transfer.TransferApiImpl;
import net.minecraft.class_124;
import net.minecraft.class_1937;
import net.minecraft.class_2246;
import net.minecraft.class_2561;
import net.minecraft.class_2583;
import net.minecraft.class_3414;
import net.minecraft.class_3417;
import net.minecraft.class_3609;
import net.minecraft.class_3611;
import net.minecraft.class_3612;

/**
 * Common fluid variant attributes, accessible both client-side and server-side.
 */
public final class FluidVariantAttributes {
	private static final ApiProviderMap<class_3611, FluidVariantAttributeHandler> HANDLERS = ApiProviderMap.create();
	private static final FluidVariantAttributeHandler DEFAULT_HANDLER = new FluidVariantAttributeHandler() { };
	private static volatile boolean coloredVanillaFluidNames = false;

	private FluidVariantAttributes() {
	}

	/**
	 * Register an attribute handler for the passed fluid.
	 */
	public static void register(class_3611 fluid, FluidVariantAttributeHandler handler) {
		if (HANDLERS.putIfAbsent(fluid, handler) != null) {
			throw new IllegalArgumentException("Duplicate handler registration for fluid " + fluid);
		}
	}

	/**
	 * Enable blue- and red-colored names for water and lava respectively.
	 */
	public static void enableColoredVanillaFluidNames() {
		coloredVanillaFluidNames = true;
	}

	/**
	 * Return the attribute handler for the passed fluid, if available, and {@code null} otherwise.
	 */
	@Nullable
	public static FluidVariantAttributeHandler getHandler(class_3611 fluid) {
		return HANDLERS.get(fluid);
	}

	/**
	 * Return the attribute handler for the passed fluid, if available, or the default instance otherwise.
	 */
	public static FluidVariantAttributeHandler getHandlerOrDefault(class_3611 fluid) {
		FluidVariantAttributeHandler handler = HANDLERS.get(fluid);
		return handler == null ? DEFAULT_HANDLER : handler;
	}

	/**
	 * Return the name that should be used for the passed fluid variant.
	 */
	public static class_2561 getName(FluidVariant variant) {
		return getHandlerOrDefault(variant.getFluid()).getName(variant);
	}

	/**
	 * Return the sound corresponding to a container of this fluid variant being filled if available,
	 * or the default (water) filling sound otherwise.
	 */
	public static class_3414 getFillSound(FluidVariant variant) {
		return getHandlerOrDefault(variant.getFluid()).getFillSound(variant)
				.or(() -> variant.getFluid().method_32359())
				.orElse(class_3417.field_15126);
	}

	/**
	 * Return the sound corresponding to a container of this fluid variant being emptied if available,
	 * or the default (water) emptying sound otherwise.
	 */
	public static class_3414 getEmptySound(FluidVariant variant) {
		return getHandlerOrDefault(variant.getFluid()).getEmptySound(variant).orElse(class_3417.field_14834);
	}

	/**
	 * Return an integer in [0, 15]: the light level emitted by this fluid variant, or 0 if it doesn't naturally emit light.
	 */
	public static int getLuminance(FluidVariant variant) {
		int luminance = getHandlerOrDefault(variant.getFluid()).getLuminance(variant);

		if (luminance < 0 || luminance > 15) {
			TransferApiImpl.LOGGER.warn("Broken FluidVariantAttributeHandler. Invalid luminance %d for fluid variant %s".formatted(luminance, variant));
			return DEFAULT_HANDLER.getLuminance(variant);
		}

		return luminance;
	}

	/**
	 * Return a non-negative integer, representing the temperature of this fluid in Kelvin.
	 * The reference values are {@value FluidConstants#WATER_TEMPERATURE} for water, and {@value FluidConstants#LAVA_TEMPERATURE} for lava.
	 */
	public static int getTemperature(FluidVariant variant) {
		int temperature = getHandlerOrDefault(variant.getFluid()).getTemperature(variant);

		if (temperature < 0) {
			TransferApiImpl.LOGGER.warn("Broken FluidVariantAttributeHandler. Invalid temperature %d for fluid variant %s".formatted(temperature, variant));
			return DEFAULT_HANDLER.getTemperature(variant);
		}

		return temperature;
	}

	/**
	 * Return a positive integer, representing the viscosity of this fluid variant.
	 * Fluids with lower viscosity generally flow faster than fluids with higher viscosity.
	 *
	 * <p>More precisely, viscosity should be {@value FluidConstants#VISCOSITY_RATIO} * {@link class_3609#method_15733} for flowable fluids.
	 * The reference values are {@value FluidConstants#WATER_VISCOSITY} for water,
	 * {@value FluidConstants#LAVA_VISCOSITY_NETHER} for lava in ultrawarm dimensions (such as the nether),
	 * and {@value FluidConstants#LAVA_VISCOSITY} for lava in other dimensions.
	 *
	 * @param world World if available, otherwise null.
	 */
	public static int getViscosity(FluidVariant variant, @Nullable class_1937 world) {
		int viscosity = getHandlerOrDefault(variant.getFluid()).getViscosity(variant, world);

		if (viscosity <= 0) {
			TransferApiImpl.LOGGER.warn("Broken FluidVariantAttributeHandler. Invalid viscosity %d for fluid variant %s".formatted(viscosity, variant));
			return DEFAULT_HANDLER.getViscosity(variant, world);
		}

		return viscosity;
	}

	/**
	 * Return true if this fluid is lighter than air.
	 * Fluids that are lighter than air generally flow upwards.
	 */
	public static boolean isLighterThanAir(FluidVariant variant) {
		return getHandlerOrDefault(variant.getFluid()).isLighterThanAir(variant);
	}

	static {
		register(class_3612.field_15910, new FluidVariantAttributeHandler() {
			@Override
			public class_2561 getName(FluidVariant fluidVariant) {
				if (coloredVanillaFluidNames) {
					return class_2246.field_10382.method_9518().method_10862(class_2583.field_24360.method_10977(class_124.field_1078));
				} else {
					return FluidVariantAttributeHandler.super.getName(fluidVariant);
				}
			}

			@Override
			public Optional<class_3414> getEmptySound(FluidVariant variant) {
				return Optional.of(class_3417.field_14834);
			}
		});
		register(class_3612.field_15908, new FluidVariantAttributeHandler() {
			@Override
			public class_2561 getName(FluidVariant fluidVariant) {
				if (coloredVanillaFluidNames) {
					return class_2246.field_10164.method_9518().method_10862(class_2583.field_24360.method_10977(class_124.field_1061));
				} else {
					return FluidVariantAttributeHandler.super.getName(fluidVariant);
				}
			}

			@Override
			public Optional<class_3414> getFillSound(FluidVariant variant) {
				return Optional.of(class_3417.field_15202);
			}

			@Override
			public Optional<class_3414> getEmptySound(FluidVariant variant) {
				return Optional.of(class_3417.field_15010);
			}

			@Override
			public int getTemperature(FluidVariant variant) {
				return FluidConstants.LAVA_TEMPERATURE;
			}

			@Override
			public int getViscosity(FluidVariant variant, @Nullable class_1937 world) {
				if (world != null && world.method_8597().comp_644()) {
					return FluidConstants.LAVA_VISCOSITY_NETHER;
				} else {
					return FluidConstants.LAVA_VISCOSITY;
				}
			}
		});
	}
}
