/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.impl.tag.client;

import java.io.BufferedReader;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.Set;

import com.google.gson.JsonElement;
import com.mojang.serialization.Dynamic;
import com.mojang.serialization.JsonOps;
import org.jspecify.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import net.fabricmc.loader.api.FabricLoader;
import net.fabricmc.loader.api.ModContainer;
import net.minecraft.class_11341;
import net.minecraft.class_2378;
import net.minecraft.class_2960;
import net.minecraft.class_3497;
import net.minecraft.class_5321;
import net.minecraft.class_6862;
import net.minecraft.class_7475;
import net.minecraft.class_7924;

public class ClientTagsLoader {
	private static final Logger LOGGER = LoggerFactory.getLogger("fabric-client-tags-api-v1");
	/**
	 * Load a given tag from the available mods into a set of {@code Identifier}s.
	 * Parsing based on {@link net.minecraft.class_3503#method_33174(net.minecraft.class_3300)}
	 */
	public static LoadedTag loadTag(class_6862<?> tagKey) {
		var tags = new HashSet<class_3497>();
		HashSet<Path> tagFiles = getTagFiles(tagKey.comp_326(), tagKey.comp_327());

		for (Path tagPath : tagFiles) {
			try (BufferedReader tagReader = Files.newBufferedReader(tagPath)) {
				JsonElement jsonElement = class_11341.method_71359(tagReader);
				class_7475 maybeTagFile = class_7475.field_39269.parse(new Dynamic<>(JsonOps.INSTANCE, jsonElement))
						.result().orElse(null);

				if (maybeTagFile != null) {
					if (maybeTagFile.comp_812()) {
						tags.clear();
					}

					tags.addAll(maybeTagFile.comp_811());
				}
			} catch (IOException e) {
				LOGGER.error("Error loading tag: " + tagKey, e);
			}
		}

		HashSet<class_2960> completeIds = new HashSet<>();
		HashSet<class_2960> immediateChildIds = new HashSet<>();
		HashSet<class_6862<?>> immediateChildTags = new HashSet<>();

		for (class_3497 tagEntry : tags) {
			tagEntry.method_26790(new class_3497.class_7474<>() {
				@Nullable
				@Override
				public class_2960 method_43948(class_2960 id, boolean required) {
					immediateChildIds.add(id);
					return id;
				}

				@Nullable
				@Override
				public Collection<class_2960> method_43949(class_2960 id) {
					class_6862<?> tag = class_6862.method_40092(tagKey.comp_326(), id);
					immediateChildTags.add(tag);
					return ClientTagsImpl.getOrCreatePartiallySyncedTag(tag).completeIds;
				}
			}, completeIds::add);
		}

		// Ensure that the tag does not refer to itself
		immediateChildTags.remove(tagKey);

		return new LoadedTag(Collections.unmodifiableSet(completeIds), Collections.unmodifiableSet(immediateChildTags),
				Collections.unmodifiableSet(immediateChildIds));
	}

	public record LoadedTag(Set<class_2960> completeIds, Set<class_6862<?>> immediateChildTags, Set<class_2960> immediateChildIds) {
	}

	/**
	 * @param registryKey the RegistryKey of the TagKey
	 * @param identifier  the Identifier of the tag
	 * @return the paths to all tag json files within the available mods
	 */
	private static HashSet<Path> getTagFiles(class_5321<? extends class_2378<?>> registryKey, class_2960 identifier) {
		return getTagFiles(class_7924.method_60916(registryKey), identifier);
	}

	/**
	 * @return the paths to all tag json files within the available mods
	 */
	private static HashSet<Path> getTagFiles(String tagType, class_2960 identifier) {
		String tagFile = "data/%s/%s/%s.json".formatted(identifier.method_12836(), tagType, identifier.method_12832());
		return getResourcePaths(tagFile);
	}

	/**
	 * @return all paths from the available mods that match the given internal path
	 */
	private static HashSet<Path> getResourcePaths(String path) {
		HashSet<Path> out = new HashSet<>();

		for (ModContainer mod : FabricLoader.getInstance().getAllMods()) {
			mod.findPath(path).ifPresent(out::add);
		}

		return out;
	}
}
