/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.impl.resource.loader;

import java.util.WeakHashMap;
import net.minecraft.class_3262;
import net.minecraft.class_5352;

/**
 * Tracks the sources of resource packs in a global weak hash map.
 * {@link class_3262} doesn't hold a reference to its {@link class_5352}
 * so we store the source in the map when the resource packs are created.
 * See {@link net.fabricmc.fabric.mixin.resource.loader.PackMixin PackMixin}.
 *
 * <p>The sources are later read for use in {@link FabricResource} and {@link FabricResourceImpl}.
 */
public final class ResourcePackSourceTracker {
	// Use a weak hash map so that if resource packs would be deleted, this won't keep them alive.
	private static final WeakHashMap<class_3262, class_5352> SOURCES = new WeakHashMap<>();

	/**
	 * Gets the source of a pack.
	 *
	 * @param pack the resource pack
	 * @return the source, or {@link class_5352#NONE} if not tracked
	 */
	public static class_5352 getSource(class_3262 pack) {
		return SOURCES.getOrDefault(pack, class_5352.field_25347);
	}

	/**
	 * Sets the source of a pack.
	 *
	 * @param pack the resource pack
	 * @param source the source
	 */
	public static void setSource(class_3262 pack, class_5352 source) {
		SOURCES.put(pack, source);
	}
}
