/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.api.recipe.v1.sync;

import java.util.Collection;
import java.util.Optional;
import java.util.stream.Stream;
import net.minecraft.class_1860;
import net.minecraft.class_1937;
import net.minecraft.class_3956;
import net.minecraft.class_5321;
import net.minecraft.class_8786;
import net.minecraft.class_9695;
import org.jetbrains.annotations.ApiStatus;
import org.jspecify.annotations.Nullable;

/**
 * This class provides access to synchronized recipes on the client.
 *
 * <p>You can access SynchronizedClientRecipes by calling getSynchronizedRecipes
 * method on {@link net.minecraft.class_10286}
 *
 * <p>See {@link RecipeSynchronization}.
 */
@ApiStatus.NonExtendable
public interface SynchronizedRecipes {
	/**
	 * Creates a stream of all recipe entries of the given {@code type} that match the
	 * given {@code input} and {@code world}.
	 *
	 * <p>If {@code input.isEmpty()} returns true, the returned stream will be always empty.
	 *
	 * @return the stream of matching recipes
	 */
	<I extends class_9695, T extends class_1860<I>> Stream<class_8786<T>> getAllMatches(class_3956<T> type, I input, class_1937 world);

	/**
	 * @return the collection of recipe entries of given type
	 */
	<I extends class_9695, T extends class_1860<I>> Collection<class_8786<T>> getAllOfType(class_3956<T> type);

	/**
	 * Finds a first recipe entry (or @{code recipe}, if it matches and isn't null) of the given {@code type} that matches the
	 * given {@code input} and {@code world}.
	 *
	 * @return the optional containing matching recipe entry or empty
	 */
	default <I extends class_9695, T extends class_1860<I>> Optional<class_8786<T>> getFirstMatch(class_3956<T> type, I input, class_1937 world, @Nullable class_5321<class_1860<?>> recipe) {
		class_8786<T> recipeEntry = recipe != null ? this.get(type, recipe) : null;
		return this.getFirstMatch(type, input, world, recipeEntry);
	}

	/**
	 * Finds a first recipe entry (or @{code recipe}, if it matches and isn't null) of the given {@code type} that matches the
	 * given {@code input} and {@code world}.
	 *
	 * @return the optional containing matching recipe entry or empty
	 */
	default <I extends class_9695, T extends class_1860<I>> Optional<class_8786<T>> getFirstMatch(class_3956<T> type, I input, class_1937 world, @Nullable class_8786<T> recipe) {
		return recipe != null && recipe.comp_1933().method_8115(input, world) ? Optional.of(recipe) : this.getFirstMatch(type, input, world);
	}

	/**
	 * Finds a first recipe entry of the given {@code type} that matches the
	 * given {@code input} and {@code world}.
	 *
	 * @return the optional containing matching recipe entry or empty
	 */
	<I extends class_9695, T extends class_1860<I>> Optional<class_8786<T>> getFirstMatch(class_3956<T> type, I input, class_1937 world);

	/**
	 * @return recipe with matching {@code key} or null if not present
	 */
	@Nullable
	class_8786<?> get(class_5321<class_1860<?>> key);

	/**
	 * @return recipe with matching {@code key} of type {@code type} or null if not present
	 */
	@Nullable
	default <T extends class_1860<?>> class_8786<T> get(class_3956<T> type, class_5321<class_1860<?>> key) {
		class_8786<?> recipeEntry = this.get(key);
		//noinspection unchecked
		return recipeEntry != null && recipeEntry.comp_1933().method_17716().equals(type) ? (class_8786<T>) recipeEntry : null;
	}

	/**
	 * @return collection of all synchronized recipe types
	 */
	Collection<class_8786<?>> recipes();
}
