/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.api.particle.v1;

import java.util.function.Function;
import net.minecraft.class_2394;
import net.minecraft.class_2396;
import net.minecraft.class_2400;
import net.minecraft.class_9129;
import net.minecraft.class_9139;
import com.mojang.serialization.MapCodec;

/**
 * Methods for creating particle types, both simple and using an existing attribute factory.
 *
 * <p>Usage:
 * <blockquote>
 * <pre>
 * public static final SimpleParticleType SIMPLE_TEST_PARTICLE = FabricParticleTypes.simple();
 * public static final SimpleParticleType CUSTOM_TEST_PARTICLE = FabricParticleTypes.simple();
 *
 * {@literal @}Override
 * public void onInitialize() {
 *     Registry.register(Registries.PARTICLE_TYPE, Identifier.of("testmod", "simple"), SIMPLE_TEST_PARTICLE);
 *     Registry.register(Registries.PARTICLE_TYPE, Identifier.of("testmod", "custom"), CUSTOM_TEST_PARTICLE);
 * }}
 * </pre>
 * </blockquote>
 */
public final class FabricParticleTypes {
	private FabricParticleTypes() { }

	/**
	 * Creates a new, default particle type for the given id.
	 */
	public static class_2400 simple() {
		return simple(false);
	}

	/**
	 * Creates a new, default particle type for the given id.
	 *
	 * @param alwaysSpawn True to always spawn the particle regardless of distance.
	 */
	public static class_2400 simple(boolean alwaysSpawn) {
		return new class_2400(alwaysSpawn) { };
	}

	/**
	 * Creates a new particle type with a custom factory and codecs for packet/data serialization.
	 *
	 * @param codec The codec for serialization.
	 * @param packetCodec The packet codec for network serialization.
	 */
	public static <T extends class_2394> class_2396<T> complex(final MapCodec<T> codec, final class_9139<? super class_9129, T> packetCodec) {
		return complex(false, codec, packetCodec);
	}

	/**
	 * Creates a new particle type with a custom factory and codecs for packet/data serialization.
	 *
	 * @param alwaysSpawn True to always spawn the particle regardless of distance.
	 * @param codec The codec for serialization.
	 * @param packetCodec The packet codec for network serialization.
	 */
	public static <T extends class_2394> class_2396<T> complex(boolean alwaysSpawn, final MapCodec<T> codec, final class_9139<? super class_9129, T> packetCodec) {
		return new class_2396<>(alwaysSpawn) {
			@Override
			public MapCodec<T> method_29138() {
				return codec;
			}

			@Override
			public class_9139<? super class_9129, T> method_56179() {
				return packetCodec;
			}
		};
	}

	/**
	 * Creates a new particle type with a custom factory and codecs for packet/data serialization.
	 * This method is useful when two different {@link class_2396}s share the same {@link class_2394} implementation.
	 *
	 * @param codecGetter A function that, given the newly created type, returns the codec for serialization.
	 * @param packetCodecGetter A function that, given the newly created type, returns the packet codec for network serialization.
	 */
	public static <T extends class_2394> class_2396<T> complex(final Function<class_2396<T>, MapCodec<T>> codecGetter, final Function<class_2396<T>, class_9139<? super class_9129, T>> packetCodecGetter) {
		return complex(false, codecGetter, packetCodecGetter);
	}

	/**
	 * Creates a new particle type with a custom factory and codecs for packet/data serialization.
	 * This method is useful when two different {@link class_2396}s share the same {@link class_2394} implementation.
	 *
	 * @param alwaysSpawn True to always spawn the particle regardless of distance.
	 * @param codecGetter A function that, given the newly created type, returns the codec for serialization.
	 * @param packetCodecGetter A function that, given the newly created type, returns the packet codec for network serialization.
	 */
	public static <T extends class_2394> class_2396<T> complex(boolean alwaysSpawn, final Function<class_2396<T>, MapCodec<T>> codecGetter, final Function<class_2396<T>, class_9139<? super class_9129, T>> packetCodecGetter) {
		return new class_2396<>(alwaysSpawn) {
			@Override
			public MapCodec<T> method_29138() {
				return codecGetter.apply(this);
			}

			@Override
			public class_9139<? super class_9129, T> method_56179() {
				return packetCodecGetter.apply(this);
			}
		};
	}
}
