/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.mixin.loot;

import java.util.List;
import java.util.Map;
import java.util.WeakHashMap;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.function.Function;

import com.google.gson.JsonElement;
import com.llamalad7.mixinextras.injector.wrapoperation.Operation;
import com.llamalad7.mixinextras.injector.wrapoperation.WrapOperation;
import com.llamalad7.mixinextras.sugar.Local;
import com.mojang.serialization.DynamicOps;
import org.spongepowered.asm.mixin.Mixin;
import org.spongepowered.asm.mixin.Unique;
import org.spongepowered.asm.mixin.injection.At;
import org.spongepowered.asm.mixin.injection.Inject;
import org.spongepowered.asm.mixin.injection.callback.CallbackInfoReturnable;
import net.fabricmc.fabric.api.loot.v3.FabricLootTableBuilder;
import net.fabricmc.fabric.api.loot.v3.LootTableEvents;
import net.fabricmc.fabric.api.loot.v3.LootTableSource;
import net.fabricmc.fabric.impl.loot.FabricLootTable;
import net.fabricmc.fabric.impl.loot.LootUtil;
import net.minecraft.class_2378;
import net.minecraft.class_2385;
import net.minecraft.class_2960;
import net.minecraft.class_3300;
import net.minecraft.class_52;
import net.minecraft.class_5321;
import net.minecraft.class_6903;
import net.minecraft.class_7225;
import net.minecraft.class_7659;
import net.minecraft.class_7780;
import net.minecraft.class_7924;
import net.minecraft.class_8490;
import net.minecraft.class_9383;

/**
 * Implements the events from {@link LootTableEvents}.
 */
@Mixin(class_9383.class)
abstract class ReloadableServerRegistriesMixin {
	/**
	 * Due to possible cross-thread handling, this uses WeakHashMap instead of ThreadLocal.
	 */
	@Unique
	private static final WeakHashMap<class_6903<JsonElement>, class_7225.class_7874> WRAPPERS = new WeakHashMap<>();

	@WrapOperation(method = "reload", at = @At(value = "INVOKE", target = "Lnet/minecraft/core/HolderLookup$Provider;createSerializationContext(Lcom/mojang/serialization/DynamicOps;)Lnet/minecraft/resources/RegistryOps;"))
	private static class_6903<JsonElement> storeOps(class_7225.class_7874 registries, DynamicOps<JsonElement> ops, Operation<class_6903<JsonElement>> original) {
		class_6903<JsonElement> created = original.call(registries, ops);
		WRAPPERS.put(created, registries);
		return created;
	}

	@WrapOperation(method = "reload", at = @At(value = "INVOKE", target = "Ljava/util/concurrent/CompletableFuture;thenApplyAsync(Ljava/util/function/Function;Ljava/util/concurrent/Executor;)Ljava/util/concurrent/CompletableFuture;"))
	private static CompletableFuture<class_7780<class_7659>> removeOps(CompletableFuture<List<class_2385<?>>> future, Function<? super List<class_2385<?>>, ? extends class_7780<class_7659>> fn, Executor executor, Operation<CompletableFuture<class_7780<class_7659>>> original, @Local class_6903<JsonElement> ops) {
		return original.call(future.thenApply(v -> {
			WRAPPERS.remove(ops);
			return v;
		}), fn, executor);
	}

	@Inject(method = "method_61240", at = @At(value = "INVOKE", target = "Ljava/util/Map;forEach(Ljava/util/function/BiConsumer;)V"))
	private static <T> void modifyLootTable(class_8490<T> lootDataType, class_3300 resourceManager, class_6903<JsonElement> registryOps, CallbackInfoReturnable<class_2385<?>> cir, @Local Map<class_2960, T> map) {
		map.replaceAll((identifier, t) -> modifyLootTable(t, identifier, registryOps));
	}

	@Unique
	private static <T> T modifyLootTable(T value, class_2960 id, class_6903<JsonElement> ops) {
		if (!(value instanceof class_52 table)) return value;

		class_5321<class_52> key = class_5321.method_29179(class_7924.field_50079, id);
		// Populated above.
		class_7225.class_7874 registries = WRAPPERS.get(ops);
		// Populated inside SimpleJsonResourceReloadListenerMixin
		LootTableSource source = LootUtil.SOURCES.get().getOrDefault(id, LootTableSource.DATA_PACK);
		// Invoke the REPLACE event for the current loot table.
		class_52 replacement = LootTableEvents.REPLACE.invoker().replaceLootTable(key, table, source, registries);

		if (replacement != null) {
			// Set the loot table to MODIFY to be the replacement loot table.
			// The MODIFY event will also see it as a replaced loot table via the source.
			table = replacement;
			source = LootTableSource.REPLACED;
		}

		// Turn the current table into a modifiable builder and invoke the MODIFY event.
		class_52.class_53 builder = FabricLootTableBuilder.copyOf(table);
		LootTableEvents.MODIFY.invoker().modifyLootTable(key, builder, source, registries);

		return (T) builder.method_338();
	}

	@SuppressWarnings("unchecked")
	@Inject(method = "method_61240", at = @At("RETURN"))
	private static <T> void onLootTablesLoaded(class_8490<T> lootDataType, class_3300 resourceManager, class_6903<JsonElement> registryOps, CallbackInfoReturnable<class_2385<?>> cir) {
		if (lootDataType != class_8490.field_44498) return;

		class_2378<class_52> lootTableRegistry = (class_2378<class_52>) cir.getReturnValue();

		LootTableEvents.ALL_LOADED.invoker().onLootTablesLoaded(resourceManager, lootTableRegistry);
		LootUtil.SOURCES.remove();
		lootTableRegistry.method_42017().forEach(reference -> ((FabricLootTable) reference.comp_349()).fabric$setRegistryEntry(reference));
	}
}
