/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.api.datagen.v1.provider;

import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.function.BiConsumer;

import com.google.common.collect.Sets;
import net.fabricmc.fabric.api.datagen.v1.DataGeneratorEntrypoint;
import net.fabricmc.fabric.api.datagen.v1.FabricDataGenerator;
import net.fabricmc.fabric.api.datagen.v1.FabricDataOutput;
import net.fabricmc.fabric.impl.datagen.loot.FabricLootTableProviderImpl;
import net.minecraft.class_1299;
import net.minecraft.class_173;
import net.minecraft.class_2434;
import net.minecraft.class_2960;
import net.minecraft.class_52;
import net.minecraft.class_5321;
import net.minecraft.class_7225;
import net.minecraft.class_7403;
import net.minecraft.class_7701;
import net.minecraft.class_7789;
import net.minecraft.class_7923;

/**
 * Extend this class and implement {@link FabricEntityLootTableProvider#method_10400()}.
 *
 * <p>Register an instance of this class with {@link FabricDataGenerator.Pack#addProvider} in a
 * {@link DataGeneratorEntrypoint}.
 */
public abstract class FabricEntityLootTableProvider extends class_7789 implements FabricLootTableProvider {
	private final FabricDataOutput output;
	private final Set<class_2960> excludedFromStrictValidation = new HashSet<>();
	private final CompletableFuture<class_7225.class_7874> registryLookupFuture;

	protected FabricEntityLootTableProvider(FabricDataOutput output, CompletableFuture<class_7225.class_7874> registryLookup) {
		super(class_7701.field_40180.method_45383(), registryLookup.join());

		this.output = output;
		this.registryLookupFuture = registryLookup;
	}

	/**
	 * Implement this method to add entity drops.
	 *
	 * <p>Use the {@link class_7789#method_46029} methods to generate entity drops.
	 *
	 * <p>See {@link class_2434#method_10400()} for examples of vanilla entity loot tables.
	 */
	@Override
	public abstract void method_10400();

	/**
	 * Disable strict validation for the given entity type.
	 */
	public void excludeFromStrictValidation(class_1299<?> entityType) {
		this.excludedFromStrictValidation.add(class_7923.field_41177.method_10221(entityType));
	}

	@Override
	public void method_10399(BiConsumer<class_5321<class_52>, class_52.class_53> biConsumer) {
		this.method_10400();

		for (Map<class_5321<class_52>, class_52.class_53> tables : this.field_40615.values()) {
			// Register each of this particular entity type's loot tables
			for (Map.Entry<class_5321<class_52>, class_52.class_53> entry : tables.entrySet()) {
				biConsumer.accept(entry.getKey(), entry.getValue());
			}
		}

		if (this.output.isStrictValidationEnabled()) {
			Set<class_2960> missing = Sets.newHashSet();

			// Find any entity types from this mod that are missing their main loot table
			for (class_2960 entityTypeId : class_7923.field_41177.method_10235()) {
				if (!entityTypeId.method_12836().equals(this.output.getModId())) {
					continue;
				}

				class_1299<?> entityType = class_7923.field_41177.method_63535(entityTypeId);

				entityType.method_16351().ifPresent(mainLootTableKey -> {
					if (!mainLootTableKey.method_29177().method_12836().equals(this.output.getModId())) {
						return;
					}

					Map<class_5321<class_52>, class_52.class_53> tables = this.field_40615.get(entityType);

					if (tables == null || !tables.containsKey(mainLootTableKey)) {
						missing.add(entityTypeId);
					}
				});
			}

			missing.removeAll(this.excludedFromStrictValidation);

			if (!missing.isEmpty()) {
				throw new IllegalStateException("Missing loot table(s) for %s".formatted(missing));
			}
		}
	}

	@Override
	public CompletableFuture<?> method_10319(class_7403 writer) {
		return FabricLootTableProviderImpl.run(writer, this, class_173.field_1173, this.output, this.registryLookupFuture);
	}

	@Override
	public String method_10321() {
		return "Entity Loot Tables";
	}
}
