/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.api.datagen.v1.provider;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.function.Function;

import org.jspecify.annotations.Nullable;
import net.fabricmc.fabric.api.datagen.v1.FabricDataGenerator;
import net.fabricmc.fabric.api.datagen.v1.FabricDataOutput;
import net.minecraft.class_11389;
import net.minecraft.class_1299;
import net.minecraft.class_1792;
import net.minecraft.class_2248;
import net.minecraft.class_2378;
import net.minecraft.class_2474;
import net.minecraft.class_2591;
import net.minecraft.class_2960;
import net.minecraft.class_3495;
import net.minecraft.class_3611;
import net.minecraft.class_5321;
import net.minecraft.class_6862;
import net.minecraft.class_7225;
import net.minecraft.class_7877;
import net.minecraft.class_7924;

/**
 * Implement this class (or one of the inner classes) to generate a tag list.
 *
 * <p>Register your implementation using {@link FabricDataGenerator.Pack#addProvider} in a {@link net.fabricmc.fabric.api.datagen.v1.DataGeneratorEntrypoint}.
 *
 * <p>When generating tags for modded dynamic registry entries (such as biomes), either the entry
 * must be added to the registry using {@link net.fabricmc.fabric.api.datagen.v1.DataGeneratorEntrypoint#buildRegistry(class_7877)}
 * or {@link class_3495#method_34891(class_2960)} must be used. Otherwise, the data generator cannot
 * find the entry and crashes.
 *
 * <p>Commonly used implementations of this class are provided:
 *
 * @see BlockTagProvider
 * @see ItemTagProvider
 * @see FluidTagProvider
 * @see EntityTypeTagProvider
 */
public abstract class FabricTagProvider<T> extends class_2474<T> {
	private final FabricDataOutput output;
	private final Map<class_2960, AliasGroupBuilder> aliasGroupBuilders = new HashMap<>();

	/**
	 * Constructs a new {@link FabricTagProvider} with the default computed path.
	 *
	 * <p>Common implementations of this class are provided.
	 *
	 * @param output        the {@link FabricDataOutput} instance
	 * @param registriesFuture      the backing registry for the tag type
	 */
	public FabricTagProvider(FabricDataOutput output, class_5321<? extends class_2378<T>> registryKey, CompletableFuture<class_7225.class_7874> registriesFuture) {
		super(output, registryKey, registriesFuture);
		this.output = output;
	}

	/**
	 * Implement this method and then use {@link FabricTagProvider#builder} to get and register new tag builders.
	 */
	protected abstract void method_10514(class_7225.class_7874 wrapperLookup);

	protected class_11389<class_5321<T>, T> builder(class_6862<T> tag) {
		class_3495 tagBuilder = this.method_27169(tag);
		return class_11389.method_71552(tagBuilder);
	}

	/**
	 * Gets an {@link AliasGroupBuilder} with the given ID.
	 *
	 * @param groupId the group ID
	 * @return the alias group builder
	 */
	protected AliasGroupBuilder aliasGroup(class_2960 groupId) {
		return aliasGroupBuilders.computeIfAbsent(groupId, key -> new AliasGroupBuilder());
	}

	/**
	 * Gets an {@link AliasGroupBuilder} with the given ID.
	 *
	 * @param group the group name
	 * @return the alias group builder
	 */
	protected AliasGroupBuilder aliasGroup(String group) {
		class_2960 groupId = class_2960.method_60655(output.getModId(), group);
		return aliasGroupBuilders.computeIfAbsent(groupId, key -> new AliasGroupBuilder());
	}

	/**
	 * {@return a read-only map of alias group builders by the alias group ID}.
	 */
	public Map<class_2960, AliasGroupBuilder> getAliasGroupBuilders() {
		return Collections.unmodifiableMap(aliasGroupBuilders);
	}

	/**
	 * Parent class for tag providers that support adding registered values directly.
	 *
	 * @apiNote This class should not be subclassed directly. Either use a subclass provided by
	 * this API, or use the regular {@link FabricTagProvider}. (Ability to add registered values
	 * directly should be considered as deprecated.)
	 */
	public abstract static class FabricValueLookupTagProvider<T> extends FabricTagProvider<T> {
		private final Function<T, class_5321<T>> valueToKey;

		protected FabricValueLookupTagProvider(FabricDataOutput output, class_5321<? extends class_2378<T>> registryKey, CompletableFuture<class_7225.class_7874> registriesFuture, Function<T, class_5321<T>> valueToKey) {
			super(output, registryKey, registriesFuture);
			this.valueToKey = valueToKey;
		}

		protected class_11389<T, T> valueLookupBuilder(class_6862<T> tag) {
			class_3495 tagBuilder = this.method_27169(tag);
			return class_11389.<T>method_71552(tagBuilder).method_71556(this.valueToKey);
		}
	}

	/**
	 * Extend this class to create {@link class_2248} tags in the "/block" tag directory.
	 */
	public abstract static class BlockTagProvider extends FabricValueLookupTagProvider<class_2248> {
		public BlockTagProvider(FabricDataOutput output, CompletableFuture<class_7225.class_7874> registriesFuture) {
			super(output, class_7924.field_41254, registriesFuture, block -> block.method_40142().method_40237());
		}
	}

	/**
	 * Extend this class to create {@link class_2591} tags in the "/block_entity_type" tag directory.
	 */
	public abstract static class BlockEntityTypeTagProvider extends FabricValueLookupTagProvider<class_2591<?>> {
		public BlockEntityTypeTagProvider(FabricDataOutput output, CompletableFuture<class_7225.class_7874> registriesFuture) {
			super(output, class_7924.field_41255, registriesFuture, type -> type.method_53254().method_40237());
		}
	}

	/**
	 * Extend this class to create {@link class_1792} tags in the "/item" tag directory.
	 */
	public abstract static class ItemTagProvider extends FabricValueLookupTagProvider<class_1792> {
		@Nullable
		private final Function<class_6862<class_2248>, class_3495> blockTagBuilderProvider;

		/**
		 * Construct an {@link ItemTagProvider} tag provider <b>with</b> an associated {@link BlockTagProvider} tag provider.
		 *
		 * @param output The {@link FabricDataOutput} instance
		 */
		public ItemTagProvider(FabricDataOutput output, CompletableFuture<class_7225.class_7874> registriesFuture, FabricTagProvider.@Nullable BlockTagProvider blockTagProvider) {
			super(output, class_7924.field_41197, registriesFuture, item -> item.method_40131().method_40237());

			this.blockTagBuilderProvider = blockTagProvider == null ? null : blockTagProvider::method_27169;
		}

		/**
		 * Construct an {@link ItemTagProvider} tag provider <b>without</b> an associated {@link BlockTagProvider} tag provider.
		 *
		 * @param output The {@link FabricDataOutput} instance
		 */
		public ItemTagProvider(FabricDataOutput output, CompletableFuture<class_7225.class_7874> registriesFuture) {
			this(output, registriesFuture, null);
		}

		/**
		 * Copy the entries from a tag with the {@link class_2248} type into this item tag.
		 *
		 * <p>The {@link ItemTagProvider} tag provider must be constructed with an associated {@link BlockTagProvider} tag provider to use this method.
		 *
		 * @param blockTag The block tag to copy from.
		 * @param itemTag  The item tag to copy to.
		 */
		public void copy(class_6862<class_2248> blockTag, class_6862<class_1792> itemTag) {
			class_3495 blockTagBuilder = Objects.requireNonNull(this.blockTagBuilderProvider, "Pass Block tag provider via constructor to use copy").apply(blockTag);
			class_3495 itemTagBuilder = this.method_27169(itemTag);
			blockTagBuilder.method_26782().forEach(itemTagBuilder::method_27064);
		}
	}

	/**
	 * Extend this class to create {@link class_3611} tags in the "/fluid" tag directory.
	 */
	public abstract static class FluidTagProvider extends FabricValueLookupTagProvider<class_3611> {
		public FluidTagProvider(FabricDataOutput output, CompletableFuture<class_7225.class_7874> registriesFuture) {
			super(output, class_7924.field_41270, registriesFuture, fluid -> fluid.method_40178().method_40237());
		}
	}

	/**
	 * Extend this class to create {@link class_1299} tags in the "/entity_type" tag directory.
	 */
	public abstract static class EntityTypeTagProvider extends FabricValueLookupTagProvider<class_1299<?>> {
		public EntityTypeTagProvider(FabricDataOutput output, CompletableFuture<class_7225.class_7874> registriesFuture) {
			super(output, class_7924.field_41266, registriesFuture, type -> type.method_40124().method_40237());
		}
	}

	/**
	 * A builder for tag alias groups.
	 */
	public final class AliasGroupBuilder {
		private final List<class_6862<T>> tags = new ArrayList<>();

		private AliasGroupBuilder() {
		}

		/**
		 * {@return a read-only list of the tags in this alias group}.
		 */
		public List<class_6862<T>> getTags() {
			return Collections.unmodifiableList(tags);
		}

		public AliasGroupBuilder add(class_6862<T> tag) {
			if (tag.comp_326() != field_40957) {
				throw new IllegalArgumentException("Tag " + tag + " isn't from the registry " + field_40957);
			}

			this.tags.add(tag);
			return this;
		}

		@SafeVarargs
		public final AliasGroupBuilder add(class_6862<T>... tags) {
			for (class_6862<T> tag : tags) {
				add(tag);
			}

			return this;
		}

		public AliasGroupBuilder add(class_2960 tag) {
			this.tags.add(class_6862.method_40092(field_40957, tag));
			return this;
		}

		public AliasGroupBuilder add(class_2960... tags) {
			for (class_2960 tag : tags) {
				this.tags.add(class_6862.method_40092(field_40957, tag));
			}

			return this;
		}
	}
}
