/*
 * This file is part of TechReborn, licensed under the MIT License (MIT).
 *
 * Copyright (c) 2020 TechReborn
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package techreborn.utils;

import com.google.common.collect.ImmutableSet;
import org.jetbrains.annotations.Nullable;
import reborncore.common.powerSystem.RcEnergyItem;

import java.util.Collections;
import java.util.HashSet;
import java.util.Set;
import net.minecraft.class_1309;
import net.minecraft.class_1657;
import net.minecraft.class_1799;
import net.minecraft.class_1937;
import net.minecraft.class_2246;
import net.minecraft.class_2338;
import net.minecraft.class_2350;
import net.minecraft.class_239;
import net.minecraft.class_2680;
import net.minecraft.class_3965;

/**
 * @author drcrazy
 */
public class ToolsUtil {
	public static void breakBlock(class_1799 tool, class_1937 world, class_2338 pos, class_1309 entityLiving, int cost) {
		if (!(entityLiving instanceof class_1657)) {
			return;
		}
		class_2680 blockState = world.method_8320(pos);
		if (blockState.method_26214(world, pos) == -1.0F) {
			return;
		}

		if (!((RcEnergyItem) tool.method_7909()).tryUseEnergy(tool, cost)) {
			return;
		}

		blockState.method_26204().method_9556(world, (class_1657) entityLiving, pos, blockState, world.method_8321(pos), tool);
		world.method_8501(pos, class_2246.field_10124.method_9564());
		world.method_8544(pos);
	}

	/**
	 * Fills in set of {@link class_2338} which should be broken by AOE mining
	 *
	 * @param worldIn      {@link class_1937} World reference
	 * @param pos          {@link class_2338} Position of originally broken block
	 * @param entityLiving {@link class_1309} Player who broke block
	 * @param radius       {@code int} Radius of additional blocks to include. E.g. for 3x3 mining radius will be 1
	 * @return {@link Set} Set of {@link class_2338} to process by tool block break logic
	 */
	public static Set<class_2338> getAOEMiningBlocks(class_1937 worldIn, class_2338 pos, @Nullable class_1309 entityLiving, int radius) {
		return getAOEMiningBlocks(worldIn, pos, entityLiving, radius, true);
	}

	/**
	 * Fills in set of {@link class_2338} which should be broken by AOE mining
	 *
	 * @param worldIn          {@link class_1937} World reference
	 * @param pos              {@link class_2338} Position of originally broken block
	 * @param entityLiving     {@link class_1309} Player who broke block
	 * @param radius           {@code int} Radius of additional blocks to include. E.g. for 3x3 mining radius will be 1
	 * @param placeDummyBlocks {@code boolean} Whether to place dummy blocks
	 * @return {@link Set} Set of {@link class_2338} to process by tool block break logic
	 */
	public static Set<class_2338> getAOEMiningBlocks(class_1937 worldIn, class_2338 pos, @Nullable class_1309 entityLiving, int radius, boolean placeDummyBlocks) {
		if (!(entityLiving instanceof class_1657 playerIn)) {
			return ImmutableSet.of();
		}
		Set<class_2338> targetBlocks = new HashSet<>();

		if (placeDummyBlocks) {
			//Put a dirt block down to raytrace with to stop it raytracing past the intended block
			worldIn.method_8501(pos, class_2246.field_10566.method_9564());
		}

		class_239 hitResult = playerIn.method_5745(20D, 0F, false);

		if (placeDummyBlocks) {
			worldIn.method_8501(pos, class_2246.field_10124.method_9564());
		}

		if (!(hitResult instanceof class_3965)) {
			return Collections.emptySet();
		}
		class_2350 direction = ((class_3965) hitResult).method_17780();

		if (direction == class_2350.field_11035 || direction == class_2350.field_11043) {
			for (int x = -radius; x <= radius; x++) {
				for (int y = -1; y <= 1 + (radius - 1) * 2; y++) {
					targetBlocks.add(pos.method_10069(x, y, 0));
				}
			}
		} else if (direction == class_2350.field_11034 || direction == class_2350.field_11039) {
			for (int z = -radius; z <= radius; z++) {
				for (int y = -1; y <= 1 + (radius - 1) * 2; y++) {
					targetBlocks.add(pos.method_10069(0, y, z));
				}
			}
		} else if (direction == class_2350.field_11033 || direction == class_2350.field_11036) {
			class_2350 playerDirection = playerIn.method_5735();
			int minX = 0;
			int maxX = 0;
			int minZ = 0;
			int maxZ = 0;

			switch (playerDirection) {
				case field_11035 -> {
					minZ = -1;
					maxZ = 1 + (radius - 1) * 2;
					minX = -radius;
					maxX = radius;
				}
				case field_11043 -> {
					minZ = -1 - (radius - 1) * 2;
					maxZ = 1;
					minX = -radius;
					maxX = radius;
				}
				case field_11039 -> {
					minZ = -radius;
					maxZ = radius;
					minX = -1 - (radius - 1) * 2;
					maxX = 1;
				}
				case field_11034 -> {
					minZ = -radius;
					maxZ = radius;
					minX = -1;
					maxX = 1 + (radius - 1) * 2;
				}
			}
			for (int x = minX; x <= maxX; x++) {
				for (int z = minZ; z <= maxZ; z++) {
					targetBlocks.add(pos.method_10069(x, 0, z));
				}
			}
		}
		return targetBlocks;
	}
}
