/*
 * This file is part of TechReborn, licensed under the MIT License (MIT).
 *
 * Copyright (c) 2024 TechReborn
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package techreborn.utils;

import reborncore.common.powerSystem.RcEnergyItem;
import techreborn.component.TRDataComponentTypes;

import java.util.List;
import java.util.function.Consumer;
import net.minecraft.class_124;
import net.minecraft.class_1297;
import net.minecraft.class_1799;
import net.minecraft.class_2561;
import net.minecraft.class_3222;

public class TRItemUtils {

	/**
	 * Checks if powered item is active
	 *
	 * @param stack {@link class_1799} Stack to check
	 * @return {@code boolean} True if powered item is active
	 */
	public static boolean isActive(class_1799 stack) {
		return !stack.method_7960() && Boolean.TRUE.equals(stack.method_58694(TRDataComponentTypes.IS_ACTIVE));
	}

	/**
	 * Check if powered item has enough energy to continue being in active state
	 *
	 * @param stack     {@link class_1799} Stack to check
	 * @param cost      {@link int} Cost of operation performed by tool
	 */
	public static void checkActive(class_1799 stack, int cost, class_1297 player) {
		if (!TRItemUtils.isActive(stack)) {
			return;
		}
		if (((RcEnergyItem) stack.method_7909()).getStoredEnergy(stack) >= cost) {
			return;
		}

		if (player instanceof class_3222 serverPlayerEntity) {
			serverPlayerEntity.method_7353(class_2561.method_43471("reborncore.message.energyError")
				.method_27692(class_124.field_1080)
				.method_27693(" ")
				.method_10852(
					class_2561.method_43471("reborncore.message.deactivating")
						.method_27692(class_124.field_1065)
				), true);
		}

		stack.method_57379(TRDataComponentTypes.IS_ACTIVE, false);
	}

	/**
	 * Switch active\inactive state for powered item
	 *
	 * @param stack     {@link class_1799} Stack to switch state
	 * @param cost      {@code int} Cost of operation performed by tool
	 */
	public static void switchActive(class_1799 stack, int cost, class_1297 entity) {
		TRItemUtils.checkActive(stack, cost, entity);

		if (!TRItemUtils.isActive(stack)) {
			stack.method_57379(TRDataComponentTypes.IS_ACTIVE, true);

			if (entity instanceof class_3222 serverPlayerEntity) {
				serverPlayerEntity.method_7353(class_2561.method_43471("reborncore.message.setTo")
					.method_27692(class_124.field_1080)
					.method_27693(" ")
					.method_10852(
						class_2561.method_43471("reborncore.message.active")
							.method_27692(class_124.field_1065)
					), true);
			}
		} else {
			stack.method_57379(TRDataComponentTypes.IS_ACTIVE, false);
			if (entity instanceof class_3222 serverPlayerEntity) {
				serverPlayerEntity.method_7353(class_2561.method_43471("reborncore.message.setTo")
					.method_27692(class_124.field_1080)
					.method_27693(" ")
					.method_10852(
						class_2561.method_43471("reborncore.message.inactive")
							.method_27692(class_124.field_1065)
					), true);
			}
		}
	}

	/**
	 * Adds active\inactive state to powered item tooltip
	 *
	 * @param stack   {@link class_1799} Stack to check
	 * @param tooltip {@link List} List of {@link class_2561} tooltip strings
	 */
	public static void buildActiveTooltip(class_1799 stack, Consumer<class_2561> tooltip) {
		if (!TRItemUtils.isActive(stack)) {
			tooltip.accept(class_2561.method_43471("reborncore.message.inactive").method_27692(class_124.field_1061));
		} else {
			tooltip.accept(class_2561.method_43471("reborncore.message.active").method_27692(class_124.field_1060));
		}
	}
}
