/*
 * This file is part of TechReborn, licensed under the MIT License (MIT).
 *
 * Copyright (c) 2025 TechReborn
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package techreborn.utils;

import java.util.Map;
import java.util.function.Predicate;
import net.minecraft.class_1937;
import net.minecraft.class_2248;
import net.minecraft.class_2338;
import net.minecraft.class_2350;
import net.minecraft.class_2382;
import net.minecraft.class_2680;
import net.minecraft.class_2758;

public class DirectionUtils {
	public enum HORIZONTAL_PART {
		ALONE,
		START,
		MIDDLE,
		END,
	}

	private static final byte FLAG_NORTH = 1;
	private static final byte FLAG_SOUTH = 1 << 1;
	private static final byte FLAG_WEST = 1 << 2;
	private static final byte FLAG_EAST = 1 << 3;

	public static final class_2382[] POSITIONS = new class_2382[]{
		new class_2382(0, 0, -1),
		new class_2382(0, 0, 1),
		new class_2382(-1, 0, 0),
		new class_2382(1, 0, 0),
		new class_2382(0, -1, 0),
		new class_2382(0, 1, 0),
	};
	public static final int HORIZONTAL_LENGTH = 4;
	public static final int ALL_LENGTH = POSITIONS.length;
	public static final int[] FLAGS = new int[ALL_LENGTH];
	public static final int[] OPP_FLAGS = new int[ALL_LENGTH];
	static {
		for (int i = 0; i < ALL_LENGTH; i++) {
			FLAGS[i] = 1 << i;
			if (i % 2 == 1) {
				OPP_FLAGS[i] = FLAGS[i - 1];
				OPP_FLAGS[i - 1] = FLAGS[i];
			}
		}
	}
	public static final Map<class_2350, byte[]> HORIZONTAL_MAP = Map.of(
		class_2350.field_11043, new byte[]{FLAG_WEST, FLAG_WEST | FLAG_EAST, FLAG_EAST},
		class_2350.field_11035, new byte[]{FLAG_EAST, FLAG_WEST | FLAG_EAST, FLAG_WEST},
		class_2350.field_11039, new byte[]{FLAG_SOUTH, FLAG_SOUTH | FLAG_NORTH, FLAG_NORTH},
		class_2350.field_11034, new byte[]{FLAG_NORTH, FLAG_SOUTH | FLAG_NORTH, FLAG_SOUTH}
	);
	public static final class_2758 HORIZONTAL_NEIGHBORS = class_2758.method_11867(
		"neighbors", 0, FLAG_NORTH | FLAG_SOUTH | FLAG_WEST | FLAG_EAST
	);

	private static int addNeighbor(class_1937 world, class_2338 pos, class_2758 property, int length, Predicate<class_2248> predicate) {
		int neighbors = 0;
		class_2338.class_2339 neighborPos = new class_2338.class_2339();
		for (int i = 0; i < length; i++) {
			neighborPos.method_35831(pos, POSITIONS[i]);
			class_2680 neighborState = world.method_8320(neighborPos);
			if (predicate.test(neighborState.method_26204())) {
				int flags = neighborState.method_11654(property) | OPP_FLAGS[i];
				world.method_8501(neighborPos, neighborState.method_11657(property, flags));
				neighbors |= FLAGS[i];
			}
		}
		return neighbors;
	}

	private static void removeNeighbor(class_1937 world, class_2338 pos, class_2680 state, class_2758 property, int length, Predicate<class_2248> predicate) {
		int neighbors = state.method_11654(property);
		if (neighbors != 0) {
			class_2338.class_2339 neighborPos = new class_2338.class_2339();
			for (int i = 0; i < length; i++) {
				if ((neighbors & FLAGS[i]) != 0) {
					neighborPos.method_35831(pos, POSITIONS[i]);
					class_2680 neighborState = world.method_8320(neighborPos);
					if (predicate.test(neighborState.method_26204())) {
						int flags = neighborState.method_11654(property) & ~OPP_FLAGS[i];
						world.method_8501(neighborPos, neighborState.method_11657(property, flags));
					}
				}
			}
		}
	}

	private static void loadNeighbors(class_1937 world, class_2338 pos, class_2680 state, class_2758 property, int length, Predicate<class_2248> predicate) {
		int neighbors = 0;
		class_2338.class_2339 neighborPos = new class_2338.class_2339();
		for (int i = 0; i < length; i++) {
			neighborPos.method_35831(pos, POSITIONS[i]);
			if (predicate.test(world.method_8320(neighborPos).method_26204())) {
				neighbors |= FLAGS[i];
			}
		}
		world.method_8501(pos, state.method_11657(property, neighbors));
	}

	public static void addHorizontalNeighbor(class_1937 world, class_2338 pos, class_2680 state, Predicate<class_2248> predicate) {
		world.method_8501(pos, state.method_11657(HORIZONTAL_NEIGHBORS, addNeighbor(world, pos, HORIZONTAL_NEIGHBORS, HORIZONTAL_LENGTH, predicate)));
	}

	public static void removeHorizontalNeighbor(class_1937 world, class_2338 pos, class_2680 state, Predicate<class_2248> predicate) {
		removeNeighbor(world, pos, state, HORIZONTAL_NEIGHBORS, HORIZONTAL_LENGTH, predicate);
	}

	public static void loadHorizontalNeighbors(class_1937 world, class_2338 pos, class_2680 state, Predicate<class_2248> predicate) {
		loadNeighbors(world, pos, state, HORIZONTAL_NEIGHBORS, HORIZONTAL_LENGTH, predicate);
	}

	public static HORIZONTAL_PART getHorizontalPart(class_2350 direction, int neighbors) {
		byte[] flags = HORIZONTAL_MAP.get(direction);
		if ((neighbors & flags[1]) == flags[1]) return HORIZONTAL_PART.MIDDLE;
		if ((neighbors & flags[0]) == flags[0]) return HORIZONTAL_PART.START;
		if ((neighbors & flags[2]) == flags[2]) return HORIZONTAL_PART.END;
		return HORIZONTAL_PART.ALONE;
	}
}
