/*
 * This file is part of TechReborn, licensed under the MIT License (MIT).
 *
 * Copyright (c) 2020 TechReborn
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package techreborn.items.armor;

import reborncore.api.items.ArmorBlockEntityTicker;
import reborncore.api.items.ArmorRemoveHandler;
import reborncore.common.powerSystem.RcEnergyTier;
import techreborn.config.TechRebornConfig;
import techreborn.init.TRItemSettings;
import techreborn.utils.TRItemUtils;

import java.util.ArrayList;
import java.util.List;
import java.util.function.Consumer;
import net.minecraft.class_10712;
import net.minecraft.class_124;
import net.minecraft.class_1268;
import net.minecraft.class_1269;
import net.minecraft.class_1293;
import net.minecraft.class_1294;
import net.minecraft.class_1304;
import net.minecraft.class_1657;
import net.minecraft.class_1741;
import net.minecraft.class_1799;
import net.minecraft.class_1836;
import net.minecraft.class_1937;
import net.minecraft.class_2561;
import net.minecraft.class_5244;
import net.minecraft.class_8051;
import net.minecraft.class_9285;
import net.minecraft.class_9334;

public class NanoSuitItem extends TREnergyArmourItem implements ArmorBlockEntityTicker, ArmorRemoveHandler {
	private static final class_9285 FULL_SUIT = new AttributeModifierBuilder().armor(10).toughness(4).build();
	private final class_9285 noPowerAttributes;
	private final class_9285 hasPowerAttributes;
	private final class_9285 fullSuitAttributes;

	public NanoSuitItem(class_1741 material, class_8051 slot, String name) {
		super(material, slot, TechRebornConfig.nanoSuitCapacity, RcEnergyTier.HIGH, name);
		switch (slot) {
			case field_41934, field_41937:
				noPowerAttributes = new AttributeModifierBuilder(slot).armor(1).build();
				hasPowerAttributes = new AttributeModifierBuilder(slot).armor(3).toughness(2).knockback(1).build();
				fullSuitAttributes = new AttributeModifierBuilder(slot).armor(5).toughness(3).knockback(1).build();
				break;
			case field_41935:
				noPowerAttributes = new AttributeModifierBuilder(slot).armor(2).build();
				hasPowerAttributes = new AttributeModifierBuilder(slot).armor(6).toughness(2).knockback(1).build();
				fullSuitAttributes = new AttributeModifierBuilder(slot).armor(10).toughness(3).knockback(1).build();
				break;
			case field_41936:
				noPowerAttributes = new AttributeModifierBuilder(slot).armor(3).build();
				hasPowerAttributes = new AttributeModifierBuilder(slot).armor(8).toughness(2).knockback(1).build();
				fullSuitAttributes = new AttributeModifierBuilder(slot).armor(10).toughness(3).knockback(1).build();
				break;
			default:
				throw new IllegalArgumentException("Invalid slot type");
		}
	}

	// TREnergyArmourItem
	@Override
	public long getEnergyMaxOutput(class_1799 stack) { return 0; }

	// ArmorBlockEntityTicker
	@Override
	public void tickArmor(class_1799 stack, boolean hasFullSuit, class_1657 playerEntity) {
		// Night Vision
		if (this.getSlotType() == class_1304.field_6169) {
			if (TRItemUtils.isActive(stack) && tryUseEnergy(stack, TechRebornConfig.suitNightVisionCost)) {
				playerEntity.method_6092(new class_1293(class_1294.field_5925, 220, 1, false, false));
			} else {
				playerEntity.method_6016(class_1294.field_5925);
			}
		}
		applyModifier(stack, hasFullSuit);
	}

	public void applyModifier(class_1799 stack, boolean hasFullSuit) {
		class_9285 attributes = stack.method_58694(class_9334.field_49636);
		if (getStoredEnergy(stack) > 0) {
			if (hasFullSuit) {
				if (attributes != fullSuitAttributes) {
					stack.method_57379(class_9334.field_49636, fullSuitAttributes);
					stack.method_57379(class_9334.field_56400, AttributeModifierBuilder.ATTRIBUTE_HIDE);
				}
			} else if (attributes != hasPowerAttributes) {
				stack.method_57379(class_9334.field_49636, hasPowerAttributes);
				stack.method_57379(class_9334.field_56400, TRItemSettings.UNBREAKABLE_HIDE);
			}
		} else if (attributes != noPowerAttributes) {
			stack.method_57379(class_9334.field_49636, noPowerAttributes);
			stack.method_57379(class_9334.field_56400, TRItemSettings.UNBREAKABLE_HIDE);
		}
	}

	@Override
	public class_1269 method_7836(class_1937 world, class_1657 user, class_1268 hand) {
		class_1799 thisStack = user.method_5998(hand);
		class_1304 slotType = this.getSlotType();
		if (user.method_5715() && slotType == class_1304.field_6169) {
			TRItemUtils.switchActive(thisStack, 1, user);
			return class_1269.field_5812;
		}
		return super.method_7836(world, user, hand);
	}

	@Override
	public void onRemoved(class_1657 playerEntity) {
		playerEntity.method_6016(class_1294.field_5925);
		class_1799 stack = playerEntity.field_7498.method_34255();
		if (stack.method_7909() instanceof NanoSuitItem nanoSuitItem) {
			nanoSuitItem.applyModifier(stack, false);
			stack.method_57381(class_9334.field_49628);
		} else {
			playerEntity.method_31548().method_67533().forEach(itemStack -> {
				if (itemStack.method_7909() instanceof NanoSuitItem nanoSuitItem) {
					nanoSuitItem.applyModifier(itemStack, false);
					itemStack.method_57381(class_9334.field_49628);
				}
			});
		}
	}

	@Override
	public void method_67187(class_1799 stack, class_9635 context, class_10712 displayComponent, Consumer<class_2561> tooltip, class_1836 type) {
		if (this.getSlotType() == class_1304.field_6169) {
			TRItemUtils.buildActiveTooltip(stack, tooltip);
		}
	}

	public void appendArmorTooltip(class_1799 stack, List<class_2561> tooltip, boolean shift) {
		List<class_2561> buffer = new ArrayList<>();
		class_9285 attributes = stack.method_58694(class_9334.field_49636);
		if (AttributeModifierBuilder.equals(attributes, hasPowerAttributes)) {
			if (shift) {
				buffer.add(class_2561.method_43471("item.modifiers.all_equipment").method_27692(class_124.field_1080));
				AttributeModifierBuilder.appendText(buffer, FULL_SUIT, class_124.field_1078);
			}
		} else if (AttributeModifierBuilder.equals(attributes, fullSuitAttributes)) {
			buffer.add(class_5244.field_39003);
			class_1304 slotType = getSlotType();
			buffer.add(AttributeModifierBuilder.text(slotType).method_27692(class_124.field_1080));
			AttributeModifierBuilder.appendText(buffer, shift ? attributes : hasPowerAttributes, class_124.field_1078);
			AttributeModifierBuilder.appendEnchantmentText(buffer, stack, slotType, class_124.field_1078);
			AttributeModifierBuilder.appendArmorEnchantmentText(buffer, stack, class_124.field_1078);
			if (!shift) {
				buffer.add(class_5244.field_39003);
				buffer.add(class_2561.method_43471("item.modifiers.full_suit").method_27692(class_124.field_1054));
				AttributeModifierBuilder.appendText(buffer, FULL_SUIT, class_124.field_1054);
			}
		} else {
			if (!shift && stack.method_57826(class_9334.field_49628)) {
				return;
			}
			buffer.add(class_2561.method_43471("item.modifiers.power").method_27692(class_124.field_1080));
			AttributeModifierBuilder.appendDiffText(buffer, noPowerAttributes, hasPowerAttributes, class_124.field_1078);
			buffer.add(class_2561.method_43471("item.modifiers.all_equipment").method_27692(class_124.field_1080));
			AttributeModifierBuilder.appendText(buffer, FULL_SUIT, class_124.field_1078);
		}
		AttributeModifierBuilder.appendEnd(tooltip, buffer);
	}
}
