/*
 * This file is part of TechReborn, licensed under the MIT License (MIT).
 *
 * Copyright (c) 2020 TechReborn
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package techreborn.items;

import net.fabricmc.fabric.api.transfer.v1.context.ContainerItemContext;
import net.fabricmc.fabric.api.transfer.v1.fluid.FluidConstants;
import net.fabricmc.fabric.api.transfer.v1.fluid.FluidStorage;
import net.fabricmc.fabric.api.transfer.v1.fluid.FluidVariant;
import net.fabricmc.fabric.api.transfer.v1.item.ItemVariant;
import net.fabricmc.fabric.api.transfer.v1.storage.base.SingleVariantItemStorage;
import net.fabricmc.fabric.api.transfer.v1.transaction.TransactionContext;
import net.minecraft.class_1268;
import net.minecraft.class_1269;
import net.minecraft.class_1657;
import net.minecraft.class_1792;
import net.minecraft.class_1799;
import net.minecraft.class_1936;
import net.minecraft.class_1937;
import net.minecraft.class_2263;
import net.minecraft.class_2338;
import net.minecraft.class_2350;
import net.minecraft.class_239;
import net.minecraft.class_2398;
import net.minecraft.class_2402;
import net.minecraft.class_2561;
import net.minecraft.class_2680;
import net.minecraft.class_3218;
import net.minecraft.class_3414;
import net.minecraft.class_3417;
import net.minecraft.class_3419;
import net.minecraft.class_3486;
import net.minecraft.class_3609;
import net.minecraft.class_3611;
import net.minecraft.class_3612;
import net.minecraft.class_3959;
import net.minecraft.class_3965;
import net.minecraft.class_5328;
import net.minecraft.class_5712;
import net.minecraft.class_6880;
import org.apache.commons.lang3.Validate;
import org.jetbrains.annotations.Nullable;
import reborncore.common.fluid.FluidUtils;
import reborncore.common.fluid.container.ItemFluidInfo;
import techreborn.component.TRDataComponentTypes;
import techreborn.init.TRContent;
import techreborn.init.TRItemSettings;

import java.util.Optional;

/**
 * Created by modmuss50 on 17/05/2016.
 */
public class DynamicCellItem extends class_1792 implements ItemFluidInfo {

	public DynamicCellItem(String name) {
		super(TRItemSettings.item(name).method_7889(16).method_57349(TRDataComponentTypes.FLUID, class_3612.field_15906.method_40178()));
	}

	// Thanks vanilla :)
	@SuppressWarnings("deprecation")
	private void playEmptyingSound(@Nullable class_1657 playerEntity, class_1936 world, class_2338 blockPos, class_3611 fluid) {
		class_3414 soundEvent = fluid.method_15791(class_3486.field_15518) ? class_3417.field_15010 : class_3417.field_14834;
		world.method_8396(playerEntity, blockPos, soundEvent, class_3419.field_15245, 1.0F, 1.0F);
	}

	public static class_1799 getCellWithFluid(class_3611 fluid, int stackSize) {
		Validate.notNull(fluid, "Can't get cell with NULL fluid");
		class_1799 stack = new class_1799(TRContent.CELL, stackSize);
		stack.method_57379(TRDataComponentTypes.FLUID, fluid.method_40178());
		stack.method_7939(stackSize);
		return stack;
	}

	public static class_1799 getCellWithFluid(class_3611 fluid) {
		return getCellWithFluid(fluid, 1);
	}

	public static class_1799 getEmptyCell(int amount) {
		return new class_1799(TRContent.CELL, amount);
	}

	private void insertOrDropStack(class_1657 playerEntity, class_3218 world, class_1799 stack) {
		if (!playerEntity.method_31548().method_7394(stack)) {
			playerEntity.method_5775(world, stack);
		}
	}

	public boolean placeFluid(@Nullable class_1657 player, class_1937 world, class_2338 pos, @Nullable class_3965 hitResult, class_1799 filledCell) {
		class_3611 fluid = getFluid(filledCell);
		if (fluid == class_3612.field_15906) {
			return false;
		}

		class_2680 blockState = world.method_8320(pos);
		boolean canPlace = blockState.method_26188(fluid);

		if (!blockState.method_26215() && !canPlace && (!(blockState.method_26204() instanceof class_2402) || !((class_2402) blockState.method_26204()).method_10310(player, world, pos, blockState, fluid))) {
			return hitResult != null && this.placeFluid(player, world, hitResult.method_17777().method_10093(hitResult.method_17780()), null, filledCell);
		} else {
			//noinspection deprecation
			if (world.method_8597().comp_644() && fluid.method_15791(class_3486.field_15517)) {
				int i = pos.method_10263();
				int j = pos.method_10264();
				int k = pos.method_10260();
				world.method_8396(player, pos, class_3417.field_15102, class_3419.field_15245, 0.5F, 2.6F + (world.field_9229.method_43057() - world.field_9229.method_43057()) * 0.8F);

				for (int l = 0; l < 8; ++l) {
					world.method_8406(class_2398.field_11237, (double) i + Math.random(), (double) j + Math.random(), (double) k + Math.random(), 0.0D, 0.0D, 0.0D);
				}
			} else if (blockState.method_26204() instanceof class_2402 && fluid == class_3612.field_15910) {
				if (((class_2402) blockState.method_26204()).method_10311(world, pos, blockState, ((class_3609) fluid).method_15729(false))) {
					this.playEmptyingSound(player, world, pos, fluid);
				}
			} else {
				//noinspection deprecation
				if (!world.method_8608() && canPlace && !blockState.method_51176()) {
					world.method_22352(pos, true);
				}

				this.playEmptyingSound(player, world, pos, fluid);
				world.method_8652(pos, fluid.method_15785().method_15759(), 11);
			}
			return true;
		}
	}

	@Override
	public class_2561 method_7864(class_1799 itemStack) {
		class_3611 fluid = getFluid(itemStack);
		if (fluid != class_3612.field_15906) {
			// TODO use translation keys for fluid and the cell https://fabric.asie.pl/wiki/tutorial:lang?s[]=translation might be useful
			return class_2561.method_43470(class_2561.method_43471("item.techreborn.cell.fluid").getString().replace("$fluid$", FluidUtils.getFluidName(fluid)));
		}
		return super.method_7864(itemStack);
	}

	@Override
	public class_1269 method_7836(class_1937 world, class_1657 player, class_1268 hand) {
		class_1799 stack = player.method_5998(hand);
		class_3611 containedFluid = getFluid(stack);

		class_3965 hitResult = method_7872(world, player, containedFluid == class_3612.field_15906 ? class_3959.class_242.field_1345 : class_3959.class_242.field_1348);
			if (hitResult.method_17783() == class_239.class_240.field_1333 || !(containedFluid instanceof class_3609 || class_3612.field_15906 == containedFluid)) {
			return class_1269.field_5811;
		}
		if (hitResult.method_17783() != class_239.class_240.field_1332) {
			return class_1269.field_5811;
		}

		class_2338 hitPos = hitResult.method_17777();
		if (!world.method_8505(player, hitPos)) {
			return class_1269.field_5814;
		}

		class_2350 side = hitResult.method_17780();
		class_2338 placePos = hitPos.method_10093(side);
		if (!player.method_7343(placePos, side, stack)) {
			return class_1269.field_5814;
		}

		class_2680 hitState = world.method_8320(hitPos);

		if (containedFluid == class_3612.field_15906) {
			if (!(hitState.method_26204() instanceof class_2263 fluidDrainable)) {
				return class_1269.field_5814;
			}
			// This will give us bucket, not a cell
			class_1799 itemStack = fluidDrainable.method_9700(player, world, hitPos, hitState);
			if (!itemStack.method_7960() && itemStack.method_7909() instanceof ItemFluidInfo) {
				class_3611 drainFluid = ((ItemFluidInfo) itemStack.method_7909()).getFluid(itemStack);
				fluidDrainable.method_32351().ifPresent((sound) -> player.method_5783(sound, 1.0F, 1.0F));
				world.method_33596(player, class_5712.field_28167, hitPos);
				// Replace bucket item with cell item
				itemStack = getCellWithFluid(drainFluid, 1);
				class_1799 resultStack = class_5328.method_30270(stack, player, itemStack, false);
				if (resultStack == stack) {
					return class_1269.field_5812;
				} else {
					return class_1269.field_5812.method_61393(resultStack);
				}
			}
		} else {
			placePos = hitState.method_26204() instanceof class_2402 ? hitPos : placePos;
			if (this.placeFluid(player, world, placePos, hitResult, stack)) {

				if (player.method_31549().field_7477) {
					return class_1269.field_5812;
				}

				if (stack.method_7947() == 1) {
					return class_1269.field_5812.method_61393(getEmpty());
				} else {
					stack.method_7934(1);
					if (!world.method_8608()) {
						insertOrDropStack(player, (class_3218) world, getEmpty());
					}

					return class_1269.field_5812;
				}
			}
		}

		return class_1269.field_5814;
	}

	// ItemFluidInfo
	@Override
	public class_1799 getEmpty() {
		return new class_1799(this);
	}

	@Override
	public class_1799 getFull(class_3611 fluid) {
		return getCellWithFluid(fluid);
	}

	@Override
	public class_3611 getFluid(class_1799 itemStack) {
		class_6880<class_3611> fluidEntry = itemStack.method_58695(TRDataComponentTypes.FLUID, class_3612.field_15906.method_40178());
		return fluidEntry.comp_349();
	}

	public void registerFluidApi() {
		FluidStorage.ITEM.registerForItems((stack, ctx) -> new CellStorage(ctx), this);
	}

	public class CellStorage extends SingleVariantItemStorage<FluidVariant> {
		public CellStorage(ContainerItemContext context) {
			super(context);
		}

		@Override
		protected FluidVariant getBlankResource() {
			return FluidVariant.blank();
		}

		@Override
		protected FluidVariant getResource(ItemVariant currentVariant) {
			Optional<? extends class_6880<class_3611>> registryEntry = currentVariant.getComponents().method_57845(TRDataComponentTypes.FLUID);

			if (registryEntry != null && registryEntry.isPresent()) {
				return FluidVariant.of(registryEntry.get().comp_349());
			}

			return FluidVariant.of(class_3612.field_15906);
		}

		@Override
		protected long getAmount(ItemVariant currentVariant) {
			return getResource(currentVariant).isBlank() ? 0 : FluidConstants.BUCKET;
		}

		@Override
		protected long getCapacity(FluidVariant variant) {
			return FluidConstants.BUCKET;
		}

		@Override
		protected ItemVariant getUpdatedVariant(ItemVariant currentVariant, FluidVariant newResource, long newAmount) {
			if (newAmount != 0 && newAmount != FluidConstants.BUCKET) {
				throw new IllegalArgumentException("Only amounts of 0 and 1 bucket are supported! This is a bug!");
			}
			// TODO: this is not ideal since we delete any extra NBT, but it probably doesn't matter in practice?
			if (newResource.isBlank() || newAmount == 0) {
				return ItemVariant.of(DynamicCellItem.this);
			} else {
				return ItemVariant.of(getCellWithFluid(newResource.getFluid()));
			}
		}

		// A few "hacks" to ensure that transfer is always exactly 0 or 1 bucket.
		@Override
		public long insert(FluidVariant insertedResource, long maxAmount, TransactionContext transaction) {
			if (isResourceBlank() && maxAmount >= FluidConstants.BUCKET) {
				return super.insert(insertedResource, FluidConstants.BUCKET, transaction);
			} else {
				return 0;
			}
		}

		@Override
		public long extract(FluidVariant extractedResource, long maxAmount, TransactionContext transaction) {
			if (!isResourceBlank() && maxAmount >= FluidConstants.BUCKET) {
				return super.extract(extractedResource, FluidConstants.BUCKET, transaction);
			} else {
				return 0;
			}
		}
	}
}
