/*
 * This file is part of TechReborn, licensed under the MIT License (MIT).
 *
 * Copyright (c) 2020 TechReborn
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package techreborn.init;

import com.google.common.base.Preconditions;
import net.fabricmc.fabric.api.object.builder.v1.block.type.BlockSetTypeBuilder;
import net.fabricmc.fabric.api.object.builder.v1.block.type.WoodTypeBuilder;
import net.fabricmc.fabric.api.tag.convention.v2.TagUtil;
import net.minecraft.class_1299;
import net.minecraft.class_1311;
import net.minecraft.class_1792;
import net.minecraft.class_1799;
import net.minecraft.class_1802;
import net.minecraft.class_1935;
import net.minecraft.class_2248;
import net.minecraft.class_2378;
import net.minecraft.class_2431;
import net.minecraft.class_2482;
import net.minecraft.class_2510;
import net.minecraft.class_2544;
import net.minecraft.class_2960;
import net.minecraft.class_3545;
import net.minecraft.class_4719;
import net.minecraft.class_4970;
import net.minecraft.class_5321;
import net.minecraft.class_6019;
import net.minecraft.class_6862;
import net.minecraft.class_7923;
import net.minecraft.class_7924;
import net.minecraft.class_8177;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Marker;
import org.slf4j.MarkerFactory;
import reborncore.api.blockentity.IUpgrade;
import reborncore.common.fluid.FluidValue;
import reborncore.common.misc.TagConvertible;
import reborncore.common.powerSystem.PowerAcceptorBlockEntity;
import reborncore.common.powerSystem.RcEnergyTier;
import techreborn.TechReborn;
import techreborn.blockentity.GuiType;
import techreborn.blockentity.generator.LightningRodBlockEntity;
import techreborn.blockentity.generator.PlasmaGeneratorBlockEntity;
import techreborn.blockentity.generator.advanced.*;
import techreborn.blockentity.generator.basic.SolidFuelGeneratorBlockEntity;
import techreborn.blockentity.generator.basic.WaterMillBlockEntity;
import techreborn.blockentity.generator.basic.WindMillBlockEntity;
import techreborn.blockentity.machine.misc.ChargeOMatBlockEntity;
import techreborn.blockentity.machine.misc.DrainBlockEntity;
import techreborn.blockentity.machine.multiblock.*;
import techreborn.blockentity.machine.tier0.block.BlockBreakerBlockEntity;
import techreborn.blockentity.machine.tier0.block.BlockPlacerBlockEntity;
import techreborn.blockentity.machine.tier1.*;
import techreborn.blockentity.machine.tier2.FishingStationBlockEntity;
import techreborn.blockentity.machine.tier2.LaunchpadBlockEntity;
import techreborn.blockentity.machine.tier2.PumpBlockEntity;
import techreborn.blockentity.machine.tier3.ChunkLoaderBlockEntity;
import techreborn.blockentity.machine.tier3.IndustrialCentrifugeBlockEntity;
import techreborn.blockentity.machine.tier3.MatterFabricatorBlockEntity;
import techreborn.blockentity.storage.energy.AdjustableSUBlockEntity;
import techreborn.blocks.GenericMachineBlock;
import techreborn.blocks.cable.CableBlock;
import techreborn.blocks.generator.BlockFusionCoil;
import techreborn.blocks.generator.BlockFusionControlComputer;
import techreborn.blocks.generator.BlockSolarPanel;
import techreborn.blocks.generator.GenericGeneratorBlock;
import techreborn.blocks.lighting.LampBlock;
import techreborn.blocks.machine.tier0.IronAlloyFurnaceBlock;
import techreborn.blocks.machine.tier0.IronFurnaceBlock;
import techreborn.blocks.machine.tier1.PlayerDetectorBlock;
import techreborn.blocks.machine.tier1.ResinBasinBlock;
import techreborn.blocks.misc.*;
import techreborn.blocks.storage.energy.*;
import techreborn.blocks.storage.fluid.TankUnitBlock;
import techreborn.blocks.storage.item.StorageUnitBlock;
import techreborn.blocks.transformers.BlockEVTransformer;
import techreborn.blocks.transformers.BlockHVTransformer;
import techreborn.blocks.transformers.BlockLVTransformer;
import techreborn.blocks.transformers.BlockMVTransformer;
import techreborn.config.TechRebornConfig;
import techreborn.entities.EntityNukePrimed;
import techreborn.events.ModRegistry;
import techreborn.items.DynamicCellItem;
import techreborn.items.UpgradeItem;
import techreborn.items.UpgraderItem;
import techreborn.items.armor.NanoSuitItem;
import techreborn.items.armor.QuantumSuitItem;
import techreborn.utils.InitUtils;
import techreborn.world.OreDistribution;

import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

public class TRContent {
	public static final Marker DATAGEN = MarkerFactory.getMarker("datagen");
	public static final class_8177 RUBBER_WOOD_SET_TYPE = BlockSetTypeBuilder.copyOf(class_8177.field_42823).build(class_2960.method_60655(TechReborn.MOD_ID, "rubber_wood"));
	public static final class_4719 RUBBER_WOOD_TYPE = WoodTypeBuilder.copyOf(class_4719.field_21676).register(class_2960.method_60655(TechReborn.MOD_ID, "rubber_wood"), RUBBER_WOOD_SET_TYPE);

	// Misc Blocks
	public static class_2248 COMPUTER_CUBE;
	public static class_2248 NUKE;
	public static class_2248 REFINED_IRON_FENCE;
	public static class_2248 REINFORCED_GLASS;
	public static class_2248 RUBBER_LEAVES;
	public static class_2248 RUBBER_LOG;
	public static class_2248 RUBBER_SLAB;
	public static class_2248 RUBBER_STAIR;
	public static class_2248 RUBBER_PLANKS;
	public static class_2248 RUBBER_SAPLING;
	public static class_2248 RUBBER_FENCE;
	public static class_2248 RUBBER_FENCE_GATE;
	public static class_2248 RUBBER_TRAPDOOR;
	public static class_2248 RUBBER_BUTTON;
	public static class_2248 RUBBER_PRESSURE_PLATE;
	public static class_2248 RUBBER_DOOR;
	public static class_2248 RUBBER_LOG_STRIPPED;
	public static class_2248 RUBBER_WOOD;
	public static class_2248 STRIPPED_RUBBER_WOOD;
	public static class_2248 POTTED_RUBBER_SAPLING;
	public static class_2248 COPPER_WALL;

	// Armor
	public static class_1792 CLOAKING_DEVICE;
	public static class_1792 LAPOTRONIC_ORBPACK;
	public static class_1792 LITHIUM_ION_BATPACK;

	// Battery
	public static class_1792 ENERGY_CRYSTAL;
	public static class_1792 LAPOTRON_CRYSTAL;
	public static class_1792 LAPOTRONIC_ORB;
	public static class_1792 LITHIUM_ION_BATTERY;
	public static class_1792 RED_CELL_BATTERY;

	// Tools
	public static class_1792 TREE_TAP;
	public static class_1792 WRENCH;
	public static class_1792 PAINTING_TOOL;

	public static class_1792 BASIC_CHAINSAW;
	public static class_1792 BASIC_DRILL;
	public static class_1792 BASIC_JACKHAMMER;
	public static class_1792 ELECTRIC_TREE_TAP;

	public static class_1792 ADVANCED_CHAINSAW;
	public static class_1792 ADVANCED_DRILL;
	public static class_1792 ADVANCED_JACKHAMMER;
	public static class_1792 ROCK_CUTTER;

	public static class_1792 INDUSTRIAL_CHAINSAW;
	public static class_1792 INDUSTRIAL_DRILL;
	public static class_1792 INDUSTRIAL_JACKHAMMER;
	public static class_1792 NANOSABER;
	public static class_1792 OMNI_TOOL;

	public static class_1792 DEBUG_TOOL;

	// Other
	public static class_1792 FREQUENCY_TRANSMITTER;
	public static class_1792 GPS;
	public static class_1792 SCRAP_BOX;
	public static class_1792 MANUAL;
	public static DynamicCellItem CELL;

	//Quantum Suit
	public static QuantumSuitItem QUANTUM_HELMET;
	public static QuantumSuitItem QUANTUM_CHESTPLATE;
	public static QuantumSuitItem QUANTUM_LEGGINGS;
	public static QuantumSuitItem QUANTUM_BOOTS;

	public static NanoSuitItem NANO_HELMET;
	public static NanoSuitItem NANO_CHESTPLATE;
	public static NanoSuitItem NANO_LEGGINGS;
	public static NanoSuitItem NANO_BOOTS;
	// Gem armor & tools
	public static class_1792 BRONZE_SWORD;
	public static class_1792 BRONZE_PICKAXE;
	public static class_1792 BRONZE_SPADE;
	public static class_1792 BRONZE_AXE;
	public static class_1792 BRONZE_HOE;
	public static class_1792 BRONZE_HELMET;
	public static class_1792 BRONZE_CHESTPLATE;
	public static class_1792 BRONZE_LEGGINGS;
	public static class_1792 BRONZE_BOOTS;
	public static class_1792 RUBY_SWORD;
	public static class_1792 RUBY_PICKAXE;
	public static class_1792 RUBY_SPADE;
	public static class_1792 RUBY_AXE;
	public static class_1792 RUBY_HOE;
	public static class_1792 RUBY_HELMET;
	public static class_1792 RUBY_CHESTPLATE;
	public static class_1792 RUBY_LEGGINGS;
	public static class_1792 RUBY_BOOTS;
	public static class_1792 SAPPHIRE_SWORD;
	public static class_1792 SAPPHIRE_PICKAXE;
	public static class_1792 SAPPHIRE_SPADE;
	public static class_1792 SAPPHIRE_AXE;
	public static class_1792 SAPPHIRE_HOE;
	public static class_1792 SAPPHIRE_HELMET;
	public static class_1792 SAPPHIRE_CHESTPLATE;
	public static class_1792 SAPPHIRE_LEGGINGS;
	public static class_1792 SAPPHIRE_BOOTS;
	public static class_1792 PERIDOT_SWORD;
	public static class_1792 PERIDOT_PICKAXE;
	public static class_1792 PERIDOT_SPADE;
	public static class_1792 PERIDOT_AXE;
	public static class_1792 PERIDOT_HOE;
	public static class_1792 PERIDOT_HELMET;
	public static class_1792 PERIDOT_CHESTPLATE;
	public static class_1792 PERIDOT_LEGGINGS;
	public static class_1792 PERIDOT_BOOTS;
	public static class_1792 SILVER_HELMET;
	public static class_1792 SILVER_CHESTPLATE;
	public static class_1792 SILVER_LEGGINGS;
	public static class_1792 SILVER_BOOTS;
	public static class_1792 STEEL_HELMET;
	public static class_1792 STEEL_CHESTPLATE;
	public static class_1792 STEEL_LEGGINGS;
	public static class_1792 STEEL_BOOTS;

	public final static class BlockTags {
		public static final class_6862<class_2248> RUBBER_LOGS = class_6862.method_40092(class_7924.field_41254, class_2960.method_60655(TechReborn.MOD_ID, "rubber_logs"));
		public static final class_6862<class_2248> OMNI_TOOL_MINEABLE = class_6862.method_40092(class_7924.field_41254, class_2960.method_60655(TechReborn.MOD_ID, "mineable/omni_tool"));
		public static final class_6862<class_2248> JACKHAMMER_MINEABLE = class_6862.method_40092(class_7924.field_41254, class_2960.method_60655(TechReborn.MOD_ID, "mineable/jackhammer"));
		public static final class_6862<class_2248> DRILL_MINEABLE = class_6862.method_40092(class_7924.field_41254, class_2960.method_60655(TechReborn.MOD_ID, "mineable/drill"));
		public static final class_6862<class_2248> NONE_SOLID_COVERS = class_6862.method_40092(class_7924.field_41254, class_2960.method_60655(TechReborn.MOD_ID, "none_solid_covers"));

		private BlockTags() {
		}
	}

	public final static class ItemTags {
		public static final class_6862<class_1792> RUBBER_LOGS = class_6862.method_40092(class_7924.field_41197, class_2960.method_60655(TechReborn.MOD_ID, "rubber_logs"));
		public static final class_6862<class_1792> INGOTS = class_6862.method_40092(class_7924.field_41197, class_2960.method_60655(TechReborn.MOD_ID, "ingots"));
		public static final class_6862<class_1792> ORES = class_6862.method_40092(class_7924.field_41197, class_2960.method_60655(TechReborn.MOD_ID, "ores"));
		public static final class_6862<class_1792> STORAGE_BLOCK = class_6862.method_40092(class_7924.field_41197, class_2960.method_60655(TechReborn.MOD_ID, "storage_blocks"));
		public static final class_6862<class_1792> DUSTS = class_6862.method_40092(class_7924.field_41197, class_2960.method_60655(TechReborn.MOD_ID, "dusts"));
		public static final class_6862<class_1792> RAW_METALS = class_6862.method_40092(class_7924.field_41197, class_2960.method_60655(TechReborn.MOD_ID, "raw_metals"));
		public static final class_6862<class_1792> SMALL_DUSTS = class_6862.method_40092(class_7924.field_41197, class_2960.method_60655(TechReborn.MOD_ID, "small_dusts"));
		public static final class_6862<class_1792> GEMS = class_6862.method_40092(class_7924.field_41197, class_2960.method_60655(TechReborn.MOD_ID, "gems"));
		public static final class_6862<class_1792> NUGGETS = class_6862.method_40092(class_7924.field_41197, class_2960.method_60655(TechReborn.MOD_ID, "nuggets"));
		public static final class_6862<class_1792> PLATES = class_6862.method_40092(class_7924.field_41197, class_2960.method_60655(TechReborn.MOD_ID, "plates"));
		public static final class_6862<class_1792> STORAGE_UNITS = class_6862.method_40092(class_7924.field_41197, class_2960.method_60655(TechReborn.MOD_ID, "storage_units"));
		public static final class_6862<class_1792> BRONZE_TOOL_MATERIALS = class_6862.method_40092(class_7924.field_41197, class_2960.method_60655(TagUtil.C_TAG_NAMESPACE, "bronze_tool_materials"));
		public static final class_6862<class_1792> RUBY_TOOL_MATERIALS = class_6862.method_40092(class_7924.field_41197, class_2960.method_60655(TagUtil.C_TAG_NAMESPACE, "ruby_tool_materials"));
		public static final class_6862<class_1792> SAPPHIRE_TOOL_MATERIALS = class_6862.method_40092(class_7924.field_41197, class_2960.method_60655(TagUtil.C_TAG_NAMESPACE, "sapphire_tool_materials"));
		public static final class_6862<class_1792> PERIDOT_TOOL_MATERIALS = class_6862.method_40092(class_7924.field_41197, class_2960.method_60655(TagUtil.C_TAG_NAMESPACE, "peridot_tool_materials"));
		public static final class_6862<class_1792> TRIM_TEMPLATES = class_6862.method_40092(class_7924.field_41197, class_2960.method_60655(TechReborn.MOD_ID, "trim_templates"));
		private ItemTags() {
		}
	}

	public interface ItemInfo extends class_1935 {
		String getName();
	}

	public interface BlockInfo extends class_1935 {
		String getName();
		class_2248 getBlock();
	}

	public interface FamilyBlockInfo extends BlockInfo {
		class_2248 getSlabBlock();
		class_2248 getStairsBlock();
		class_2248 getWallBlock();
	}

	public interface MachineBlockInfo {
		String getName();
		class_2248 getFrame();
		class_2248 getCasing();
	}

	public enum SolarPanels implements BlockInfo {
		BASIC(RcEnergyTier.MICRO, TechRebornConfig.basicGenerationRateD, TechRebornConfig.basicGenerationRateN),
		ADVANCED(RcEnergyTier.LOW, TechRebornConfig.advancedGenerationRateD, TechRebornConfig.advancedGenerationRateN),
		INDUSTRIAL(RcEnergyTier.MEDIUM, TechRebornConfig.industrialGenerationRateD, TechRebornConfig.industrialGenerationRateN),
		ULTIMATE(RcEnergyTier.HIGH, TechRebornConfig.ultimateGenerationRateD, TechRebornConfig.ultimateGenerationRateN),
		QUANTUM(RcEnergyTier.EXTREME, TechRebornConfig.quantumGenerationRateD, TechRebornConfig.quantumGenerationRateN),
		CREATIVE(RcEnergyTier.INFINITE, Integer.MAX_VALUE / 100, Integer.MAX_VALUE / 100);

		public final String name;
		public final class_2248 block;

		// Generation of EU during Day
		public final int generationRateD;
		// Generation of EU during Night
		public final int generationRateN;
		// Internal EU storage of solar panel
		public final int internalCapacity;
		public final RcEnergyTier powerTier;

		SolarPanels(RcEnergyTier tier, int generationRateD, int generationRateN) {
			name = this.toString().toLowerCase(Locale.ROOT);
			powerTier = tier;
			block = new BlockSolarPanel(this, name + "_solar_panel");
			this.generationRateD = generationRateD;
			this.generationRateN = generationRateN;

			internalCapacity = generationRateD * TechRebornConfig.solarInternalCapacityMultiplier;

			InitUtils.setup(block, name + "_solar_panel");
		}

		@Override
		public String getName() {
			return name;
		}

		@Override
		public class_1792 method_8389() {
			return block.method_8389();
		}

		@Override
		public class_2248 getBlock() {
			return block;
		}
	}

	public enum StorageUnit implements BlockInfo {
		BUFFER(1, false),
		CRUDE(TechRebornConfig.crudeStorageUnitMaxStorage, true),
		BASIC(TechRebornConfig.basicStorageUnitMaxStorage, true),
		ADVANCED(TechRebornConfig.advancedStorageUnitMaxStorage, true),
		INDUSTRIAL(TechRebornConfig.industrialStorageUnitMaxStorage, true),
		QUANTUM(TechRebornConfig.quantumStorageUnitMaxStorage, false),
		CREATIVE(Integer.MAX_VALUE, false);

		public final String name;
		public final class_2248 block;
		public final class_1792 upgrader;

		// How many items it can hold
		public final int capacity;


		StorageUnit(int capacity, boolean upgradable) {
			name = this.toString().toLowerCase(Locale.ROOT);
			block = new StorageUnitBlock(this, name.equals("buffer") ? "storage_buffer" : name + "_storage_unit");
			this.capacity = capacity;

			if (name.equals("buffer"))
				InitUtils.setup(block, "storage_buffer");
			else
				InitUtils.setup(block, name + "_storage_unit");
			if (upgradable) {
				if (name.equals("buffer"))
					upgrader = InitUtils.setup(new UpgraderItem("storage_buffer_upgrader"), "storage_buffer_upgrader");
				else
					upgrader = InitUtils.setup(new UpgraderItem(name + "_unit_upgrader"), name + "_unit_upgrader");
			}
			else
				upgrader = null;
		}

		@Override
		public String getName() {
			return name;
		}

		@Override
		public class_2248 getBlock() {
			return block;
		}

		public class_2248 asBlock() {
			return block;
		}

		@Override
		public class_1792 method_8389() {
			return block.method_8389();
		}

		public Optional<class_1792> getUpgrader() {
			return Optional.ofNullable(upgrader);
		}

		public static Optional<StorageUnit> getUpgradableFor(UpgraderItem item) {
			if (item == null)
				return Optional.empty();
			for (StorageUnit unit : StorageUnit.values()) {
				if (item.equals(unit.getUpgrader().orElse(null)))
					return Optional.of(unit);
			}
			return Optional.empty();
		}
	}

	public enum TankUnit implements BlockInfo {
		BASIC(TechRebornConfig.basicTankUnitCapacity),
		ADVANCED(TechRebornConfig.advancedTankUnitMaxStorage),
		INDUSTRIAL(TechRebornConfig.industrialTankUnitCapacity),
		QUANTUM(TechRebornConfig.quantumTankUnitCapacity),
		CREATIVE(Integer.MAX_VALUE / 1000);

		public final String name;
		public final class_2248 block;

		// How many blocks it can hold
		public final FluidValue capacity;


		TankUnit(int capacity) {
			name = this.toString().toLowerCase(Locale.ROOT);
			block = new TankUnitBlock(this, name + "_tank_unit");
			this.capacity = FluidValue.BUCKET.multiply(capacity);

			InitUtils.setup(block, name + "_tank_unit");
		}

		@Override
		public String getName() {
			return name;
		}

		@Override
		public class_2248 getBlock() {
			return block;
		}

		public class_2248 asBlock() {
			return block;
		}

		@Override
		public class_1792 method_8389() {
			return block.method_8389();
		}

		public Optional<class_1792> getUpgrader() {
			try {
				return StorageUnit.valueOf(name()).getUpgrader();
			}
			catch (IllegalArgumentException ex) {
				return Optional.empty();
			}
		}

		public static Optional<TankUnit> getUpgradableFor(UpgraderItem item) {
			if (item == null)
				return Optional.empty();
			for (TankUnit unit : TankUnit.values()) {
				if (item.equals(unit.getUpgrader().orElse(null)))
					return Optional.of(unit);
			}
			return Optional.empty();
		}
	}

	public enum Cables implements BlockInfo {
		COPPER(128, 12.0, true, RcEnergyTier.MEDIUM),
		TIN(32, 12.0, true, RcEnergyTier.LOW),
		GOLD(512, 12.0, true, RcEnergyTier.HIGH),
		HV(2048, 12.0, true, RcEnergyTier.EXTREME),
		GLASSFIBER(8192, 12.0, false, RcEnergyTier.INSANE),
		INSULATED_COPPER(128, 10.0, false, RcEnergyTier.MEDIUM),
		INSULATED_GOLD(512, 10.0, false, RcEnergyTier.HIGH),
		INSULATED_HV(2048, 10.0, false, RcEnergyTier.EXTREME),
		SUPERCONDUCTOR(Integer.MAX_VALUE / 4, 10.0, false, RcEnergyTier.INFINITE);


		public final String name;
		public final CableBlock block;

		public final int transferRate;
		public final int defaultTransferRate;
		public final double cableThickness;
		public final boolean canKill;
		public final boolean defaultCanKill;
		public final RcEnergyTier tier;


		Cables(int transferRate, double cableThickness, boolean canKill, RcEnergyTier tier) {
			name = this.toString().toLowerCase(Locale.ROOT);
			this.transferRate = transferRate;
			this.defaultTransferRate = transferRate;
			this.cableThickness = cableThickness / 2 / 16;
			this.canKill = canKill;
			this.defaultCanKill = canKill;
			this.tier = tier;
			this.block = new CableBlock(this, name + "_cable");
			InitUtils.setup(block, name + "_cable");
		}

		@Override
		public String getName() {
			return name;
		}

		@Override
		public CableBlock getBlock() {
			return block;
		}

		public class_1799 getStack() {
			return new class_1799(block);
		}

		@Override
		public class_1792 method_8389() {
			return block.method_8389();
		}
	}



	private final static Map<Ores, Ores> deepslateMap = new HashMap<>();

	private final static Map<Ores, Ores> unDeepslateMap = new HashMap<>();

	public enum Ores implements BlockInfo, TagConvertible<class_1792> {
		// when changing ores also change data/minecraft/tags/blocks for correct mining level
		BAUXITE(OreDistribution.BAUXITE),
		CINNABAR(OreDistribution.CINNABAR),
		GALENA(OreDistribution.GALENA),
		IRIDIUM(OreDistribution.IRIDIUM, true),
		LEAD(OreDistribution.LEAD),
		PERIDOT(OreDistribution.PERIDOT),
		PYRITE(OreDistribution.PYRITE),
		RUBY(OreDistribution.RUBY),
		SAPPHIRE(OreDistribution.SAPPHIRE),
		SHELDONITE(OreDistribution.SHELDONITE),
		SILVER(OreDistribution.SILVER),
		SODALITE(OreDistribution.SODALITE),
		SPHALERITE(OreDistribution.SPHALERITE),
		TIN(OreDistribution.TIN),
		TUNGSTEN(OreDistribution.TUNGSTEN, true),

		DEEPSLATE_BAUXITE(BAUXITE),
		DEEPSLATE_GALENA(GALENA),
		DEEPSLATE_IRIDIUM(IRIDIUM),
		DEEPSLATE_LEAD(LEAD),
		DEEPSLATE_PERIDOT(PERIDOT),
		DEEPSLATE_RUBY(RUBY),
		DEEPSLATE_SAPPHIRE(SAPPHIRE),
		DEEPSLATE_SHELDONITE(SHELDONITE),
		DEEPSLATE_SILVER(SILVER),
		DEEPSLATE_SODALITE(SODALITE),
		DEEPSLATE_TIN(TIN),
		DEEPSLATE_TUNGSTEN(TUNGSTEN);

		public final String name;
		public final class_2248 block;
		public final OreDistribution distribution;
		private final boolean industrial;
		private final class_6862<class_1792> tag;

		Ores(OreDistribution distribution, class_6019 experienceDroppedFallback, boolean industrial) {
			name = this.toString().toLowerCase(Locale.ROOT);
			block = new class_2431(distribution != null ? distribution.experienceDropped : experienceDroppedFallback, TRBlockSettings.ore(name.startsWith("deepslate"), name + "_ore"));
			this.industrial = industrial;
			InitUtils.setup(block, name + "_ore");
			tag = class_6862.method_40092(class_7924.field_41197, class_2960.method_60655("c", "ores/" +
					(name.startsWith("deepslate_") ? name.substring(name.indexOf('_')+1): name)));
			this.distribution = distribution;
		}

		Ores(OreDistribution distribution, boolean industrial) {
			this(distribution, null, industrial);
		}

		Ores(OreDistribution distribution) {
			this(distribution, false);
		}

		Ores(TRContent.Ores stoneOre) {
			this(null, stoneOre.distribution != null ? stoneOre.distribution.experienceDropped : null, stoneOre.industrial);
			deepslateMap.put(stoneOre, this);
			unDeepslateMap.put(this, stoneOre);
		}

		@Override
		public String getName() {
			return name;
		}

		@Override
		public class_1792 method_8389() {
			return block.method_8389();
		}

		@Override
		public class_2248 getBlock() {
			return block;
		}

		public boolean isIndustrial() {
			return industrial;
		}

		@Override
		public class_6862<class_1792> asTag() {
			return tag;
		}

		@Nullable
		public TRContent.Ores getDeepslate() {
			Preconditions.checkArgument(!isDeepslate());
			return deepslateMap.get(this);
		}

		public TRContent.Ores getUnDeepslate() {
			Preconditions.checkArgument(isDeepslate());
			return unDeepslateMap.get(this);
		}

		public boolean isDeepslate() {
			return name.startsWith("deepslate_");
		}
	}

	/**
	 * The base tag name for chrome items. "chromium" is proper spelling, but changing the enum constant names or
	 * directly registry names will result in item loss upon updating. Hence, only the base tag is changed, where needed.
	 */
	public static final String CHROME_TAG_NAME_BASE = "chromium";

	public enum StorageBlocks implements FamilyBlockInfo, TagConvertible<class_1792> {
		ADVANCED_ALLOY(5f, 6f),
		ALUMINUM(),
		BRASS(),
		BRONZE(5f, 6f),
		CHROME(false, 5f, 6f, CHROME_TAG_NAME_BASE),
		ELECTRUM(),
		HOT_TUNGSTENSTEEL(true, 5f, 6f),
		INVAR(),
		IRIDIUM(5f, 6f),
		IRIDIUM_REINFORCED_STONE(30f, 800f),
		IRIDIUM_REINFORCED_TUNGSTENSTEEL(50f, 1200f),
		LEAD(),
		NICKEL(5f, 6f),
		PERIDOT(5f, 6f),
		PLATINUM(5f, 6f),
		RAW_IRIDIUM(2f, 2f),
		RAW_LEAD(2f, 2f),
		RAW_SILVER(2f, 2f),
		RAW_TIN(2f, 2f),
		RAW_TUNGSTEN(2f, 2f),
		RED_GARNET(5f, 6f),
		REFINED_IRON(5f, 6f),
		RUBY(5f, 6f),
		SAPPHIRE(5f, 6f),
		SILVER(5f, 6f),
		STEEL(5f, 6f),
		TIN(),
		TITANIUM(5f, 6f),
		TUNGSTEN(5f, 6f),
		TUNGSTENSTEEL(30f, 800f),
		YELLOW_GARNET(5f, 6f),
		ZINC(5f, 6f);

		public final String name;
		private final class_2248 block;
		private final class_2510 stairsBlock;
		private final class_2482 slabBlock;
		private final class_2544 wallBlock;
		private final class_6862<class_1792> tag;

		StorageBlocks(boolean isHot, float hardness, float resistance, String tagNameBase) {
			name = this.toString().toLowerCase(Locale.ROOT);
			block = new BlockStorage(isHot, hardness, resistance, name + "_storage_block");
			InitUtils.setup(block, name + "_storage_block");
			tag = class_6862.method_40092(class_7924.field_41197, class_2960.method_60655("c", "storage_blocks/" + Objects.requireNonNullElse(tagNameBase, name)));

			stairsBlock = new TechRebornStairsBlock(block.method_9564(), class_4970.class_2251.method_9630(block).method_63500(TRBlockSettings.key(name + "_storage_block_stairs")));
			InitUtils.setup(stairsBlock, name + "_storage_block_stairs");

			slabBlock = new class_2482(class_4970.class_2251.method_9630(block).method_63500(TRBlockSettings.key(name + "_storage_block_slab")));
			InitUtils.setup(slabBlock, name + "_storage_block_slab");

			wallBlock = new class_2544(class_4970.class_2251.method_9630(block).method_63500(TRBlockSettings.key(name + "_storage_block_wall")));
			InitUtils.setup(wallBlock, name + "_storage_block_wall");
		}

		StorageBlocks(boolean isHot, float hardness, float resistance) {
			this(isHot, hardness, resistance, null);
		}

		StorageBlocks(float hardness, float resistance) {
			this(false, hardness, resistance, null);
		}

		StorageBlocks() {
			this(false, 3f, 6f);
		}

		@Override
		public String getName() {
			return name;
		}

		@Override
		public class_1792 method_8389() {
			return block.method_8389();
		}

		@Override
		public class_6862<class_1792> asTag() {
			return tag;
		}

		@Override
		public class_2248 getBlock() {
			return block;
		}

		@Override
		public class_2510 getStairsBlock() {
			return stairsBlock;
		}

		@Override
		public class_2482 getSlabBlock() {
			return slabBlock;
		}

		@Override
		public class_2544 getWallBlock() {
			return wallBlock;
		}

		public static Stream<class_2248> blockStream() {
			return Arrays.stream(values())
					.map(StorageBlocks::allBlocks)
					.flatMap(Collection::stream);
		}

		private List<class_2248> allBlocks() {
			return List.of(block, stairsBlock, slabBlock, wallBlock);
		}
	}

	public enum MachineBlocks implements MachineBlockInfo {
		BASIC(1020 / 25),
		ADVANCED(1700 / 25),
		INDUSTRIAL(2380 / 25);

		public final String name;
		public final class_2248 frame;
		public final class_2248 casing;

		MachineBlocks(int casingHeatCapacity) {
			name = this.toString().toLowerCase(Locale.ROOT);
			frame = new BlockMachineFrame(name + "_machine_frame");
			InitUtils.setup(frame, name + "_machine_frame");
			casing = new BlockMachineCasing(casingHeatCapacity, name + "_machine_casing");
			InitUtils.setup(casing, name + "_machine_casing");
		}

		@Override
		public String getName() {
			return name;
		}

		@Override
		public class_2248 getFrame() {
			return frame;
		}

		@Override
		public class_2248 getCasing() {
			return casing;
		}

		public static class_1935[] getCasings() {
			return Arrays.stream(MachineBlocks.values())
					.map((Function<MachineBlocks, class_1935>) machineBlocks -> machineBlocks.casing)
					.toArray(class_1935[]::new);
		}
	}


	public enum Machine implements BlockInfo {
		ALLOY_SMELTER(new GenericMachineBlock(GuiType.ALLOY_SMELTER, AlloySmelterBlockEntity::new, "alloy_smelter")),
		ASSEMBLY_MACHINE(new GenericMachineBlock(GuiType.ASSEMBLING_MACHINE, AssemblingMachineBlockEntity::new, "assembly_machine")),
		AUTO_CRAFTING_TABLE(new GenericMachineBlock(GuiType.AUTO_CRAFTING_TABLE, AutoCraftingTableBlockEntity::new, "auto_crafting_table")),
		CHEMICAL_REACTOR(new GenericMachineBlock(GuiType.CHEMICAL_REACTOR, ChemicalReactorBlockEntity::new, "chemical_reactor")),
		COMPRESSOR(new GenericMachineBlock(GuiType.COMPRESSOR, CompressorBlockEntity::new, "compressor")),
		DISTILLATION_TOWER(new GenericMachineBlock(GuiType.DISTILLATION_TOWER, DistillationTowerBlockEntity::new, "distillation_tower")),
		EXTRACTOR(new GenericMachineBlock(GuiType.EXTRACTOR, ExtractorBlockEntity::new, "extractor")),
		RESIN_BASIN(new ResinBasinBlock(ResinBasinBlockEntity::new, "resin_basin")),
		FLUID_REPLICATOR(new GenericMachineBlock(GuiType.FLUID_REPLICATOR, FluidReplicatorBlockEntity::new, "fluid_replicator")),
		GRINDER(new GenericMachineBlock(GuiType.GRINDER, GrinderBlockEntity::new, "grinder")),
		ELECTRIC_FURNACE(new GenericMachineBlock(GuiType.ELECTRIC_FURNACE, ElectricFurnaceBlockEntity::new, "electric_furnace")),
		IMPLOSION_COMPRESSOR(new GenericMachineBlock(GuiType.IMPLOSION_COMPRESSOR, ImplosionCompressorBlockEntity::new, "implosion_compressor")),
		INDUSTRIAL_BLAST_FURNACE(new GenericMachineBlock(GuiType.BLAST_FURNACE, IndustrialBlastFurnaceBlockEntity::new, "industrial_blast_furnace")),
		INDUSTRIAL_CENTRIFUGE(new GenericMachineBlock(GuiType.CENTRIFUGE, IndustrialCentrifugeBlockEntity::new, "industrial_centrifuge")),
		INDUSTRIAL_ELECTROLYZER(new GenericMachineBlock(GuiType.INDUSTRIAL_ELECTROLYZER, IndustrialElectrolyzerBlockEntity::new, "industrial_electrolyzer")),
		INDUSTRIAL_GRINDER(new GenericMachineBlock(GuiType.INDUSTRIAL_GRINDER, IndustrialGrinderBlockEntity::new, "industrial_grinder")),
		INDUSTRIAL_SAWMILL(new GenericMachineBlock(GuiType.SAWMILL, IndustrialSawmillBlockEntity::new, "industrial_sawmill")),
		IRON_ALLOY_FURNACE(new IronAlloyFurnaceBlock("iron_alloy_furnace")),
		IRON_FURNACE(new IronFurnaceBlock("iron_furnace")),
		MATTER_FABRICATOR(new GenericMachineBlock(GuiType.MATTER_FABRICATOR, MatterFabricatorBlockEntity::new, "matter_fabricator")),
		RECYCLER(new GenericMachineBlock(GuiType.RECYCLER, RecyclerBlockEntity::new, "recycler")),
		ROLLING_MACHINE(new GenericMachineBlock(GuiType.ROLLING_MACHINE, RollingMachineBlockEntity::new, "rolling_machine")),
		SCRAPBOXINATOR(new GenericMachineBlock(GuiType.SCRAPBOXINATOR, ScrapboxinatorBlockEntity::new, "scrapboxinator")),
		VACUUM_FREEZER(new GenericMachineBlock(GuiType.VACUUM_FREEZER, VacuumFreezerBlockEntity::new, "vacuum_freezer")),
		SOLID_CANNING_MACHINE(new GenericMachineBlock(GuiType.SOLID_CANNING_MACHINE, SolidCanningMachineBlockEntity::new, "solid_canning_machine")),
		WIRE_MILL(new GenericMachineBlock(GuiType.WIRE_MILL, WireMillBlockEntity::new, "wire_mill")),
		GREENHOUSE_CONTROLLER(new GenericMachineBlock(GuiType.GREENHOUSE_CONTROLLER, GreenhouseControllerBlockEntity::new, "greenhouse_controller")),
		BLOCK_BREAKER(new GenericMachineBlock(GuiType.BLOCK_BREAKER, BlockBreakerBlockEntity::new, "block_breaker")),
		BLOCK_PLACER(new GenericMachineBlock(GuiType.BLOCK_PLACER, BlockPlacerBlockEntity::new, "block_placer")),
		LAUNCHPAD(new GenericMachineBlock(GuiType.LAUNCHPAD, LaunchpadBlockEntity::new, "launchpad")),
		ELEVATOR(new GenericMachineBlock(GuiType.ELEVATOR, ElevatorBlockEntity::new, "elevator")),
		FISHING_STATION(new GenericMachineBlock(GuiType.FISHING_STATION, FishingStationBlockEntity::new, "fishing_station")),

		DIESEL_GENERATOR(new GenericGeneratorBlock(GuiType.DIESEL_GENERATOR, DieselGeneratorBlockEntity::new, "diesel_generator")),
		DRAGON_EGG_SYPHON(new GenericGeneratorBlock(null, DragonEggSyphonBlockEntity::new, "dragon_egg_syphon")),
		FUSION_COIL(new BlockFusionCoil("fusion_coil")),
		FUSION_CONTROL_COMPUTER(new BlockFusionControlComputer("fusion_control_computer")),
		GAS_TURBINE(new GenericGeneratorBlock(GuiType.GAS_TURBINE, GasTurbineBlockEntity::new, "gas_turbine")),
		LIGHTNING_ROD(new GenericGeneratorBlock(null, LightningRodBlockEntity::new, "lightning_rod")),
		PLASMA_GENERATOR(new GenericGeneratorBlock(GuiType.PLASMA_GENERATOR, PlasmaGeneratorBlockEntity::new, "plasma_generator")),
		SEMI_FLUID_GENERATOR(new GenericGeneratorBlock(GuiType.SEMIFLUID_GENERATOR, SemiFluidGeneratorBlockEntity::new, "semi_fluid_generator")),
		SOLID_FUEL_GENERATOR(new GenericGeneratorBlock(GuiType.GENERATOR, SolidFuelGeneratorBlockEntity::new, "solid_fuel_generator")),
		THERMAL_GENERATOR(new GenericGeneratorBlock(GuiType.THERMAL_GENERATOR, ThermalGeneratorBlockEntity::new, "thermal_generator")),
		WATER_MILL(new GenericGeneratorBlock(null, WaterMillBlockEntity::new, "water_mill")),
		WIND_MILL(new GenericGeneratorBlock(null, WindMillBlockEntity::new, "wind_mill")),

		DRAIN(new GenericMachineBlock(null, DrainBlockEntity::new, "drain")),
		PUMP(new GenericMachineBlock(GuiType.PUMP, PumpBlockEntity::new, "pump")),
		ADJUSTABLE_SU(new AdjustableSUBlock("adjustable_su")),
		CHARGE_O_MAT(new GenericMachineBlock(GuiType.CHARGEBENCH, ChargeOMatBlockEntity::new, "charge_o_mat")),
		INTERDIMENSIONAL_SU(new InterdimensionalSUBlock("interdimensional_su")),
		LAPOTRONIC_SU(new LapotronicSUBlock("lapotronic_su")),
		LSU_STORAGE(new LSUStorageBlock("lsu_storage")),
		LOW_VOLTAGE_SU(new LowVoltageSUBlock("low_voltage_su")),
		MEDIUM_VOLTAGE_SU(new MediumVoltageSUBlock("medium_voltage_su")),
		HIGH_VOLTAGE_SU(new HighVoltageSUBlock("high_voltage_su")),
		LV_TRANSFORMER(new BlockLVTransformer("lv_transformer")),
		MV_TRANSFORMER(new BlockMVTransformer("mv_transformer")),
		HV_TRANSFORMER(new BlockHVTransformer("hv_transformer")),
		EV_TRANSFORMER(new BlockEVTransformer("ev_transformer")),

		ALARM(new BlockAlarm("alarm")),
		CHUNK_LOADER(new GenericMachineBlock(GuiType.CHUNK_LOADER, ChunkLoaderBlockEntity::new, "chunk_loader")),
		LAMP_INCANDESCENT(new LampBlock(4, 10, 8, "lamp_incandescent")),
		LAMP_LED(new LampBlock(1, 1, 12, "lamp_led")),
		PLAYER_DETECTOR(new PlayerDetectorBlock("player_detector")),;

		public final String name;
		public final class_2248 block;

		<B extends class_2248> Machine(B block) {
			this.name = this.toString().toLowerCase(Locale.ROOT);
			this.block = block;
			InitUtils.setup(block, name);
		}

		@Override
		public String getName() {
			return name;
		}

		public class_1799 getStack() {
			return new class_1799(block);
		}

		@Override
		public class_1792 method_8389() {
			return block.method_8389();
		}

		@Override
		public class_2248 getBlock() {
			return block;
		}
	}

	public enum Dusts implements ItemInfo, TagConvertible<class_1792> {
		ALMANDINE, ALUMINUM, AMETHYST, ANDESITE, ANDRADITE, ASHES, BASALT, BAUXITE, BRASS, BRONZE, CALCITE, CHARCOAL, CHROME(CHROME_TAG_NAME_BASE),
		CINNABAR, CLAY, COAL, DARK_ASHES, DIAMOND, DIORITE, ELECTRUM, EMERALD, ENDER_EYE, ENDER_PEARL, ENDSTONE,
		FLINT, GALENA, GRANITE, GROSSULAR, INVAR, LAZURITE, MAGNESIUM, MANGANESE, MARBLE, NETHERRACK,
		NICKEL, OBSIDIAN, OLIVINE, PERIDOT, PHOSPHOROUS, PLATINUM, PYRITE, PYROPE, QUARTZ, RED_GARNET, RUBY, SALTPETER,
		SAPPHIRE, SAW, SODALITE, SPESSARTINE, SPHALERITE, STEEL, SULFUR, TITANIUM, UVAROVITE, YELLOW_GARNET, ZINC;

		public final String name;
		private final class_1792 item;
		private final class_6862<class_1792> tag;

		Dusts(String tagNameBase) {
			name = this.toString().toLowerCase(Locale.ROOT);
			item = new class_1792(TRItemSettings.item(name + "_dust"));
			InitUtils.setup(item, name + "_dust");
			tag = class_6862.method_40092(class_7924.field_41197, class_2960.method_60655("c", "dusts/" + Objects.requireNonNullElse(tagNameBase, name)));
		}

		Dusts() {
			this(null);
		}

		@Override
		public String getName() {
			return name;
		}

		public class_1799 getStack() {
			return new class_1799(item);
		}

		public class_1799 getStack(int amount) {
			return new class_1799(item, amount);
		}

		@Override
		public class_1792 method_8389() {
			return item;
		}

		@Override
		public class_6862<class_1792> asTag() {
			return tag;
		}
	}

	public enum RawMetals implements ItemInfo, TagConvertible<class_1792> {
		IRIDIUM, LEAD, SILVER, TIN, TUNGSTEN;

		public final String name;
		private final class_1792 item;
		private final Ores ore;
		private final StorageBlocks storageBlock;
		private final class_6862<class_1792> tag;

		RawMetals() {
			name = this.toString().toLowerCase(Locale.ROOT);
			item = new class_1792(TRItemSettings.item("raw_" + name));
			Ores oreVariant = null;
			try {
				oreVariant = Ores.valueOf(this.toString());
			}
			catch (IllegalArgumentException ex) {
				if (InitUtils.isDatagenRunning())
					TechReborn.LOGGER.warn(DATAGEN, "Raw metal {} has no ore block equivalent!", name);
			}
			ore = oreVariant;
			StorageBlocks blockVariant = null;
			try {
				blockVariant = StorageBlocks.valueOf("RAW_" + this.toString());
			}
			catch (IllegalArgumentException ex) {
				if (InitUtils.isDatagenRunning())
					TechReborn.LOGGER.warn(DATAGEN, "Raw metal {} has no storage block equivalent!", name);
			}
			storageBlock = blockVariant;
			InitUtils.setup(item, "raw_" + name);
			tag = class_6862.method_40092(class_7924.field_41197, class_2960.method_60655("c", "raw_materials/" + name));
		}

		@Override
		public String getName() {
			return name;
		}

		@Override
		public class_1792 method_8389() {
			return item;
		}

		@Override
		public class_6862<class_1792> asTag() {
			return tag;
		}

		public StorageBlocks getStorageBlock() {
			return storageBlock;
		}

		public Ores getOre() {
			return ore;
		}

		/**
		 * Returns a map that maps the raw metals to their storage block equivalent.
		 * @return A non {@code null} map mapping the raw metals to their storage block equivalent.
		 * If a storage block equivalent doesn't exist, the raw metal will not be in the keys of this map.
		 */
		public static @NotNull Map<RawMetals, StorageBlocks> getRM2SBMap() {
			return Arrays.stream(values())
					.map(rawMetal -> new class_3545<>(rawMetal, rawMetal.getStorageBlock()))
					.filter(entry -> entry.method_15441() != null) // ensure storage block equivalent exists
					.collect(Collectors.toMap(class_3545::method_15442, class_3545::method_15441));
		}

		/**
		 * Returns a map that maps the raw metals to their ore block equivalent.
		 * @return A non {@code null} map mapping the raw metals to their ore block equivalent.
		 * If an ore block equivalent doesn't exist, the raw metal will not be in the keys of this map.
		 */
		public static @NotNull Map<RawMetals, Ores> getRM2OBMap() {
			return Arrays.stream(values())
					.map(rawMetal -> new class_3545<>(rawMetal, rawMetal.getOre()))
					.filter(entry -> entry.method_15441() != null) // ensure ore block equivalent exists
					.collect(Collectors.toMap(class_3545::method_15442, class_3545::method_15441));
		}
	}

	public enum SmallDusts implements ItemInfo, TagConvertible<class_1792> {
		ALMANDINE, ANDESITE, ANDRADITE, ASHES, BASALT, BAUXITE, CALCITE, CHARCOAL, CHROME(CHROME_TAG_NAME_BASE),
		CINNABAR, CLAY, COAL, DARK_ASHES, DIAMOND, DIORITE, ELECTRUM, EMERALD, ENDER_EYE, ENDER_PEARL, ENDSTONE,
		FLINT, GALENA, GLOWSTONE(class_1802.field_8601), GRANITE, GROSSULAR, INVAR, LAZURITE, MAGNESIUM, MANGANESE, MARBLE,
		NETHERRACK, NICKEL, OBSIDIAN, OLIVINE, PERIDOT, PHOSPHOROUS, PLATINUM, PYRITE, PYROPE, QUARTZ, REDSTONE(class_1802.field_8725),
		RED_GARNET, RUBY, SALTPETER, SAPPHIRE, SAW, SODALITE, SPESSARTINE, SPHALERITE, STEEL, SULFUR, TITANIUM,
		TUNGSTEN(RawMetals.TUNGSTEN), UVAROVITE, YELLOW_GARNET, ZINC;

		public final String name;
		private final class_1792 item;
		private final class_1935 dust;
		private final class_6862<class_1792> tag;

		SmallDusts(String tagNameBase, class_1935 dustVariant) {
			name = this.toString().toLowerCase(Locale.ROOT);
			item = new class_1792(TRItemSettings.item(name + "_small_dust"));
			if (dustVariant == null)
				try {
					dustVariant = Dusts.valueOf(this.toString());
				}
				catch (IllegalArgumentException ex) {
					if (InitUtils.isDatagenRunning())
						TechReborn.LOGGER.warn(DATAGEN, "Small dust {} has no dust equivalent!", name);
				}
			dust = dustVariant;
			InitUtils.setup(item, name + "_small_dust");
			tag = class_6862.method_40092(class_7924.field_41197, class_2960.method_60655("c", "small_dusts/" + Objects.requireNonNullElse(tagNameBase, name)));
		}

		SmallDusts(String tagNameBase) {
			this(tagNameBase, null);
		}

		SmallDusts(class_1935 dustVariant) {
			this(null, dustVariant);
		}

		SmallDusts() {
			this(null, null);
		}

		@Override
		public String getName() {
			return name;
		}

		public class_1799 getStack() {
			return new class_1799(item);
		}

		public class_1799 getStack(int amount) {
			return new class_1799(item, amount);
		}

		@Override
		public class_1792 method_8389() {
			return item;
		}

		@Override
		public class_6862<class_1792> asTag() {
			return tag;
		}

		public class_1935 getDust() {
			return dust;
		}

		/**
		 * Returns a map that maps the small dusts to their dust equivalent,
		 * as it was specified in the enum. Note that the dust equivalent
		 * doesn't have to be a TR dust (see redstone and glowstone dust)
		 * and also not a dust at all (see tungsten).
		 * @return A non {@code null} map mapping the small dusts to their dust equivalent.
		 * If a dust equivalent doesn't exist, the small dust will not be in the keys of this map.
		 */
		public static @NotNull Map<SmallDusts, class_1935> getSD2DMap() {
			return Arrays.stream(values())
					.map(smallDust -> new class_3545<>(smallDust, smallDust.getDust()))
					.filter(entry -> entry.method_15441() != null) // ensure dust equivalent exists
					.collect(Collectors.toMap(class_3545::method_15442, class_3545::method_15441));
		}
	}

	public enum Gems implements ItemInfo, TagConvertible<class_1792> {
		PERIDOT, RED_GARNET, RUBY, SAPPHIRE, YELLOW_GARNET;

		public final String name;
		private final class_1792 item;
		private final Dusts dust;
		private final Ores ore;
		private final StorageBlocks storageBlock;
		private final class_6862<class_1792> tag;

		Gems() {
			name = this.toString().toLowerCase(Locale.ROOT);
			item = new class_1792(TRItemSettings.item(name + "_gem"));
			Dusts dustVariant = null;
			try {
				dustVariant = Dusts.valueOf(this.toString());
			}
			catch (IllegalArgumentException ex) {
				if (InitUtils.isDatagenRunning())
					TechReborn.LOGGER.warn(DATAGEN, "Gem {} has no dust item equivalent!", name);
			}
			dust = dustVariant;
			Ores oreVariant = null;
			try {
				oreVariant = Ores.valueOf(this.toString());
			}
			catch (IllegalArgumentException ex) {
				if (InitUtils.isDatagenRunning())
					TechReborn.LOGGER.info(DATAGEN, "Gem {} has no ore block equivalent.", name);
			}
			ore = oreVariant;
			StorageBlocks blockVariant = null;
			try {
				blockVariant = StorageBlocks.valueOf(this.toString());
			}
			catch (IllegalArgumentException ex) {
				TechReborn.LOGGER.warn(DATAGEN, "Gem {} has no storage block equivalent!", name);
			}
			storageBlock = blockVariant;
			InitUtils.setup(item, name + "_gem");
			tag = class_6862.method_40092(class_7924.field_41197, class_2960.method_60655("c", "gems/" + name));
		}

		@Override
		public String getName() {
			return name;
		}

		public class_1799 getStack() {
			return new class_1799(item);
		}

		public class_1799 getStack(int amount) {
			return new class_1799(item, amount);
		}

		@Override
		public class_1792 method_8389() {
			return item;
		}

		@Override
		public class_6862<class_1792> asTag() {
			return tag;
		}

		public Dusts getDust() {
			return dust;
		}

		public Ores getOre() {
			return ore;
		}

		public StorageBlocks getStorageBlock() {
			return storageBlock;
		}

		/**
		 * Returns a map that maps the gems to their dust item equivalent.
		 * @return A non {@code null} map mapping the gems to their dust item equivalent.
		 * If a dust item equivalent doesn't exist, the gem will not be in the keys of this map.
		 */
		public static @NotNull Map<Gems, Dusts> getG2DMap() {
			return Arrays.stream(values())
					.map(gem -> new class_3545<>(gem, gem.getDust()))
					.filter(entry -> entry.method_15441() != null) // ensure dust item equivalent exists
					.collect(Collectors.toMap(class_3545::method_15442, class_3545::method_15441));
		}

		/**
		 * Returns a map that maps the gems to their storage block equivalent.
		 * @return A non {@code null} map mapping the gems to their storage block equivalent.
		 * If a storage block equivalent doesn't exist, the gem will not be in the keys of this map.
		 */
		public static @NotNull Map<Gems, StorageBlocks> getG2SBMap() {
			return Arrays.stream(values())
					.map(gem -> new class_3545<>(gem, gem.getStorageBlock()))
					.filter(entry -> entry.method_15441() != null) // ensure storage block equivalent exists
					.collect(Collectors.toMap(class_3545::method_15442, class_3545::method_15441));
		}
	}


	public enum Ingots implements ItemInfo, TagConvertible<class_1792> {
		ADVANCED_ALLOY, ALUMINUM, BRASS, BRONZE, CHROME(CHROME_TAG_NAME_BASE), ELECTRUM, HOT_TUNGSTENSTEEL, INVAR, IRIDIUM_ALLOY, IRIDIUM,
		LEAD, MIXED_METAL, NICKEL, PLATINUM, REFINED_IRON, SILVER, STEEL, TIN, TITANIUM, TUNGSTEN, TUNGSTENSTEEL, ZINC;

		public final String name;
		private final class_1792 item;
		private final Dusts dust;
		private final StorageBlocks storageBlock;
		private final class_6862<class_1792> tag;

		Ingots(String tagNameBase) {
			name = this.toString().toLowerCase(Locale.ROOT);
			item = new class_1792(TRItemSettings.item(name + "_ingot"));
			Dusts dustVariant = null;
			try {
				dustVariant = Dusts.valueOf(this.toString());
			}
			catch (IllegalArgumentException ex) {
				try {
					RawMetals.valueOf(this.toString());
					if (InitUtils.isDatagenRunning())
						TechReborn.LOGGER.info(DATAGEN, "Ingot {} has no dust item equivalent, but a raw metal.", name);
				}
				catch (IllegalArgumentException ex2) {
					if (InitUtils.isDatagenRunning())
						TechReborn.LOGGER.warn(DATAGEN, "Ingot {} has no dust item equivalent AND no raw metal!", name);
				}
			}
			dust = dustVariant;
			StorageBlocks blockVariant = null;
			try {
				blockVariant = StorageBlocks.valueOf(this.toString());
			}
			catch (IllegalArgumentException ex) {
				if (InitUtils.isDatagenRunning())
					TechReborn.LOGGER.warn(DATAGEN, "Ingot {} has no storage block equivalent!", name);
			}
			storageBlock = blockVariant;
			InitUtils.setup(item, name + "_ingot");
			tag = class_6862.method_40092(class_7924.field_41197, class_2960.method_60655("c", "ingots/" + Objects.requireNonNullElse(tagNameBase, name)));
		}

		Ingots() {
			this(null);
		}

		@Override
		public String getName() {
			return name;
		}

		public class_1799 getStack() {
			return new class_1799(item);
		}

		public class_1799 getStack(int amount) {
			return new class_1799(item, amount);
		}

		@Override
		public class_1792 method_8389() {
			return item;
		}

		@Override
		public class_6862<class_1792> asTag() {
			return tag;
		}

		public Dusts getDust() {
			return dust;
		}

		public StorageBlocks getStorageBlock() {
			return storageBlock;
		}

		/**
		 * Returns a map that maps the ingots to their dust item equivalent.
		 * @return A non {@code null} map mapping the ingots to their dust item equivalent.
		 * If a dust item equivalent doesn't exist, the ingot will not be in the keys of this map.
		 */
		public static @NotNull Map<Ingots, Dusts> getI2DMap() {
			return Arrays.stream(values())
					.map(gem -> new class_3545<>(gem, gem.getDust()))
					.filter(entry -> entry.method_15441() != null) // ensure dust item equivalent exists
					.collect(Collectors.toMap(class_3545::method_15442, class_3545::method_15441));
		}

		/**
		 * Returns a map that maps the ingots to their storage block equivalent.
		 * @return A non {@code null} map mapping the ingots to their storage block equivalent.
		 * If a storage block equivalent doesn't exist, the raw metal will not be in the keys of this map.
		 */
		public static @NotNull Map<Ingots, class_1935> getI2SBMap() {
			return Arrays.stream(values())
					.map(ingot -> new class_3545<>(ingot, ingot.getStorageBlock()))
					.filter(entry -> entry.method_15441() != null) // ensure storage block equivalent exists
					.collect(Collectors.toMap(class_3545::method_15442, class_3545::method_15441));
		}
	}

	public enum Nuggets implements ItemInfo, TagConvertible<class_1792> {
		ALUMINUM, BRASS, BRONZE, CHROME(CHROME_TAG_NAME_BASE), COPPER(class_1802.field_27022, false), DIAMOND(class_1802.field_8477, true),
		ELECTRUM, EMERALD(class_1802.field_8687, true), HOT_TUNGSTENSTEEL, INVAR, IRIDIUM, LEAD,
		NETHERITE, /* We do NOT link to the netherite ingot here, because we want custom conversion recipes! */
		NICKEL, PLATINUM, REFINED_IRON, SILVER, STEEL, TIN, TITANIUM, TUNGSTEN, TUNGSTENSTEEL, ZINC;

		public final String name;
		private final class_1792 item;
		private final class_1935 ingot;
		private final boolean ofGem;
		private final class_6862<class_1792> tag;

		Nuggets(String tagNameBase, class_1935 ingotVariant, boolean ofGem) {
			name = this.toString().toLowerCase(Locale.ROOT);
			item = new class_1792(TRItemSettings.item(name + "_nugget"));
			if (ingotVariant == null)
				try {
					ingotVariant = Ingots.valueOf(this.toString());
				}
				catch (IllegalArgumentException ex) {
					if (InitUtils.isDatagenRunning())
						TechReborn.LOGGER.warn(DATAGEN, "Nugget {} has no ingot equivalent!", name);
				}
			ingot = ingotVariant;
			this.ofGem = ofGem;
			InitUtils.setup(item, name + "_nugget");
			tag = class_6862.method_40092(class_7924.field_41197, class_2960.method_60655("c", "nuggets/" + Objects.requireNonNullElse(tagNameBase, name)));
		}

		Nuggets(class_1935 ingotVariant, boolean ofGem) {
			this(null, ingotVariant, ofGem);
		}

		Nuggets(String tagNameBase) {
			this(tagNameBase, null, false);
		}

		Nuggets() {
			this(null, false);
		}

		@Override
		public String getName() {
			return name;
		}

		public class_1799 getStack() {
			return new class_1799(item);
		}

		public class_1799 getStack(int amount) {
			return new class_1799(item, amount);
		}

		@Override
		public class_1792 method_8389() {
			return item;
		}

		@Override
		public class_6862<class_1792> asTag() {
			return tag;
		}

		public class_1935 getIngot() {
			return ingot;
		}

		public boolean isOfGem() {
			return false;
		}

		/**
		 * Returns a map that maps the nuggets to their ingot equivalent,
		 * as it was specified in the enum. Note that the ingot equivalent
		 * doesn't have to be an ingot at all (see emerald and diamond).
		 * @return A non {@code null} map mapping the nuggets to their ingot equivalent.
		 * If an ingot equivalent doesn't exist, the raw metal will not be in the keys of this map.
		 */
		public static @NotNull Map<Nuggets, class_1935> getN2IMap() {
			return Arrays.stream(values())
					.map(nugget -> new class_3545<>(nugget, nugget.getIngot()))
					.filter(entry -> entry.method_15441() != null) // ensure ingot equivalent exists
					.collect(Collectors.toMap(class_3545::method_15442, class_3545::method_15441));
		}
	}

	public enum Parts implements ItemInfo {
		CARBON_FIBER,
		CARBON_MESH,

		ELECTRONIC_CIRCUIT,
		ADVANCED_CIRCUIT,
		INDUSTRIAL_CIRCUIT,

		MACHINE_PARTS,
		BASIC_DISPLAY,
		DIGITAL_DISPLAY,

		DATA_STORAGE_CORE,
		DATA_STORAGE_CHIP,
		ENERGY_FLOW_CHIP,
		SUPERCONDUCTOR,

		DIAMOND_SAW_BLADE,
		DIAMOND_GRINDING_HEAD,
		TUNGSTEN_GRINDING_HEAD,

		CUPRONICKEL_HEATING_COIL,
		KANTHAL_HEATING_COIL,
		NICHROME_HEATING_COIL,

		NEUTRON_REFLECTOR,
		THICK_NEUTRON_REFLECTOR,
		IRIDIUM_NEUTRON_REFLECTOR,

		//java vars can't start with numbers, so these get suffixes
		WATER_COOLANT_CELL_10K,
		WATER_COOLANT_CELL_30K,
		WATER_COOLANT_CELL_60K,

		HELIUM_COOLANT_CELL_60K,
		HELIUM_COOLANT_CELL_180K,
		HELIUM_COOLANT_CELL_360K,

		NAK_COOLANT_CELL_60K,
		NAK_COOLANT_CELL_180K,
		NAK_COOLANT_CELL_360K,

		RUBBER,
		SAP,
		SCRAP,
		UU_MATTER,
		PLANTBALL,
		COMPRESSED_PLANTBALL,
		SPONGE_PIECE,
		TEMPLATE_TEMPLATE,

		SYNTHETIC_REDSTONE_CRYSTAL;

		public final String name;
		public final class_1792 item;

		Parts() {
			name = this.toString().toLowerCase(Locale.ROOT);
			item = new class_1792(TRItemSettings.item(name));
			InitUtils.setup(item, name);
		}

		@Override
		public String getName() {
			return name;
		}

		public class_1799 getStack() {
			return new class_1799(item);
		}

		public class_1799 getStack(int amount) {
			return new class_1799(item, amount);
		}

		@Override
		public class_1792 method_8389() {
			return item;
		}
	}

	public enum Plates implements ItemInfo, TagConvertible<class_1792> {
		ADVANCED_ALLOY,
		ALUMINUM,
		BRASS,
		BRONZE,
		CARBON(Parts.CARBON_MESH),
		CHROME(CHROME_TAG_NAME_BASE),
		COAL(Dusts.COAL, class_1802.field_8797),
		COPPER(class_1802.field_27022, class_1802.field_27071),
		DIAMOND(Dusts.DIAMOND, class_1802.field_8603),
		ELECTRUM,
		EMERALD(Dusts.EMERALD, class_1802.field_8733),
		GOLD(class_1802.field_8695, class_1802.field_8494),
		INVAR,
		IRIDIUM_ALLOY(true),
		IRIDIUM,
		IRON(class_1802.field_8620, class_1802.field_8773),
		LAPIS(class_1802.field_8055),
		LAZURITE(Dusts.LAZURITE),
		LEAD,
		MAGNALIUM,
		NICKEL,
		OBSIDIAN(Dusts.OBSIDIAN, class_1802.field_8281),
		PERIDOT,
		PLATINUM,
		QUARTZ(Dusts.QUARTZ),
		RED_GARNET,
		REDSTONE(class_1802.field_8793),
		REFINED_IRON,
		RUBY,
		SAPPHIRE,
		SILICON,
		SILVER,
		STEEL,
		TIN,
		TITANIUM,
		TUNGSTEN,
		TUNGSTENSTEEL,
		WOOD,
		YELLOW_GARNET,
		ZINC;

		public final String name;
		private final class_1792 item;
		private final class_1935 source;
		private final class_1935 sourceBlock;
		private final boolean industrial;
		private final class_6862<class_1792> tag;

		Plates(class_1935 source, class_1935 sourceBlock, boolean industrial, String tagNameBase) {
			name = this.toString().toLowerCase(Locale.ROOT);
			item = new class_1792(TRItemSettings.item(name + "_plate"));
			class_1935 sourceVariant = null;
			if (source != null) {
				sourceVariant = source;
			}
			else {
				try {
					sourceVariant = Ingots.valueOf(this.toString());
				}
				catch (IllegalArgumentException ex) {
					try {
						sourceVariant = Gems.valueOf(this.toString());
					}
					catch (IllegalArgumentException ex2) {
						if (InitUtils.isDatagenRunning())
							TechReborn.LOGGER.warn(DATAGEN, "Plate {} has no identifiable source!", name);
					}
				}
			}
			if (sourceBlock != null) {
				this.sourceBlock = sourceBlock;
			}
			else {
				if (sourceVariant instanceof Gems gem)
					this.sourceBlock = gem.getStorageBlock();
				else if (sourceVariant instanceof Ingots ingot)
					this.sourceBlock = ingot.getStorageBlock();
				else {
					if (InitUtils.isDatagenRunning())
						TechReborn.LOGGER.info(DATAGEN, "Plate {} has no identifiable source block.", name);
					this.sourceBlock = null;
				}
			}
			if (sourceVariant instanceof Gems gem)
				this.source = gem.getDust();
			else
				this.source = sourceVariant;
			this.industrial = industrial;
			InitUtils.setup(item, name + "_plate");

			if (tagNameBase == null) {
				tagNameBase = name;
			}

			tag = class_6862.method_40092(class_7924.field_41197, class_2960.method_60655("c", "plates/" + tagNameBase));
		}

		Plates(String tagNameBase) {
			this(null, null, false, tagNameBase);
		}

		Plates(class_1935 source, class_1935 sourceBlock) {
			this(source, sourceBlock, false, null);
		}

		Plates(class_1935 source) {
			this(source, null, false, null);
		}

		Plates(boolean industrial) {
			this(null, null, industrial, null);
		}

		Plates() {
			this(null, null, false, null);
		}

		@Override
		public String getName() {
			return name;
		}

		public class_1799 getStack() {
			return new class_1799(item);
		}

		public class_1799 getStack(int amount) {
			return new class_1799(item, amount);
		}

		@Override
		public class_1792 method_8389() {
			return item;
		}

		public class_1935 getSource() {
			return source;
		}

		public class_1935 getSourceBlock() {
			return sourceBlock;
		}

		public boolean isIndustrial() {
			return industrial;
		}

		@Override
		public class_6862<class_1792> asTag() {
			return tag;
		}
	}

	public enum Upgrades implements ItemInfo {
		OVERCLOCKER((blockEntity, handler, stack) -> {
			PowerAcceptorBlockEntity powerAcceptor = null;
			if (blockEntity instanceof PowerAcceptorBlockEntity) {
				powerAcceptor = (PowerAcceptorBlockEntity) blockEntity;
			}
			if (handler != null) {
				handler.addSpeedMultiplier(TechRebornConfig.overclockerSpeed);
				handler.addPowerMultiplier(TechRebornConfig.overclockerPower);
			}
			if (powerAcceptor != null) {
				powerAcceptor.extraPowerInput += powerAcceptor.getMaxInput(null);
				powerAcceptor.extraPowerStorage += powerAcceptor.getBaseMaxPower();
			}
		}),
		TRANSFORMER((blockEntity, handler, stack) -> {
			PowerAcceptorBlockEntity powerAcceptor = null;
			if (blockEntity instanceof PowerAcceptorBlockEntity) {
				powerAcceptor = (PowerAcceptorBlockEntity) blockEntity;
			}
			if (powerAcceptor != null) {
				powerAcceptor.extraTier += 1;
			}
		}),
		ENERGY_STORAGE((blockEntity, handler, stack) -> {
			PowerAcceptorBlockEntity powerAcceptor = null;
			if (blockEntity instanceof PowerAcceptorBlockEntity) {
				powerAcceptor = (PowerAcceptorBlockEntity) blockEntity;
			}
			if (powerAcceptor != null) {
				powerAcceptor.extraPowerStorage += TechRebornConfig.energyStoragePower;
			}
		}),
		SUPERCONDUCTOR((blockEntity, handler, stack) -> {
			AdjustableSUBlockEntity aesu = null;
			if (blockEntity instanceof AdjustableSUBlockEntity) {
				aesu = (AdjustableSUBlockEntity) blockEntity;
			}
			if (aesu != null) {
				aesu.superconductors += TechRebornConfig.superConductorCount;
			}
		}),
		MUFFLER((blockEntity, handler, stack) -> {
			blockEntity.muffle();
		});

		public final String name;
		public final class_1792 item;

		Upgrades(IUpgrade upgrade) {
			name = this.toString().toLowerCase(Locale.ROOT);
			item = new UpgradeItem(name + "_upgrade", upgrade);
			InitUtils.setup(item, name + "_upgrade");
		}

		@Override
		public String getName() {
			return name;
		}

		@Override
		public class_1792 method_8389() {
			return item;
		}

		public static Upgrades fromItem(UpgradeItem item) {
			for (Upgrades upgrade : values()) {
				if (upgrade.item == item) {
					return upgrade;
				}
			}

			throw new IllegalArgumentException("Item is not an upgrade item");
		}
	}

	public static final class_1299<EntityNukePrimed> ENTITY_NUKE = class_1299.class_1300.method_5903((class_1299.class_4049<EntityNukePrimed>) EntityNukePrimed::new, class_1311.field_17715)
		.method_17687(1f, 1f)
		.method_27299(10)
		.method_5905(class_5321.method_29179(class_7924.field_41266, class_2960.method_60655(TechReborn.MOD_ID, "nuke")));

	public static void register() {
		ModRegistry.register();
		TRItemGroup.register();

		class_2378.method_10230(class_7923.field_41177, class_2960.method_60655(TechReborn.MOD_ID, "nuke"), ENTITY_NUKE);
	}
}
