/*
 * This file is part of TechReborn, licensed under the MIT License (MIT).
 *
 * Copyright (c) 2020 TechReborn
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package techreborn.compat.rei;

import com.mojang.serialization.Codec;
import com.mojang.serialization.codecs.RecordCodecBuilder;
import me.shedaniel.rei.api.common.category.CategoryIdentifier;
import me.shedaniel.rei.api.common.display.Display;
import me.shedaniel.rei.api.common.display.DisplaySerializer;
import me.shedaniel.rei.api.common.entry.EntryIngredient;
import me.shedaniel.rei.api.common.util.CollectionUtils;
import me.shedaniel.rei.api.common.util.EntryIngredients;
import me.shedaniel.rei.plugin.common.displays.crafting.CraftingDisplay;
import net.minecraft.class_1869;
import net.minecraft.class_2960;
import net.minecraft.class_7923;
import net.minecraft.class_8786;
import net.minecraft.class_9135;
import net.minecraft.class_9139;
import reborncore.common.crafting.RebornRecipe;
import techreborn.init.ModRecipes;
import techreborn.recipe.recipes.RollingMachineRecipe;

import java.util.List;
import java.util.Objects;
import java.util.Optional;

public class RollingMachineDisplay implements CraftingDisplay {
	public static final DisplaySerializer<RollingMachineDisplay> SERIALIZER = DisplaySerializer.of(
		RecordCodecBuilder.mapCodec(instance -> instance.group(
			EntryIngredient.codec().listOf().fieldOf("inputs").forGetter(RollingMachineDisplay::getInputEntries),
			EntryIngredient.codec().listOf().fieldOf("outputs").forGetter(RollingMachineDisplay::getOutputEntries),
			class_2960.field_25139.optionalFieldOf("location").forGetter(RollingMachineDisplay::getDisplayLocation),
			Codec.INT.fieldOf("width").forGetter(RollingMachineDisplay::getWidth),
			Codec.INT.fieldOf("height").forGetter(RollingMachineDisplay::getHeight),
			Codec.INT.fieldOf("energy").forGetter(RollingMachineDisplay::getEnergy),
			Codec.INT.fieldOf("time").forGetter(RollingMachineDisplay::getTime)
		).apply(instance, RollingMachineDisplay::new)),
		class_9139.method_64265(
			EntryIngredient.streamCodec().method_56433(class_9135.method_56363()), RollingMachineDisplay::getInputEntries,
			EntryIngredient.streamCodec().method_56433(class_9135.method_56363()), RollingMachineDisplay::getOutputEntries,
			class_9135.method_56382(class_2960.field_48267), RollingMachineDisplay::getDisplayLocation,
			class_9135.field_49675, RollingMachineDisplay::getWidth,
			class_9135.field_49675, RollingMachineDisplay::getHeight,
			class_9135.field_49675, RollingMachineDisplay::getEnergy,
			class_9135.field_49675, RollingMachineDisplay::getTime,
			RollingMachineDisplay::new
		)
	);

	private final List<EntryIngredient> inputs;
	private final List<EntryIngredient> outputs;
	private final Optional<class_2960> location;
	private final int width;
	private final int height;
	private final int energy;
	private final int time;

	public RollingMachineDisplay(
		List<EntryIngredient> inputs,
		List<EntryIngredient> outputs,
		Optional<class_2960> location,
		int width,
		int height,
		int energy,
		int time
	) {
		this.inputs = inputs;
		this.outputs = outputs;
		this.location = location;
		this.width = width;
		this.height = height;
		this.energy = energy;
		this.time = time;
	}

	public RollingMachineDisplay(class_8786<RebornRecipe> entry) {
		RollingMachineRecipe recipe = (RollingMachineRecipe) entry.comp_1933();
		this.energy = recipe.power();
		this.time = recipe.time();
		class_1869 shapedRecipe = recipe.getShapedRecipe();
		this.inputs = CollectionUtils.map(
			shapedRecipe.method_61693(),
			opt -> opt.map(EntryIngredients::ofIngredient).orElse(EntryIngredient.empty())
		);
		this.outputs = List.of(EntryIngredients.of(shapedRecipe.method_17727(null, null)));
		this.location = Optional.of(entry.comp_1932().method_29177());
		this.width = shapedRecipe.method_8150();
		this.height = shapedRecipe.method_8158();
	}

	public int getEnergy() {
		return energy;
	}

	public int getTime() {
		return time;
	}

	@Override
	public boolean isShapeless() {
		return false;
	}

	@Override
	public int getWidth() {
		return width;
	}

	@Override
	public int getHeight() {
		return height;
	}

	@Override
	public Optional<class_2960> getDisplayLocation() {
		return location;
	}

	@Override
	public List<EntryIngredient> getInputEntries() {
		return inputs;
	}

	@Override
	public List<EntryIngredient> getOutputEntries() {
		return outputs;
	}

	@Override
	public CategoryIdentifier<?> getCategoryIdentifier() {
		return CategoryIdentifier.of(Objects.requireNonNull(class_7923.field_41188.method_10221(ModRecipes.ROLLING_MACHINE)));
	}

	@Override
	public DisplaySerializer<? extends Display> getSerializer() {
		return SERIALIZER;
	}
}
