/*
 * This file is part of TechReborn, licensed under the MIT License (MIT).
 *
 * Copyright (c) 2020 TechReborn
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package techreborn.compat.rei;

import com.google.common.collect.Lists;
import com.mojang.serialization.Codec;
import com.mojang.serialization.codecs.RecordCodecBuilder;
import me.shedaniel.rei.api.common.category.CategoryIdentifier;
import me.shedaniel.rei.api.common.display.Display;
import me.shedaniel.rei.api.common.display.DisplaySerializer;
import me.shedaniel.rei.api.common.entry.EntryIngredient;
import me.shedaniel.rei.api.common.util.EntryIngredients;
import net.minecraft.class_2960;
import net.minecraft.class_9135;
import net.minecraft.class_9139;
import techreborn.recipe.recipes.FluidGeneratorRecipe;

import java.util.List;
import java.util.Optional;

public class FluidGeneratorRecipeDisplay implements Display {
	public static final DisplaySerializer<FluidGeneratorRecipeDisplay> SERIALIZER = DisplaySerializer.of(
		RecordCodecBuilder.mapCodec(instance -> instance.group(
			EntryIngredient.codec().listOf().fieldOf("inputs").forGetter(FluidGeneratorRecipeDisplay::getInputEntries),
			Codec.STRING.fieldOf("category").forGetter(d -> d.category.getIdentifier().toString()),
			Codec.INT.fieldOf("totalEnergy").forGetter(FluidGeneratorRecipeDisplay::getTotalEnergy)
		).apply(instance, FluidGeneratorRecipeDisplay::new)),
		class_9139.method_56436(
			EntryIngredient.streamCodec().method_56433(class_9135.method_56363()),
			FluidGeneratorRecipeDisplay::getInputEntries,
			class_9135.field_48554,
			d -> d.category.getIdentifier().toString(),
			class_9135.field_49675,
			FluidGeneratorRecipeDisplay::getTotalEnergy,
			FluidGeneratorRecipeDisplay::new
		)
	);

	private final List<EntryIngredient> inputs;
	private final CategoryIdentifier<?> category;
	private final int totalEnergy;

	public FluidGeneratorRecipeDisplay(List<EntryIngredient> inputs, String category, int totalEnergy) {
		this.inputs = inputs;
		this.category = CategoryIdentifier.of(category);
		this.totalEnergy = totalEnergy;
	}

	public FluidGeneratorRecipeDisplay(FluidGeneratorRecipe recipe, class_2960 category) {
		this.category = CategoryIdentifier.of(category);
		this.inputs = Lists.newArrayList();
		this.totalEnergy = recipe.power() * 1000;
		inputs.add(EntryIngredients.of(recipe.getFluid(), 1000));
	}

	@Override
	public List<EntryIngredient> getInputEntries() {
		return inputs;
	}

	@Override
	public List<EntryIngredient> getOutputEntries() {
		return Lists.newArrayList();
	}

	@Override
	public CategoryIdentifier<?> getCategoryIdentifier() {
		return category;
	}

	@Override
	public Optional<class_2960> getDisplayLocation() {
		return Optional.empty();
	}

	@Override
	public DisplaySerializer<? extends Display> getSerializer() {
		return SERIALIZER;
	}

	public int getTotalEnergy() {
		return totalEnergy;
	}
}
