/*
 * This file is part of TechReborn, licensed under the MIT License (MIT).
 *
 * Copyright (c) 2020 TechReborn
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package techreborn.client.events;

import reborncore.common.blockentity.MachineBaseBlockEntity;
import techreborn.config.TechRebornConfig;
import techreborn.init.TRContent;

import java.text.DecimalFormat;
import java.text.DecimalFormatSymbols;
import java.text.NumberFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Locale;
import net.minecraft.class_1074;
import net.minecraft.class_124;
import net.minecraft.class_2561;
import net.minecraft.class_310;
import net.minecraft.class_5250;


public class ToolTipAssistUtils {

	// Colour constants
	private static final class_124 instructColour = class_124.field_1078;

	private static final class_124 infoColour = class_124.field_1065;
	private static final class_124 statColour = class_124.field_1065;

	private static final class_124 posColour = class_124.field_1060;
	private static final class_124 negColour = class_124.field_1061;

	public static List<class_2561> getUpgradeStats(TRContent.Upgrades upgradeType, int count, boolean shiftHeld) {
		List<class_2561> tips = new ArrayList<>();
		boolean shouldStackCalculate = count > 1;

		switch (upgradeType) {
			case OVERCLOCKER -> {
				tips.add(getStatStringUnit(class_1074.method_4662("techreborn.tooltip.upgrade.speed_increase"), calculateSpeed(TechRebornConfig.overclockerSpeed * 100, count, shiftHeld), "%", true));
				tips.add(getStatStringUnit(class_1074.method_4662("techreborn.tooltip.upgrade.energy_increase"), calculateEnergyIncrease(TechRebornConfig.overclockerPower + 1, count, shiftHeld), "x", false));
			}
			case TRANSFORMER -> shouldStackCalculate = false;
			case ENERGY_STORAGE -> tips.add(getStatStringUnit(class_1074.method_4662("techreborn.tooltip.upgrade.storage_increase"), calculateValue(TechRebornConfig.energyStoragePower, count, shiftHeld), " E", true));
			case SUPERCONDUCTOR -> tips.add(getStatStringUnit(class_1074.method_4662("techreborn.tooltip.upgrade.flow_increase"), calculateValue(Math.pow(2, (TechRebornConfig.superConductorCount + 2)) * 100, count, shiftHeld), "%", true));
		}

		// Add reminder that they can use shift to calculate the entire stack
		if (shouldStackCalculate && !shiftHeld) {
			tips.add(class_2561.method_43470(instructColour + class_1074.method_4662("techreborn.tooltip.stack_info")));
		}

		return tips;
	}

	public static void addInfo(String inKey, List<class_2561> list) {
		addInfo(inKey, list, true);
	}

	public static void addInfo(String inKey, List<class_2561> list, boolean hidden) {
		String key = ("techreborn.message.info." + inKey);

		if (class_1074.method_4663(key)) {
			if (!hidden || class_310.method_1551().method_74187()) {
				String info = class_1074.method_4662(key);
				List<class_5250> infoLines = Arrays.stream(info.split("\\r?\\n"))
					.map(infoLine -> class_2561.method_43470(infoColour + infoLine)).toList();
				list.addAll(1, infoLines);
			} else {
				list.add(class_2561.method_43470(instructColour + class_1074.method_4662("techreborn.tooltip.more_info")));
			}
		}
	}


	private static int calculateValue(double value, int count, boolean shiftHeld) {
		int calculatedVal;

		if (shiftHeld) {
			calculatedVal = (int) value * Math.min(count, 4);
		} else {
			calculatedVal = (int) value;
		}

		return calculatedVal;
	}

	private static double calculateEnergyIncrease(double value, int count, boolean shiftHeld) {
		double calculatedVal;

		if (shiftHeld) {
			calculatedVal = Math.pow(value, Math.min(count, 4));
		} else {
			calculatedVal = value;
		}

		return calculatedVal;
	}

	private static double calculateSpeed(double value, int count, boolean shiftHeld) {
		double calculatedVal;

		if (shiftHeld) {
			calculatedVal = Math.min(value * Math.min(count, 4), MachineBaseBlockEntity.SPEED_CAP * 100);
		} else {
			calculatedVal = value;
		}

		return calculatedVal;
	}

	private static class_2561 getStatStringUnit(String text, double value, String unit, boolean isPositive) {
		DecimalFormatSymbols symbols = new DecimalFormatSymbols(Locale.US); // Always use dot
		NumberFormat formatter = new DecimalFormat("##.##", symbols); // Round to 2 decimal places
		return class_2561.method_43470(statColour + text + ": " + ((isPositive) ? posColour : negColour) + formatter.format(value) + unit);
	}
}
