/*
 * This file is part of TechReborn, licensed under the MIT License (MIT).
 *
 * Copyright (c) 2020 TechReborn
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package techreborn.blockentity.transformers;

import org.jetbrains.annotations.Nullable;
import reborncore.api.IListInfoProvider;
import reborncore.api.IToolDrop;
import reborncore.common.powerSystem.PowerAcceptorBlockEntity;
import reborncore.common.powerSystem.PowerSystem;
import reborncore.common.powerSystem.RcEnergyTier;
import reborncore.common.util.StringUtils;
import techreborn.blocks.transformers.BlockTransformer;
import techreborn.config.TechRebornConfig;

import java.util.List;
import net.minecraft.class_124;
import net.minecraft.class_1657;
import net.minecraft.class_1799;
import net.minecraft.class_2248;
import net.minecraft.class_2338;
import net.minecraft.class_2350;
import net.minecraft.class_2561;
import net.minecraft.class_2591;
import net.minecraft.class_2680;

/**
 * Created by Rushmead
 */
public class TransformerBlockEntity extends PowerAcceptorBlockEntity implements IToolDrop, IListInfoProvider {

	public final String name;
	public final class_2248 wrenchDrop;
	public final RcEnergyTier inputTier;
	public final RcEnergyTier outputTier;
	public final int maxInput;
	public final int maxOutput;
	public final int maxStorage;

	public TransformerBlockEntity(class_2591<?> blockEntityType, class_2338 pos, class_2680 state, String name, class_2248 wrenchDrop, RcEnergyTier tier) {
		super(blockEntityType, pos, state);
		this.wrenchDrop = wrenchDrop;
		this.inputTier = tier;
		if (tier != RcEnergyTier.MICRO) {
			outputTier = RcEnergyTier.values()[tier.ordinal() - 1];
		} else {
			outputTier = RcEnergyTier.MICRO;
		}
		this.name = name;
		this.maxInput = tier.getMaxInput();
		this.maxOutput = tier.getMaxOutput();
		this.maxStorage = tier.getMaxInput() * 2;

	}

	// PowerAcceptorBlockEntity
	@Override
	protected boolean canAcceptEnergy(@Nullable class_2350 side) {
		if (side == null) {
			return true;
		}
		if (TechRebornConfig.IC2TransformersStyle) {
			return getFacing() == side;
		}
		return getFacing() != side;
	}

	@Override
	protected boolean canProvideEnergy(@Nullable class_2350 side) {
		if (side == null) {
			return true;
		}
		if (TechRebornConfig.IC2TransformersStyle) {
			return getFacing() != side;
		}
		return getFacing() == side;
	}

	@Override
	public long getBaseMaxPower() {
		return maxStorage;
	}

	@Override
	public long getBaseMaxOutput() {
		return outputTier.getMaxOutput();
	}

	@Override
	public long getBaseMaxInput() {
		return inputTier.getMaxInput();
	}

	@Override
	public void checkTier() {
		//Nope
	}

	// MachineBaseBlockEntity
	@Override
	public class_2350 getFacingEnum() {
		if (field_11863 == null) {
			return null;
		}
		class_2248 block = field_11863.method_8320(field_11867).method_26204();
		if (block instanceof BlockTransformer) {
			return ((BlockTransformer) block).getFacing(field_11863.method_8320(field_11867));
		}
		return null;
	}

	@Override
	public boolean canBeUpgraded() {
		return false;
	}

	// IToolDrop
	@Override
	public class_1799 getToolDrop(class_1657 playerIn) {
		return new class_1799(wrenchDrop);
	}

	// IListInfoProvider
	@Override
	public void addInfo(List<class_2561> info, boolean isReal, boolean hasData) {
		info.add(
				class_2561.method_43471("reborncore.tooltip.energy.inputRate")
						.method_27692(class_124.field_1080)
						.method_27693(": ")
						.method_27693(PowerSystem.getLocalizedPower(getMaxInput(null)))
						.method_27692(class_124.field_1065)
		);
		info.add(
				class_2561.method_43471("techreborn.tooltip.input_tier")
						.method_27692(class_124.field_1080)
						.method_27693(": ")
						.method_27693(StringUtils.toFirstCapitalAllLowercase(inputTier.toString()))
						.method_27692(class_124.field_1065)
		);
		info.add(
				class_2561.method_43471("reborncore.tooltip.energy.outputRate")
						.method_27692(class_124.field_1080)
						.method_27693(": ")
						.method_27693(PowerSystem.getLocalizedPower(getMaxOutput(null)))
						.method_27692(class_124.field_1065)
		);
		info.add(
				class_2561.method_43471("techreborn.tooltip.output_tier")
						.method_27692(class_124.field_1080)
						.method_27693(": ")
						.method_27693(StringUtils.toFirstCapitalAllLowercase(outputTier.toString()))
						.method_27692(class_124.field_1065)
		);
	}
}
