/*
 * This file is part of TechReborn, licensed under the MIT License (MIT).
 *
 * Copyright (c) 2020 TechReborn
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package techreborn.blockentity.storage.item;

import net.fabricmc.fabric.api.transfer.v1.item.InventoryStorage;
import net.fabricmc.fabric.api.transfer.v1.item.ItemVariant;
import net.fabricmc.fabric.api.transfer.v1.item.base.SingleStackStorage;
import net.fabricmc.fabric.api.transfer.v1.storage.SlottedStorage;
import net.fabricmc.fabric.api.transfer.v1.storage.Storage;
import net.fabricmc.fabric.api.transfer.v1.storage.base.CombinedSlottedStorage;
import net.minecraft.class_11362;
import net.minecraft.class_11368;
import net.minecraft.class_11372;
import net.minecraft.class_11580;
import net.minecraft.class_124;
import net.minecraft.class_1657;
import net.minecraft.class_1799;
import net.minecraft.class_1809;
import net.minecraft.class_1937;
import net.minecraft.class_2338;
import net.minecraft.class_2350;
import net.minecraft.class_2371;
import net.minecraft.class_2487;
import net.minecraft.class_2509;
import net.minecraft.class_2520;
import net.minecraft.class_2561;
import net.minecraft.class_2680;
import net.minecraft.class_3545;
import net.minecraft.class_6903;
import net.minecraft.class_8942;
import net.minecraft.class_9135;
import net.minecraft.class_9334;
import org.jetbrains.annotations.Nullable;
import reborncore.api.IListInfoProvider;
import reborncore.api.IToolDrop;
import reborncore.api.blockentity.InventoryProvider;
import reborncore.common.blockentity.MachineBaseBlockEntity;
import reborncore.common.screen.BuiltScreenHandler;
import reborncore.common.screen.BuiltScreenHandlerProvider;
import reborncore.common.screen.builder.ScreenHandlerBuilder;
import reborncore.common.util.ItemUtils;
import reborncore.common.util.RebornInventory;
import reborncore.common.util.WorldUtils;
import techreborn.init.TRBlockEntities;
import techreborn.init.TRContent;

import java.util.List;

import static techreborn.TechReborn.LOGGER;

public class StorageUnitBaseBlockEntity extends MachineBaseBlockEntity implements InventoryProvider, IToolDrop, IListInfoProvider, BuiltScreenHandlerProvider {

	// Inventory constants
	public static final int INPUT_SLOT = 0;
	public static final int OUTPUT_SLOT = 1;

	// Client sync variables for GUI, what and how much stored
	public int storedAmount = 0;

	protected final RebornInventory<StorageUnitBaseBlockEntity> inventory;
	private int maxCapacity;
	private int serverCapacity = -1;

	private class_1799 storeItemStack;
	// Fabric transfer API support for the internal stack (one per direction);
	private final SingleStackStorage[] internalStoreStorage = new SingleStackStorage[6];

	private TRContent.StorageUnit type;

	// A locked storage unit will continue behaving as if it contains
	// the locked-in item, even if the stored amount drops to zero.
	private class_1799 lockedItemStack = class_1799.field_8037;

	public StorageUnitBaseBlockEntity(class_2338 pos, class_2680 state) {
		super(TRBlockEntities.STORAGE_UNIT, pos, state);
		inventory = new RebornInventory<>(2, "ItemInventory", 64, this);
	}

	public StorageUnitBaseBlockEntity(class_2338 pos, class_2680 state, TRContent.StorageUnit type) {
		super(TRBlockEntities.STORAGE_UNIT, pos, state);
		inventory = new RebornInventory<>(2, "ItemInventory", 64, this);
		configureEntity(type);
	}
	private void configureEntity(TRContent.StorageUnit type) {
		// Set capacity to local config unless overridden by server
		if(serverCapacity == -1){
			this.maxCapacity = type.capacity;
		}
		storeItemStack = class_1799.field_8037;
		this.type = type;
	}

	public boolean isLocked() {
		return lockedItemStack != class_1799.field_8037;
	}

	public void setLocked(boolean value) {
		if (isLocked() == value) {
			return;
		}

		// Only set lockedItem in response to user input
		class_1799 stack = getStoredStack().method_7972();
		stack.method_7939(1);
		lockedItemStack = value ? stack : class_1799.field_8037;
		syncWithAll();
	}

	public boolean canModifyLocking() {
		// Can always be unlocked
		if (isLocked()) {
			return true;
		}

		// Can only lock if there is an item to lock
		return !method_5442();
	}

	private void populateOutput() {
		// Set to storeItemStack to get the stack type
		class_1799 output = storeItemStack.method_7972();

		int outputSlotCount = inventory.method_5438(OUTPUT_SLOT).method_7947();

		// Set to current outputSlot count
		output.method_7939(outputSlotCount);

		// Calculate amount needed to fill stack in output slot
		int amountToFill = getStoredStack().method_7914() - outputSlotCount;

		if (storeItemStack.method_7947() >= amountToFill) {
			storeItemStack.method_7934(amountToFill);

			if (storeItemStack.method_7960()) {
				storeItemStack = class_1799.field_8037;
			}

			output.method_7933(amountToFill);
		} else {
			output.method_7933(storeItemStack.method_7947());
			storeItemStack = class_1799.field_8037;
		}

		inventory.method_5447(OUTPUT_SLOT, output);
	}

	private void addStoredItemCount(int amount) {
		storeItemStack.method_7933(amount);
	}

	public class_1799 getStoredStack() {
		return storeItemStack.method_7960() ? inventory.method_5438(OUTPUT_SLOT) : storeItemStack;
	}

	// Returns the ItemStack to be displayed to the player via UI / model
	public class_1799 getDisplayedStack() {
		if (!isLocked()) {
			return getStoredStack();
		} else {
			// Render the locked stack even if the unit is empty
			return lockedItemStack;
		}
	}

	public class_1799 getAll() {
		class_1799 returnStack = class_1799.field_8037;

		if (!method_5442()) {
			returnStack = getStoredStack().method_7972();
			returnStack.method_7939(getCurrentCapacity());
		}

		return returnStack;
	}

	public class_1799 processInput(class_1799 inputStack) {
		if (!method_5437(INPUT_SLOT, inputStack)){
			return inputStack;
		}

		// Amount of items that can be added before reaching capacity
		int reminder = maxCapacity - getCurrentCapacity();
		class_2371<class_1799> optionalShulkerStack = ItemUtils.getBlockEntityStacks(inputStack);
		if (isLocked() && ItemUtils.canExtractFromCachedShulker(optionalShulkerStack, lockedItemStack) > 0 ) {
			class_3545<Integer, class_1799> pair = ItemUtils.extractFromShulker(inputStack, optionalShulkerStack, lockedItemStack, reminder);
			if (pair.method_15442() != 0) {
				int amount = pair.method_15442();
				if (storeItemStack.method_7960()) {
					storeItemStack = lockedItemStack.method_7972();
					amount = amount -1;
				}
				addStoredItemCount(amount);
				inputStack = pair.method_15441().method_7972();
				inventory.setHashChanged();
			}
			return inputStack;
		}
		if (inputStack.method_7947() <= reminder) {
			// Add full stack
			if (storeItemStack == class_1799.field_8037){
				// copy input stack into stored if everything is in OUTPUT_SLOT
				storeItemStack = inputStack.method_7972();
			}
			else {
				addStoredItemCount(inputStack.method_7947());
			}

			inputStack = class_1799.field_8037;
		} else {
			// Add only what is needed to reach max capacity
			if (storeItemStack == class_1799.field_8037) {
				storeItemStack = inputStack.method_7972();
				storeItemStack.method_7939(reminder);
			} else {
				addStoredItemCount(reminder);
			}
			inputStack.method_7934(reminder);
		}

		inventory.setHashChanged();
		return inputStack;
	}

	// Creative function
	private void fillToCapacity() {
		storeItemStack = getStoredStack();
		storeItemStack.method_7939(maxCapacity);

		inventory.method_5447(OUTPUT_SLOT, class_1799.field_8037);
	}

	public boolean isFull() {
		return getCurrentCapacity() == maxCapacity;
	}

	public int getCurrentCapacity() {
		return storeItemStack.method_7947() + inventory.method_5438(OUTPUT_SLOT).method_7947();
	}

	// MachineBaseBlockEntity
	@Override
	public void tick(class_1937 world, class_2338 pos, class_2680 state, MachineBaseBlockEntity blockEntity) {
		super.tick(world, pos, state, blockEntity);
		if (world == null || world.method_8608()) {
			return;
		}
		// If there is an item in the input AND stored is less than max capacity
		if (!inventory.method_5438(INPUT_SLOT).method_7960() && !isFull()) {
			inventory.method_5447(INPUT_SLOT, processInput(inventory.method_5438(INPUT_SLOT)));
		}

		// Fill output slot with goodies when stored has items and output count is less than max stack size
		if (storeItemStack.method_7947() > 0 && inventory.method_5438(OUTPUT_SLOT).method_7947() < getStoredStack().method_7914()) {
			populateOutput();
		}

		if (type == TRContent.StorageUnit.CREATIVE) {
			if (!isFull() && !method_5442()) {
				fillToCapacity();
			}
			// void input items for creative storage (#2205)
			if (!inventory.method_5438(INPUT_SLOT).method_7960()){
				inventory.method_5447(INPUT_SLOT, class_1799.field_8037);
			}
		}

		if (inventory.hasChanged()) {
			syncWithAll();
			inventory.resetHasChanged();
		}
	}

	@Override
	public boolean method_5442() {
		return getCurrentCapacity() == 0;
	}

	@Override
	public boolean method_5492(int index, class_1799 stack, @Nullable class_2350 direction) {
		return super.method_5492(index, stack, direction) && method_5437(INPUT_SLOT, stack);
	}

	@Override
	public void method_11014(class_11368 view) {
		super.method_11014(view);

		view.method_71441("unitType").ifPresentOrElse(name -> {
			this.type = TRContent.StorageUnit.valueOf(name);
			configureEntity(type);
		}, () -> {
			this.type = TRContent.StorageUnit.QUANTUM;
		});

		storeItemStack = class_1799.field_8037;

		view.method_71426("storedStack", class_1799.field_24671).ifPresent(stack -> {
			storeItemStack = stack;
		});

		if (!storeItemStack.method_7960()) {
			storeItemStack.method_7939(Math.min(view.method_71424("storedQuantity", 0), this.maxCapacity));
		}

		// Renderer only
		storedAmount = view.method_71424("totalStoredAmount", 0);

		view.method_71426("lockedItem", class_1799.field_24671).ifPresent(stack -> {
			lockedItemStack = stack;
		});
	}

	@Override
	public void method_11007(class_11372 view) {
		super.method_11007(view);

		view.method_71469("unitType", this.type.name());

		if (!storeItemStack.method_7960()) {
			class_1799 temp = storeItemStack.method_7972();
			if (storeItemStack.method_7947() > storeItemStack.method_7914()) {
				temp.method_7939(storeItemStack.method_7914());
			}
			view.method_71468("storedStack", class_1799.field_24671, temp);
			view.method_71465("storedQuantity", Math.min(storeItemStack.method_7947(), maxCapacity));
		} else {
			view.method_71465("storedQuantity", 0);
		}

		// Renderer only
		view.method_71465("totalStoredAmount", getCurrentCapacity());

		if (isLocked()) {
			view.method_71468("lockedItem", class_1799.field_24671, lockedItemStack);
		}
	}

	@Override
	public void onBreak(class_1937 world, class_1657 playerEntity, class_2338 blockPos, class_2680 blockState) {
		super.onBreak(world, playerEntity, blockPos, blockState);

		// No need to drop anything for creative peeps
		if (type == TRContent.StorageUnit.CREATIVE) {
			this.inventory.method_5448();
			return;
		}

		if (storeItemStack != class_1799.field_8037) {
			if (storeItemStack.method_7914() == 64) {
				// Drop stacks (In one clump, reduce lag)
				WorldUtils.dropItem(storeItemStack, world, field_11867);
			} else {
				int size = storeItemStack.method_7914();

				for (int i = 0; i < storeItemStack.method_7947() / size; i++) {
					class_1799 toDrop = storeItemStack.method_7972();
					toDrop.method_7939(size);
					WorldUtils.dropItem(toDrop, world, field_11867);
				}

				if (storeItemStack.method_7947() % size != 0) {
					class_1799 toDrop = storeItemStack.method_7972();
					toDrop.method_7939(storeItemStack.method_7947() % size);
					WorldUtils.dropItem(toDrop, world, field_11867);
				}

			}
		}

		// Inventory gets dropped automatically
	}

	@Override
	public boolean method_5437(int slot, class_1799 inputStack) {
		if (slot != INPUT_SLOT) {
			return false;
		}
		if (inputStack == class_1799.field_8037) {
			return false;
		}
		// Do not allow player heads into storage due to lag. Fix #2888
		if (inputStack.method_7909() instanceof class_1809) {
			return false;
		}
		// do not allow other storage units to avoid NBT overflow. Fix #2580
		if (inputStack.method_31573(TRContent.ItemTags.STORAGE_UNITS)) {
			return false;
		}

		if (isLocked()) {
			//allow shulker bundle extraction when locked
			if (ItemUtils.canExtractAnyFromShulker(inputStack, lockedItemStack)) {
				return true;
			}
			return ItemUtils.isItemEqual(lockedItemStack, inputStack, true, true);
		}

		if (method_5442()){
			return true;
		}

		return ItemUtils.isItemEqual(getStoredStack(), inputStack, true, true);
	}

	@Override
	public boolean canBeUpgraded() {
		return false;
	}

	// InventoryProvider
	@Override
	public RebornInventory<StorageUnitBaseBlockEntity> getInventory() {
		return inventory;
	}

	// IToolDrop
	@Override
	public class_1799 getToolDrop(class_1657 entityPlayer) {
		class_1799 dropStack = new class_1799(getBlockType(), 1);
		if (field_11863 != null){
			try (class_8942.class_11340 logging = new class_8942.class_11340(method_71402(), LOGGER)) {
				class_11362 view = class_11362.method_71459(logging, field_11863.method_30349());
				method_11007(view);
				dropStack.method_57379(class_9334.field_49611, class_11580.method_72535(method_11017(), view.method_71475()));
			}
		}

		return dropStack;
	}

	// IListInfoProvider
	@Override
	public void addInfo(final List<class_2561> info, final boolean isReal, boolean hasData) {
		if (isReal || hasData) {
			if (!this.method_5442()) {
				info.add(
						class_2561.method_43470(String.valueOf(this.getCurrentCapacity()))
								.method_10852(class_2561.method_43471("techreborn.tooltip.unit.divider"))
								.method_10852(this.getStoredStack().method_7964())
				);
			} else {
				info.add(class_2561.method_43471("techreborn.tooltip.unit.empty"));
			}
		}

		info.add(
				class_2561.method_43471("techreborn.tooltip.unit.capacity")
						.method_27692(class_124.field_1080)
						.method_10852(
								class_2561.method_43470(String.valueOf(this.getMaxCapacity()))
										.method_27692(class_124.field_1065)
										.method_27693(" ")
										.method_10852(class_2561.method_43471("techreborn.tooltip.unit.items"))
										.method_27693(" (")
										.method_27693(String.valueOf(this.getMaxCapacity() / 64))
										.method_27693(" ")
										.method_10852(class_2561.method_43471("techreborn.tooltip.unit.stacks"))
										.method_27693(")")
						)
		);
	}

	// BuiltScreenHandlerProvider
	@Override
	public BuiltScreenHandler createScreenHandler(int syncID, final class_1657 playerEntity) {
		return new ScreenHandlerBuilder("chest").player(playerEntity.method_31548()).inventory().hotbar().addInventory()
				.blockEntity(this)
				.slot(INPUT_SLOT, 100, 53)
				.outputSlot(OUTPUT_SLOT, 140, 53)
				.sync(class_9135.field_49675, this::isLockedInt, this::setLockedInt)
				.sync(class_9135.field_48556, this::getStoredStackNBT, this::setStoredStackFromNBT)
				.sync(class_9135.field_49675, this::getStoredAmount, this::setStoredAmount)
				.sync(class_9135.field_49675, this::getMaxCapacity, this::setMaxCapacity)
				.addInventory().create(this, syncID);

		// Note that inventory is synced, and it gets the stack from that
	}

	// The int methods are only for ContainerBuilder.sync()
	private int isLockedInt() {
		return isLocked() ? 1 : 0;
	}

	private void setLockedInt(int lockedInt) {
		setLocked(lockedInt == 1);
	}

	public int getStoredAmount() {
		return this.getCurrentCapacity();
	}

	public void setStoredAmount(int storedAmount) {
		this.storedAmount = storedAmount;
	}

	// Sync between server/client if configs are mis-matched.
	public int getMaxCapacity() {
		return this.maxCapacity;
	}

	public void setMaxCapacity(int maxCapacity) {
		this.maxCapacity = maxCapacity;
		this.serverCapacity = maxCapacity;
	}

	public class_2487 getStoredStackNBT() {
		class_2487 tag = new class_2487();
		class_6903<class_2520> ops = field_11863.method_30349().method_57093(class_2509.field_11560);
		class_1799 stack = getStoredStack();

		tag.method_10569("count", stack.method_7947());

		if (!stack.method_7960()) {
			// We are not allowed to serialize empty or large stacks
			class_1799 singleStack = stack.method_7972();
			singleStack.method_7939(1);
			tag.method_67493("item", class_1799.field_24671, ops, singleStack);
		}

		return tag;
	}

	public void setStoredStackFromNBT(class_2487 tag) {
		if (!tag.method_10545("item")) {
			storeItemStack = class_1799.field_8037;
		} else {
			class_6903<class_2520> ops = field_11863.method_30349().method_57093(class_2509.field_11560);
			storeItemStack = tag.method_67492("item", class_1799.field_24671, ops).orElse(class_1799.field_8037);
		}

		storeItemStack.method_7939(tag.method_10550("count").orElse(0));
	}

	private SlottedStorage<ItemVariant> getInternalStoreStorage(@Nullable class_2350 direction) {
		// Quick fix to handle null sides. https://github.com/TechReborn/TechReborn/issues/3175
		final class_2350 side = direction != null ? direction : class_2350.field_11033;

		if (internalStoreStorage[side.method_10146()] == null) {
			internalStoreStorage[side.method_10146()] = new SingleStackStorage() {
				@Override
				protected class_1799 getStack() {
					return storeItemStack;
				}

				@Override
				protected void setStack(class_1799 stack) {
					if (stack.method_7960()) {
						// Ensure we maintain reference equality to EMPTY
						storeItemStack = class_1799.field_8037;
					} else {
						storeItemStack = stack;
					}
				}

				@Override
				protected int getCapacity(ItemVariant itemVariant) {
					// subtract capacity of output slot (super capacity is the default capacity)
					return maxCapacity - super.getCapacity(itemVariant);
				}

				@Override
				protected boolean canInsert(ItemVariant itemVariant) {
					// Check insertion with the same rules as the input slot
					return StorageUnitBaseBlockEntity.this.method_5492(INPUT_SLOT, itemVariant.toStack(), side);
				}

				@Override
				protected boolean canExtract(ItemVariant itemVariant) {
					// Check extraction with the same rules as the output slot
					return StorageUnitBaseBlockEntity.this.method_5493(OUTPUT_SLOT, itemVariant.toStack(), side);
				}

				@Override
				protected void onFinalCommit() {
					inventory.setHashChanged();
				}
			};
		}
		return internalStoreStorage[side.method_10146()];
	}

	public Storage<ItemVariant> getExposedStorage(class_2350 side) {
		return new CombinedSlottedStorage<>(List.of(
				getInternalStoreStorage(side),
				InventoryStorage.of(this, side)
		));
	}
}
