/*
 * This file is part of TechReborn, licensed under the MIT License (MIT).
 *
 * Copyright (c) 2020 TechReborn
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package techreborn.blockentity.storage.fluid;

import org.apache.commons.lang3.text.WordUtils;
import org.jetbrains.annotations.Nullable;
import reborncore.api.IListInfoProvider;
import reborncore.api.IToolDrop;
import reborncore.api.blockentity.InventoryProvider;
import reborncore.common.blockentity.MachineBaseBlockEntity;
import reborncore.common.fluid.FluidUtils;
import reborncore.common.fluid.FluidValue;
import reborncore.common.fluid.container.FluidInstance;
import reborncore.common.screen.BuiltScreenHandler;
import reborncore.common.screen.BuiltScreenHandlerProvider;
import reborncore.common.screen.builder.ScreenHandlerBuilder;
import reborncore.common.util.RebornInventory;
import reborncore.common.util.Tank;
import techreborn.init.TRBlockEntities;
import techreborn.init.TRContent;

import java.util.List;
import net.minecraft.class_11362;
import net.minecraft.class_11368;
import net.minecraft.class_11372;
import net.minecraft.class_11580;
import net.minecraft.class_124;
import net.minecraft.class_1657;
import net.minecraft.class_1799;
import net.minecraft.class_1937;
import net.minecraft.class_2338;
import net.minecraft.class_2561;
import net.minecraft.class_2680;
import net.minecraft.class_8942;
import net.minecraft.class_9135;
import net.minecraft.class_9334;

import static techreborn.TechReborn.LOGGER;

public class TankUnitBaseBlockEntity extends MachineBaseBlockEntity implements InventoryProvider, IToolDrop, IListInfoProvider, BuiltScreenHandlerProvider {
	protected Tank tank;
	private long serverMaxCapacity = -1;

	protected final RebornInventory<TankUnitBaseBlockEntity> inventory = new RebornInventory<>(2, "TankInventory", 64, this);

	private TRContent.TankUnit type;

	public TankUnitBaseBlockEntity(class_2338 pos, class_2680 state) {
		super(TRBlockEntities.TANK_UNIT, pos, state);
	}

	public TankUnitBaseBlockEntity(class_2338 pos, class_2680 state, TRContent.TankUnit type) {
		super(TRBlockEntities.TANK_UNIT, pos, state);
		configureEntity(type);
	}

	private void configureEntity(TRContent.TankUnit type) {
		this.type = type;
		this.tank = new Tank("TankStorage", serverMaxCapacity == -1 ? type.capacity : FluidValue.fromRaw(serverMaxCapacity));
	}

	protected boolean canDrainTransfer(){
		if (inventory == null || inventory.method_5439() < 2){
			return false;
		}
		class_1799 firstStack = inventory.method_5438(0);
		if (firstStack.method_7960()){
			return false;
		}
		class_1799 secondStack = inventory.method_5438(1);
		return secondStack.method_7947() < secondStack.method_7914();
	}

	// MachineBaseBlockEntity
	@Override
	public void tick(class_1937 world, class_2338 pos, class_2680 state, MachineBaseBlockEntity blockEntity) {
		super.tick(world, pos, state, blockEntity);

		if (world == null || world.method_8608()){
			return;
		}

		if (canDrainTransfer() && FluidUtils.isContainer(inventory.method_5438(0))) {
			boolean didSomething = FluidUtils.drainContainers(tank, inventory, 0, 1);
			if(!didSomething && FluidUtils.fillContainers(tank, inventory, 0, 1)){
				didSomething = true;
			}
			if(didSomething){
				if(inventory.method_5438(1).method_7960() && !inventory.method_5438(0).method_7960() && inventory.method_5438(0).method_7947() == 1){
					inventory.method_5447(1, inventory.method_5438(0));
					inventory.method_5447(0, class_1799.field_8037);
				}
				syncWithAll();
			}
		}
		// allow infinite fluid input for creative tank
		if (type == TRContent.TankUnit.CREATIVE) {
			if (!tank.isEmpty() && !tank.getFluidAmount().equals(tank.getFluidValueCapacity().fraction(2))) {
				tank.setFluidAmount(tank.getFluidValueCapacity().fraction(2));
			}
		}
		// Void excessive fluid in creative tank (#2205)
		if (type == TRContent.TankUnit.CREATIVE && tank.isFull()) {
			FluidUtils.drainContainers(tank, inventory, 0, 1, true);
		}
	}

	@Override
	public boolean canBeUpgraded() {
		return false;
	}

	@Override
	public void method_11014(class_11368 view) {
		super.method_11014(view);
		view.method_71441("unitType").ifPresent(name -> {
			this.type = TRContent.TankUnit.valueOf(name);
			configureEntity(type);
			tank.read(view);
		});
	}

	@Override
	public void method_11007(class_11372 view) {
		super.method_11007(view);
		view.method_71469("unitType", this.type.name());
		tank.write(view);
	}

	@Override
	public FluidValue fluidTransferAmount() {
		// Full capacity should be filled in four minutes (4 minutes * 20 ticks per second / slotTransferSpeed equals 4)
		return type.capacity.fraction(1200);
	}

	// InventoryProvider
	@Override
	public RebornInventory<TankUnitBaseBlockEntity> getInventory() {
		return this.inventory;
	}

	// IToolDrop
	@Override
	public class_1799 getToolDrop(class_1657 playerEntity) {
		class_1799 dropStack = new class_1799(getBlockType(), 1);
		if (field_11863 != null){
			try (class_8942.class_11340 logging = new class_8942.class_11340(method_71402(), LOGGER)) {
				class_11362 view = class_11362.method_71459(logging, field_11863.method_30349());
				method_11007(view);
				dropStack.method_57379(class_9334.field_49611, class_11580.method_72535(method_11017(), view.method_71475()));
			}
		}

		return dropStack;
	}

	// IListInfoProvider
	@Override
	public void addInfo(final List<class_2561> info, final boolean isReal, boolean hasData) {
		if (isReal || hasData) {
			if (!this.tank.getFluidInstance().isEmpty()) {
				info.add(
						class_2561.method_43470(String.valueOf(this.tank.getFluidAmount()))
								.method_10852(class_2561.method_43471("techreborn.tooltip.unit.divider"))
								.method_27693(WordUtils.capitalize(FluidUtils.getFluidName(this.tank.getFluid())))
				);
			} else {
				info.add(class_2561.method_43471("techreborn.tooltip.unit.empty"));
			}
		}
		info.add(
				class_2561.method_43471("techreborn.tooltip.unit.capacity")
						.method_27692(class_124.field_1080)
						.method_10852(class_2561.method_43470(String.valueOf(this.tank.getFluidValueCapacity()))
								.method_27692(class_124.field_1065))
		);
	}

	// BuiltScreenHandlerProvider
	@Override
	public BuiltScreenHandler createScreenHandler(int syncID, final class_1657 player) {
		return new ScreenHandlerBuilder("tank").player(player.method_31548()).inventory().hotbar()
				.addInventory().blockEntity(this).fluidSlot(0, 100, 53).outputSlot(1, 140, 53)
				.sync(tank)
				.sync(class_9135.field_48551, this::getMaxCapacity, this::setMaxCapacity)

				.addInventory().create(this, syncID);
	}

	// Sync between server/client if configs are mis-matched.
	public long getMaxCapacity() {
		return this.tank.getFluidValueCapacity().getRawValue();
	}

	public void setMaxCapacity(long maxCapacity) {
		FluidInstance instance = tank.getFluidInstance();
		this.tank = new Tank("TankStorage", FluidValue.fromRaw(maxCapacity));
		this.tank.setFluidInstance(instance);
		this.serverMaxCapacity = maxCapacity;
	}

	@Nullable
	@Override
	public Tank getTank() {
		return tank;
	}
}
