/*
 * This file is part of TechReborn, licensed under the MIT License (MIT).
 *
 * Copyright (c) 2020 TechReborn
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package techreborn.blockentity.storage.energy.lesu;

import reborncore.common.blockentity.MachineBaseBlockEntity;
import reborncore.common.powerSystem.RcEnergyTier;
import reborncore.common.screen.BuiltScreenHandler;
import reborncore.common.screen.BuiltScreenHandlerProvider;
import reborncore.common.screen.builder.ScreenHandlerBuilder;
import techreborn.blockentity.storage.energy.EnergyStorageBlockEntity;
import techreborn.config.TechRebornConfig;
import techreborn.init.TRBlockEntities;
import techreborn.init.TRContent;

import java.util.HashSet;
import java.util.LinkedList;
import net.minecraft.class_11368;
import net.minecraft.class_11372;
import net.minecraft.class_1657;
import net.minecraft.class_1937;
import net.minecraft.class_2338;
import net.minecraft.class_2350;
import net.minecraft.class_2680;
import net.minecraft.class_9135;

public class LapotronicSUBlockEntity extends EnergyStorageBlockEntity implements BuiltScreenHandlerProvider {

	public static final class_2350[] DIRECTIONS = class_2350.values();
	public static final int DIRECTIONS_LENGTH = DIRECTIONS.length;
	public static final byte[] FLAGS = new byte[DIRECTIONS_LENGTH];
	public static final byte[] OPP_FLAGS = new byte[DIRECTIONS_LENGTH];
	static {
		for (int i = 0; i < DIRECTIONS_LENGTH; i++) {
			FLAGS[i] = (byte) (1 << DIRECTIONS[i].ordinal());
			OPP_FLAGS[i] = (byte) ( 1 << DIRECTIONS[i].method_10153().ordinal());
		}
	}

	private int connectedBlocks = 0;
	public byte neighbors = 0b000000;

	public LapotronicSUBlockEntity(class_2338 pos, class_2680 state) {
		super(TRBlockEntities.LAPOTRONIC_SU, pos, state, "LESU", 2, TRContent.Machine.LAPOTRONIC_SU.block, RcEnergyTier.LOW, TechRebornConfig.lesuStoragePerBlock);
		checkOverfill = false;
		this.maxOutput = TechRebornConfig.lesuBaseOutput;
	}

	private void setMaxStorage() {
		maxStorage = (connectedBlocks + 1) * TechRebornConfig.lesuStoragePerBlock;
		if (maxStorage < 0) {
			maxStorage = Integer.MAX_VALUE;
		}
	}

	private void setIORate() {
		maxOutput = TechRebornConfig.lesuBaseOutput + (connectedBlocks * TechRebornConfig.lesuExtraIOPerBlock);
		if (connectedBlocks < 32) {
			maxInput = RcEnergyTier.LOW.getMaxInput();
		} else if (connectedBlocks < 128) {
			maxInput = RcEnergyTier.MEDIUM.getMaxInput();
		} else {
			maxInput = RcEnergyTier.HIGH.getMaxInput();
		}
	}

	// EnergyStorageBlockEntity
	@Override
	public void tick(class_1937 world, class_2338 pos, class_2680 state, MachineBaseBlockEntity blockEntity) {
		super.tick(world, pos, state, blockEntity);
		if (world == null || world.method_8608()) {
			return;
		}
		if (getEnergy() > getMaxStoredPower()) {
			setEnergy(getMaxStoredPower());
		}
	}

	// MachineBaseBlockEntity
	@Override
	public void onLoad() {
		super.onLoad();
		if (field_11863 == null || field_11863.method_8608()) return;

		// 1. Collect information and change the relationship between surrounding blocks
		byte flagInvalidNeighbors = 0b000000;
		LinkedList<LSUStorageBlockEntity> canConnect = new LinkedList<>();
		HashSet<class_2338> visited = new HashSet<>();
		for (int i = 0; i < DIRECTIONS_LENGTH; i++) {
			if ((neighbors & FLAGS[i]) != 0) {
				if (field_11863.method_8321(field_11867.method_10093(DIRECTIONS[i])) instanceof LSUStorageBlockEntity lsu_storage) {
					if (lsu_storage.master == null) {
						canConnect.add(lsu_storage);
						lsu_storage.addTo(visited);
					}
				} else {
					flagInvalidNeighbors |= FLAGS[i];
				}
			}
		}

		// 2. Compatible with older versions: initialize neighbors
		if (flagInvalidNeighbors != 0b000000) {
			neighbors ^= flagInvalidNeighbors;
			method_5431();
		}

		// 3. Expand outward layer by layer to search for connectable blocks and connect them
		LSUStorageBlockEntity lsu_storage;
		class_2338 linkPos;
		while (!canConnect.isEmpty()) {
			lsu_storage = canConnect.poll();
			lsu_storage.master = this;
			connectedBlocks++;
			for (int i = 0; i < DIRECTIONS_LENGTH; i++) {
				if ((lsu_storage.neighbors & FLAGS[i]) != 0) {
					linkPos = lsu_storage.posOffset(DIRECTIONS[i]);
					if (visited.add(linkPos) && field_11863.method_8321(linkPos) instanceof LSUStorageBlockEntity link_lsu_storage) {
						lsu_storage.links |= FLAGS[i];
						canConnect.add(link_lsu_storage);
					}
				}
			}
		}

		// 4. Update energy configuration
		setMaxStorage();
		setIORate();
	}

	@Override
	public void method_66473(class_2338 pos, class_2680 oldState) {
		disconnectNetwork();
		super.method_66473(pos, oldState);
	}

	public void disconnectNetwork() {
		if (field_11863 == null) return;

		// 1. Collect surrounding connected blocks
		LinkedList<LSUStorageBlockEntity> canDelete = new LinkedList<>();
		LSUStorageBlockEntity lsu_storage;
		for (int i = 0; i < DIRECTIONS_LENGTH; i++) {
			if ((neighbors & FLAGS[i]) != 0) {
				lsu_storage = fastGetLSUS(DIRECTIONS[i]);
				if (lsu_storage.master == this) {
					canDelete.add(lsu_storage);
				}
			}
		}

		// 2. Expand outwards layer by layer, looking for connected blocks and disconnect them
		while (!canDelete.isEmpty()) {
			lsu_storage = canDelete.poll();
			for (int i = 0; i < DIRECTIONS_LENGTH; i++) {
				if ((lsu_storage.links & FLAGS[i]) != 0) {
					canDelete.add(lsu_storage.fastGetLSUS(DIRECTIONS[i]));
				}
			}
			lsu_storage.master = null;
			lsu_storage.links = 0b000000;
		}
	}

	public final void checkNeighbors() {
		if (field_11863 == null) return;
		for (int i = 0; i < DIRECTIONS_LENGTH; i++) {
			if (field_11863.method_8321(field_11867.method_10093(DIRECTIONS[i])) instanceof LSUStorageBlockEntity) {
				neighbors |= FLAGS[i];
			}
		}
		if (neighbors != 0b000000) {
			method_5431();
		}
	}

	public LSUStorageBlockEntity fastGetLSUS(class_2350 direction) {
		assert field_11863 != null;
		return (LSUStorageBlockEntity) field_11863.method_8321(field_11867.method_10093(direction));
	}

	// IContainerProvider
	@Override
	public BuiltScreenHandler createScreenHandler(int syncID, final class_1657 player) {
		return new ScreenHandlerBuilder("lesu").player(player.method_31548()).inventory().hotbar().armor().complete(8, 18)
				.addArmor().addInventory().blockEntity(this).energySlot(0, 62, 45).energySlot(1, 98, 45).syncEnergyValue()
				.sync(class_9135.field_49675, this::getConnectedBlocksNum, this::setConnectedBlocksNum).addInventory().create(this, syncID);
	}

	public int getConnectedBlocksNum() {
		return connectedBlocks;
	}

	public void setConnectedBlocksNum(int value) {
		this.connectedBlocks = value;
		setMaxStorage();
		setIORate();
	}

	@Override
	public void method_11007(class_11372 view) {
		super.method_11007(view);
		view.method_71462("neighbors", neighbors);
	}

	@Override
	public void method_11014(class_11368 view) {
		super.method_11014(view);
		neighbors = view.method_71421("neighbors", (byte) 0b111111);
	}
}
