/*
 * This file is part of TechReborn, licensed under the MIT License (MIT).
 *
 * Copyright (c) 2020 TechReborn
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package techreborn.blockentity.storage.energy.idsu;

import net.fabricmc.fabric.api.transfer.v1.transaction.TransactionContext;
import net.minecraft.class_11368;
import net.minecraft.class_11372;
import net.minecraft.class_1657;
import net.minecraft.class_2338;
import net.minecraft.class_2350;
import net.minecraft.class_2680;
import org.apache.commons.lang3.StringUtils;
import org.jetbrains.annotations.Nullable;
import reborncore.common.powerSystem.RcEnergyTier;
import reborncore.common.screen.BuiltScreenHandler;
import reborncore.common.screen.BuiltScreenHandlerProvider;
import reborncore.common.screen.builder.ScreenHandlerBuilder;
import team.reborn.energy.api.EnergyStorage;
import team.reborn.energy.api.base.DelegatingEnergyStorage;
import team.reborn.energy.api.base.SimpleEnergyStorage;
import techreborn.blockentity.storage.energy.EnergyStorageBlockEntity;
import techreborn.config.TechRebornConfig;
import techreborn.init.TRBlockEntities;
import techreborn.init.TRContent;

public class InterdimensionalSUBlockEntity extends EnergyStorageBlockEntity implements BuiltScreenHandlerProvider {

	public String ownerUdid;

	// This is the energy value that is synced to the client
	private long clientEnergy;

	public InterdimensionalSUBlockEntity(class_2338 pos, class_2680 state) {
		super(TRBlockEntities.INTERDIMENSIONAL_SU, pos, state, "IDSU", 2, TRContent.Machine.INTERDIMENSIONAL_SU.block, RcEnergyTier.INSANE, TechRebornConfig.idsuMaxEnergy);
	}

	@Override
	public EnergyStorage getSideEnergyStorage(@Nullable class_2350 side) {
		if (ownerUdid == null || ownerUdid.isEmpty()) {
			return EnergyStorage.EMPTY;
		}
		if (field_11863.method_8608()) {
			// Can't access the global storage, return a dummy. (Only for existence checks)
			return new SimpleEnergyStorage(TechRebornConfig.idsuMaxEnergy, 0, 0);
		}
		EnergyStorage globalStorage = IDSUManager.getPlayer(field_11863.method_8503(), ownerUdid).getStorage();
		return new DelegatingEnergyStorage(globalStorage, null) {
			@Override
			public long insert(long maxAmount, TransactionContext transaction) {
				return backingStorage.get().insert(Math.min(maxAmount, getMaxInput(side)), transaction);
			}

			@Override
			public long extract(long maxAmount, TransactionContext transaction) {
				return backingStorage.get().extract(Math.min(maxAmount, getMaxOutput(side)), transaction);
			}
		};
	}

	@Override
	public long getStored() {
		if (ownerUdid == null || ownerUdid.isEmpty()) {
			return 0;
		}
		if (field_11863.method_8608()) {
			return clientEnergy;
		}
		return IDSUManager.getPlayer(field_11863.method_8503(), ownerUdid).getEnergy();
	}

	@Override
	public void setStored(long energy) {
		if (ownerUdid == null || ownerUdid.isEmpty()) {
			return;
		}
		if (field_11863.method_8608()) {
			clientEnergy = energy;
		} else {
			IDSUManager.getPlayer(field_11863.method_8503(), ownerUdid).setEnergy(energy);
		}
	}

	@Override
	public void useEnergy(long extract) {
		if (ownerUdid == null || ownerUdid.isEmpty()) {
			return;
		}
		if (field_11863.method_8608()) {
			throw new UnsupportedOperationException("cannot set energy on the client!");
		}
		long energy = IDSUManager.getPlayer(field_11863.method_8503(), ownerUdid).getEnergy();
		if (extract > energy) {
			extract = energy;
		}

		setStored(energy - extract);
	}

	@Override
	protected boolean shouldHandleEnergyNBT() {
		return false;
	}

	@Override
	public void method_11014(class_11368 view) {
		super.method_11014(view);
		this.ownerUdid = view.method_71428("ownerUdid", null);
	}

	@Override
	public void method_11007(class_11372 view) {
		super.method_11007(view);
		if (ownerUdid == null || StringUtils.isEmpty(ownerUdid)) {
			return;
		}
		view.method_71469("ownerUdid", this.ownerUdid);
	}

	@Override
	public BuiltScreenHandler createScreenHandler(int syncID, final class_1657 player) {
		return new ScreenHandlerBuilder("idsu").player(player.method_31548()).inventory().hotbar().armor()
				.complete(8, 18).addArmor().addInventory().blockEntity(this).energySlot(0, 62, 45).energySlot(1, 98, 45)
				.syncEnergyValue().addInventory().create(this, syncID);
	}
}
