/*
 * This file is part of TechReborn, licensed under the MIT License (MIT).
 *
 * Copyright (c) 2022 TechReborn
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package techreborn.blockentity.machine.tier2;

import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import reborncore.common.blockentity.MachineBaseBlockEntity;
import reborncore.common.blocks.BlockMachineBase;
import reborncore.common.fluid.FluidValue;
import reborncore.common.fluid.container.FluidInstance;
import reborncore.common.screen.BuiltScreenHandler;
import reborncore.common.screen.BuiltScreenHandlerProvider;
import reborncore.common.screen.builder.ScreenHandlerBuilder;
import reborncore.common.util.RebornInventory;
import reborncore.common.util.Tank;
import techreborn.blockentity.machine.GenericMachineBlockEntity;
import techreborn.config.TechRebornConfig;
import techreborn.init.TRBlockEntities;
import techreborn.init.TRContent;

import java.util.ArrayList;
import net.minecraft.class_11368;
import net.minecraft.class_11372;
import net.minecraft.class_1264;
import net.minecraft.class_1657;
import net.minecraft.class_1792;
import net.minecraft.class_1799;
import net.minecraft.class_1802;
import net.minecraft.class_1937;
import net.minecraft.class_2246;
import net.minecraft.class_2248;
import net.minecraft.class_2338;
import net.minecraft.class_2371;
import net.minecraft.class_2680;
import net.minecraft.class_3417;
import net.minecraft.class_3419;
import net.minecraft.class_3610;
import net.minecraft.class_3611;
import net.minecraft.class_3612;
import net.minecraft.class_5321;
import net.minecraft.class_9135;

/**
 * @author maxvar (coding), ashendi (textures)
 */
public class PumpBlockEntity extends GenericMachineBlockEntity implements BuiltScreenHandlerProvider {

	public static final FluidValue TANK_CAPACITY = FluidValue.BUCKET.multiply(16);
	public static final int DEFAULT_RANGE = 10;
	public static final int DEFAULT_DEPTH = 10;
	public static final int MIN_RANGE = 0;
	public static final int MIN_DEPTH = 1;
	public static final int MAX_RANGE = 50;
	public static final int MAX_DEPTH = 50;
	private java.util.Iterator<class_2338> finder;
	@Nullable
	private Tank tank;
	private boolean exhausted;
	private class_2338 pumpedTargetBlockPos;
	private long timeToPump;
	private int range;
	private int depth;

	public PumpBlockEntity(class_2338 pos, class_2680 state) {
		super(TRBlockEntities.PUMP, pos, state, "Pump", TechRebornConfig.pumpMaxInput, TechRebornConfig.pumpMaxEnergy, TRContent.Machine.PUMP.block, 0);
		this.inventory = new RebornInventory<>(1, "PumpBlockEntity", 64, this);
		this.exhausted = false;
		this.range = DEFAULT_RANGE;
		this.depth = DEFAULT_DEPTH;
	}

	public boolean getExhausted() {
		return exhausted;
	}

	public void setExhausted(boolean exhausted) {
		this.exhausted = exhausted;
	}

	public int getRange() {
		return range;
	}

	public void setRange(int newRange) {
		if (newRange < MIN_RANGE) newRange = MIN_RANGE;
		if (newRange > MAX_RANGE) newRange = MAX_RANGE;
		if (newRange != range) {
			range = newRange;
			reset();
		}
	}

	public int getDepth() {
		return depth;
	}

	public void setDepth(int newDepth) {
		if (newDepth < MIN_DEPTH) newDepth = MIN_DEPTH;
		if (newDepth > MAX_DEPTH) newDepth = MAX_DEPTH;
		if (newDepth != depth) {
			depth = newDepth;
			reset();
		}
	}

	private void reset() {
		finder = null;
		exhausted = false;
		pumpedTargetBlockPos = null;
		field_11863.method_8501(field_11867, field_11863.method_8320(field_11867).method_11657(BlockMachineBase.ACTIVE, false));
	}

	private void setupFinder() {
		this.finder = new BlockPosIterable(field_11867, range, depth).iterator();
	}

	@Override
	@NotNull
	public Tank getTank() {
		if (this.tank == null) {
			this.tank = createTank();
		}

		return tank;
	}

	private Tank createTank() {
		return new Tank("PumpBlockEntity", TANK_CAPACITY);
	}

	@Override
	public class_1799 getToolDrop(class_1657 p0) {
		return TRContent.Machine.PUMP.getStack();
	}

	@Override
	public void method_11014(class_11368 view) {
		super.method_11014(view);
		getTank().read(view);
		this.range = view.method_71424("range", 0);
		this.depth = view.method_71424("depth", 0);
		finder = null;
	}

	@Override
	public void method_11007(class_11372 view) {
		super.method_11007(view);
		getTank().write(view);
		view.method_71465("range", range);
		view.method_71465("depth", depth);
	}

	@Override
	public BuiltScreenHandler createScreenHandler(int syncID, class_1657 player) {
		return new ScreenHandlerBuilder("pump")
			.player(player.method_31548()).inventory().hotbar().addInventory().blockEntity(this)
			.energySlot(0, 8, 72)
			.sync(getTank())
			.syncEnergyValue()
			.sync(class_9135.field_49675, this::getDepth, this::setDepth)
			.sync(class_9135.field_49675, this::getRange, this::setRange)
			.sync(class_9135.field_48547, this::getExhausted, this::setExhausted)
			.addInventory()
			.create(this, syncID);
	}

	@Override
	public void tick(class_1937 world, class_2338 pos, class_2680 state, MachineBaseBlockEntity blockEntity) {
		super.tick(world, pos, state, blockEntity);

		if (world == null || world.method_8608()) return;

		//do nothing if all liquids have been exhausted
		if (this.exhausted) return;

		//has a target block to pump?
		if (pumpedTargetBlockPos != null) {
			//pumping time completed?
			//has space to store?
			//has enough energy to pump?
			if ((world.method_8510() >= timeToPump)) {
				//not enough energy to pump?
				if (getEnergy() < (long) (TechRebornConfig.pumpEnergyToCollect * getPowerMultiplier())) {
					//don't drop target, retry it again later
					timeToPump = world.method_8510() + (long) (TechRebornConfig.pumpTicksToComplete * (1 - getSpeedMultiplier()));
					return;
				}
				//recheck the target
				class_2680 blockState = world.method_8320(pumpedTargetBlockPos);
				class_3611 fluid = getFluid(blockState);
				//no longer fluid there?
				if (fluid == class_3612.field_15906) {
					//play oops
					if (!isMuffled()) {
						world.method_8396(null, this.field_11867, class_3417.field_14701, class_3419.field_15245, 1.0f, 1.0f);
					}
					//drop target (and find the next)
					pumpedTargetBlockPos = null;
					world.method_8501(pos, world.method_8320(pos).method_11657(BlockMachineBase.ACTIVE, false));
					return;
				}
				//cannot fit fluid into the tank?
				if (!getTank().canFit(fluid, FluidValue.BUCKET)) {
					//don't drop target, retry it again later
					timeToPump = world.method_8510() + (long) (TechRebornConfig.pumpTicksToComplete * (1 - getSpeedMultiplier()));
					return;
				}
				//fill tank
				if (getTank().getFluidInstance().isEmpty()) {
					getTank().setFluidInstance(new FluidInstance(fluid, FluidValue.BUCKET));
				} else {
					getTank().modifyFluid(fluidInstance -> fluidInstance.addAmount(FluidValue.BUCKET));
				}
				//play sound
				if (!isMuffled()) {
					world.method_8396(null, this.field_11867, getTank().getFluid().method_32359().orElse(class_3417.field_15126), class_3419.field_15245, 1.0f, 1.0f);
				}
				//consume energy
				this.useEnergy((long) (TechRebornConfig.pumpEnergyToCollect * getPowerMultiplier()));
				//extract drops
				class_2371<class_1799> drops = getDrops(blockState);
				if (!drops.isEmpty()) class_1264.method_17349(world, pumpedTargetBlockPos, drops);
				//replace target with solid based on dimension
				final class_2248 replacementBlock;
				final class_5321<class_1937> worldRegistryKey = world.method_27983();
				if (worldRegistryKey == class_1937.field_25180) replacementBlock = class_2246.field_23869;
				else if (worldRegistryKey == class_1937.field_25181) replacementBlock = class_2246.field_10471;
				else replacementBlock = class_2246.field_10445;
				world.method_8501(pumpedTargetBlockPos, replacementBlock.method_9564());
				pumpedTargetBlockPos = null;
			}
		} else if (!getTank().isFull()) {
			//find next target
			findNextToPump(world);
			if (pumpedTargetBlockPos != null) {
				timeToPump = world.method_8510() + (long) (TechRebornConfig.pumpTicksToComplete * (1 - getSpeedMultiplier()));
			} else {
				//else - consider exhausted
				world.method_8501(pos, world.method_8320(pos).method_11657(BlockMachineBase.ACTIVE, false));
				this.exhausted = true;
			}
		}

	}

	private void findNextToPump(class_1937 world) {
		if (finder == null) {
			setupFinder();
		}
		while (finder.hasNext()) {
			class_2338 blockPos = finder.next();

			class_2680 blockState = world.method_8320(blockPos);
			class_3611 fluid = getFluid(blockState);
			if (fluid != class_3612.field_15906 && (fluid == getTank().getFluid() || getTank().getFluid() == class_3612.field_15906)) {
				//if any found - start pumping
				world.method_8501(field_11867, world.method_8320(field_11867).method_11657(BlockMachineBase.ACTIVE, true));
				pumpedTargetBlockPos = blockPos;
				return;
			}
		}
	}

	@NotNull
	private class_3611 getFluid(class_2680 blockState) {
		class_3610 fluidState = blockState.method_26227();
		class_3611 fluid = fluidState.method_15772();
		if (fluidState.method_15761() == 8) return fluid;
		else return class_3612.field_15906;

	}

	@NotNull
	private class_2371<class_1799> getDrops(class_2680 blockState) {
		class_2248 block = blockState.method_26204();
		class_1792 item = block.method_8389();
		if (item == class_1802.field_8162) {
			return class_2371.method_37434(0);
		} else {
			class_1799 itemStack = item.method_7854();
			return class_2371.method_10213(1, itemStack);
		}
	}

	public void handleDepthGuiInputFromClient(int buttonAmount) {
		setDepth(depth + buttonAmount);
	}

	public void handleRangeGuiInputFromClient(int buttonAmount) {
		setRange(range + buttonAmount);
	}

	static class BlockPosIterable implements Iterable<class_2338> {
		int index;
		final int layerSize;
		final int m;
		final ArrayList<MeasuredPos> layer;

		public BlockPosIterable(class_2338 centerTop, int range, int depth) {
			this.index = 0;
			this.layerSize = (range * 2 + 1) * (range * 2 + 1);
			this.m = layerSize * depth;
			layer = new ArrayList<>(layerSize);
			for (int i = centerTop.method_10263() - range; i <= centerTop.method_10263() + range; i++)
				for (int j = centerTop.method_10260() - range; j <= centerTop.method_10260() + range; j++)
					layer.add(new MeasuredPos(i, centerTop.method_10264(), j, centerTop.method_40081(i, centerTop.method_10264(), j)));
			layer.sort((o1, o2) -> (int) (o1.weight - o2.weight));
		}

		/**
		 * Returns an iterator over elements of type {@code T}.
		 *
		 * @return an Iterator.
		 */
		@NotNull
		@Override
		public java.util.Iterator<class_2338> iterator() {
			return new java.util.Iterator<>() {
				@Override
				public boolean hasNext() {
					return index < m;
				}

				@Override
				public class_2338 next() {
					final class_2338 pos;
					if (TechRebornConfig.pumpIterateOutwards) {
						pos = layer.get(index % layerSize).method_10087(1 + index / layerSize);
					} else {
						pos = layer.get((m - index - 1) % layerSize).method_10087(1 + (m - index - 1) / layerSize);
					}
					index++;
					return pos;
				}
			};
		}

		static class MeasuredPos extends class_2338 {
			private final double weight;

			public MeasuredPos(int x, int y, int z, double weight) {
				super(x, y, z);
				this.weight = weight;
			}
		}
	}
}
