/*
 * This file is part of TechReborn, licensed under the MIT License (MIT).
 *
 * Copyright (c) 2020 TechReborn
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package techreborn.blockentity.machine.tier2;

import org.jetbrains.annotations.Nullable;
import reborncore.api.IToolDrop;
import reborncore.common.blockentity.MachineBaseBlockEntity;
import reborncore.common.blockentity.RedstoneConfiguration;
import reborncore.common.powerSystem.PowerAcceptorBlockEntity;
import reborncore.common.screen.BuiltScreenHandler;
import reborncore.common.screen.BuiltScreenHandlerProvider;
import reborncore.common.screen.builder.ScreenHandlerBuilder;
import techreborn.config.TechRebornConfig;
import techreborn.init.TRBlockEntities;
import techreborn.init.TRContent;

import java.util.List;
import net.minecraft.class_11368;
import net.minecraft.class_11372;
import net.minecraft.class_1297;
import net.minecraft.class_1657;
import net.minecraft.class_1799;
import net.minecraft.class_1937;
import net.minecraft.class_2338;
import net.minecraft.class_2350;
import net.minecraft.class_238;
import net.minecraft.class_2680;
import net.minecraft.class_3417;
import net.minecraft.class_3419;
import net.minecraft.class_9135;

public class LaunchpadBlockEntity extends PowerAcceptorBlockEntity implements IToolDrop, BuiltScreenHandlerProvider {

	public static final int MAX_SELECTION = 3;
	private int selection = TechRebornConfig.launchpadDefaultSelection;

	public LaunchpadBlockEntity(class_2338 pos, class_2680 state) {
		super(TRBlockEntities.LAUNCHPAD, pos, state);
	}

	public void handleGuiInputFromClient(int amount) {
		selection += amount;
		ensureSelectionInRange();
	}

	public void ensureSelectionInRange() {
		if (selection > MAX_SELECTION) {
			selection = MAX_SELECTION;
		}
		if (selection <= 0) {
			selection = 0;
		}
	}

	public double selectedSpeed() {
		return switch(selection) {
			case 0 -> TechRebornConfig.launchpadSpeedLow;
			case 1 -> TechRebornConfig.launchpadSpeedMedium;
			case 2 -> TechRebornConfig.launchpadSpeedHigh;
			case MAX_SELECTION -> TechRebornConfig.launchpadSpeedExtreme;
			default -> throw new IllegalArgumentException("Impossible launchpad selection value!");
		};
	}

	public int selectedEnergyCost() {
		return switch(selection) {
			case 0 -> TechRebornConfig.launchpadEnergyLow;
			case 1 -> TechRebornConfig.launchpadEnergyMedium;
			case 2 -> TechRebornConfig.launchpadEnergyHigh;
			case MAX_SELECTION -> TechRebornConfig.launchpadEnergyExtreme;
			default -> throw new IllegalArgumentException("Impossible launchpad selection value!");
		};
	}

	public String selectedTranslationKey() {
		return switch(selection) {
			case 0 -> "techreborn.message.info.block.techreborn.launchpad.low";
			case 1 -> "techreborn.message.info.block.techreborn.launchpad.medium";
			case 2 -> "techreborn.message.info.block.techreborn.launchpad.high";
			case MAX_SELECTION -> "techreborn.message.info.block.techreborn.launchpad.extreme";
			default -> throw new IllegalArgumentException("Impossible launchpad selection value!");
		};
	}

	// PowerAcceptorBlockEntity
	@Override
	public void tick(class_1937 world, class_2338 pos, class_2680 state, MachineBaseBlockEntity blockEntity) {
		super.tick(world, pos, state, blockEntity);
		if (world == null || getStored() <= 0 || !isActive(RedstoneConfiguration.Element.POWER_IO)) {
			return;
		}

		if (world.method_8510() % TechRebornConfig.launchpadInterval != 0) {
			return;
		}

		ensureSelectionInRange();
		final double speed = selectedSpeed();
		final int energyCost = selectedEnergyCost();

		if (getStored() > energyCost) {
			List<class_1297> entities = world.method_18467(class_1297.class, new class_238(0d,1d,0d,1d,2d,1d).method_996(pos));
			if (entities.isEmpty()) {
				return;
			}
			world.method_8396(null, pos, class_3417.field_15134, class_3419.field_15245, 1f, 1f);
			for (class_1297 entity : entities) {
				entity.method_5762(0d, speed, 0d);
			}
			useEnergy(energyCost);
		}
	}

	@Override
	public long getBaseMaxPower() {
		return TechRebornConfig.launchpadMaxEnergy;
	}

	@Override
	public boolean canProvideEnergy(@Nullable class_2350 side) {
		return false;
	}

	@Override
	public long getBaseMaxOutput() {
		return 0;
	}

	@Override
	public long getBaseMaxInput() {
		return TechRebornConfig.launchpadMaxInput;
	}

	@Override
	public void method_11014(class_11368 view) {
		super.method_11014(view);
		selection = view.method_71424("selection", 0);
	}

	@Override
	public void method_11007(class_11372 view) {
		super.method_11007(view);
		view.method_71465("selection", selection);
	}

	// MachineBaseBlockEntity
	@Override
	public boolean hasSlotConfig() {
		return false;
	}

	@Override
	public boolean canBeUpgraded() {
		return false;
	}

	// IToolDrop
	@Override
	public class_1799 getToolDrop(class_1657 p0) {
		return TRContent.Machine.LAUNCHPAD.getStack();
	}

	// BuiltScreenHandlerProvider
	@Override
	public BuiltScreenHandler createScreenHandler(int syncID, class_1657 player) {
		return new ScreenHandlerBuilder("launchpad")
				.player(player.method_31548())
				.inventory().hotbar().addInventory()
				.blockEntity(this)
				.syncEnergyValue()
				.sync(class_9135.field_49675, this::getSelection, this::setSelection)
				.addInventory().create(this, syncID);
	}

	public int getSelection() {
		return selection;
	}

	public void setSelection(int selection) {
		this.selection = selection;
	}
}
