/*
 * This file is part of TechReborn, licensed under the MIT License (MIT).
 *
 * Copyright (c) 2020 TechReborn
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package techreborn.blockentity.machine.tier2;

import it.unimi.dsi.fastutil.objects.ObjectArrayList;
import org.jetbrains.annotations.Nullable;
import reborncore.api.IToolDrop;
import reborncore.api.blockentity.InventoryProvider;
import reborncore.common.blockentity.MachineBaseBlockEntity;
import reborncore.common.powerSystem.PowerAcceptorBlockEntity;
import reborncore.common.screen.BuiltScreenHandler;
import reborncore.common.screen.BuiltScreenHandlerProvider;
import reborncore.common.screen.builder.ScreenHandlerBuilder;
import reborncore.common.util.ItemUtils;
import reborncore.common.util.RebornInventory;
import techreborn.config.TechRebornConfig;
import techreborn.init.TRBlockEntities;
import techreborn.init.TRContent;

import java.util.List;
import net.minecraft.class_1657;
import net.minecraft.class_173;
import net.minecraft.class_1799;
import net.minecraft.class_181;
import net.minecraft.class_1937;
import net.minecraft.class_2338;
import net.minecraft.class_2350;
import net.minecraft.class_243;
import net.minecraft.class_2680;
import net.minecraft.class_3218;
import net.minecraft.class_3610;
import net.minecraft.class_3612;
import net.minecraft.class_39;
import net.minecraft.class_52;
import net.minecraft.class_8567;

public class FishingStationBlockEntity extends PowerAcceptorBlockEntity implements IToolDrop, InventoryProvider, BuiltScreenHandlerProvider {

	private final RebornInventory<FishingStationBlockEntity> inventory = new RebornInventory<>(7, "FishingStationBlockEntity", 64, this);

	public FishingStationBlockEntity(class_2338 pos, class_2680 state) {
		super(TRBlockEntities.FISHING_STATION, pos, state);
	}

	private boolean insertIntoInv(List<class_1799> stacks) {
		boolean result = false;
		for (class_1799 stack : stacks) {
			for (int i = 0; i < 6; i++) {
				if (insertIntoInv(i, stack)) result = true;
				if (stack.method_7960()) break;
			}
		}
		return result;
	}

	private boolean insertIntoInv(int slot, class_1799 stack) {
		class_1799 targetStack = inventory.method_5438(slot);
		if (targetStack.method_7960()) {
			inventory.method_5447(slot, stack.method_7972());
			stack.method_7934(stack.method_7947());
			return true;
		} else {
			if (ItemUtils.isItemEqual(stack, targetStack, true, false)) {
				int freeStackSpace = targetStack.method_7914() - targetStack.method_7947();
				if (freeStackSpace > 0) {
					int transferAmount = Math.min(freeStackSpace, stack.method_7947());
					targetStack.method_7933(transferAmount);
					stack.method_7934(transferAmount);
					return true;
				}
			}
		}
		return false;
	}

	// PowerAcceptorBlockEntity
	@Override
	public void tick(class_1937 world, class_2338 pos, class_2680 state, MachineBaseBlockEntity blockEntity) {
		super.tick(world, pos, state, blockEntity);
		if (world == null || world.method_8608()){
			return;
		}

		charge(6);

		long useRequirement = getEuPerTick(TechRebornConfig.fishingStationEnergyPerCatch);

		if (getStored() < useRequirement) {
			return;
		}

		int speed = (int) Math.round(getSpeedMultiplier() / TechRebornConfig.overclockerSpeed) + 1;

		if (world.method_8510() % (TechRebornConfig.fishingStationInterval/speed) != 0) {
			return;
		}

		class_2338 frontPos = pos.method_10093(getFacing());
		class_3610 frontFluid = world.method_8316(frontPos);
		if (!frontFluid.method_33659(class_3612.field_15910)) {
			return;
		}


		final class_8567 lootContextParameterSet = new class_8567.class_8568((class_3218) world)
			.method_51874(class_181.field_24424, class_243.method_24953(frontPos))
			.method_51874(class_181.field_1229, TRContent.Machine.FISHING_STATION.getStack())
			.method_51875(class_173.field_1176);

		final class_52 lootTable = world.method_8503().method_58576().method_58295(class_39.field_353);
		final ObjectArrayList<class_1799> list = lootTable.method_51878(lootContextParameterSet);
		if (insertIntoInv(list)){
			useEnergy(useRequirement);
		}
	}

	@Override
	public long getBaseMaxPower() {
		return TechRebornConfig.fishingStationMaxEnergy;
	}

	@Override
	public boolean canProvideEnergy(@Nullable class_2350 side) {
		return false;
	}

	@Override
	public long getBaseMaxOutput() {
		return 0;
	}

	@Override
	public long getBaseMaxInput() {
		return TechRebornConfig.fishingStationMaxInput;
	}

	// IToolDrop
	@Override
	public class_1799 getToolDrop(class_1657 p0) {
		return TRContent.Machine.FISHING_STATION.getStack();
	}

	// InventoryProvider
	@Override
	public RebornInventory<FishingStationBlockEntity> getInventory() {
		return this.inventory;
	}

	// BuiltScreenHandlerProvider
	@Override
	public BuiltScreenHandler createScreenHandler(int syncID, class_1657 player) {
		return new ScreenHandlerBuilder("fishing_station")
				.player(player.method_31548())
				.inventory().hotbar().addInventory()
				.blockEntity(this)
				.outputSlot(0, 30, 22).outputSlot(1, 48, 22)
				.outputSlot(2, 30, 40).outputSlot(3, 48, 40)
				.outputSlot(4, 30, 58).outputSlot(5, 48, 58)
				.energySlot(6, 8, 72).syncEnergyValue()
				.addInventory().create(this, syncID);
	}
}
