/*
 * This file is part of TechReborn, licensed under the MIT License (MIT).
 *
 * Copyright (c) 2020 TechReborn
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package techreborn.blockentity.machine.tier1;

import org.apache.commons.lang3.tuple.Pair;
import org.jetbrains.annotations.Nullable;
import reborncore.api.IToolDrop;
import reborncore.api.blockentity.InventoryProvider;
import reborncore.common.blockentity.MachineBaseBlockEntity;
import reborncore.common.blocks.BlockMachineBase;
import reborncore.common.crafting.RecipeUtils;
import reborncore.common.powerSystem.PowerAcceptorBlockEntity;
import reborncore.common.screen.BuiltScreenHandler;
import reborncore.common.screen.BuiltScreenHandlerProvider;
import reborncore.common.screen.builder.ScreenHandlerBuilder;
import reborncore.common.util.ItemUtils;
import reborncore.common.util.RebornInventory;
import techreborn.config.TechRebornConfig;
import techreborn.init.ModRecipes;
import techreborn.init.TRBlockEntities;
import techreborn.init.TRContent;
import techreborn.recipe.recipes.RollingMachineRecipe;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;
import net.minecraft.class_11368;
import net.minecraft.class_11372;
import net.minecraft.class_1657;
import net.minecraft.class_1703;
import net.minecraft.class_1715;
import net.minecraft.class_1792;
import net.minecraft.class_1799;
import net.minecraft.class_1856;
import net.minecraft.class_1937;
import net.minecraft.class_2338;
import net.minecraft.class_2350;
import net.minecraft.class_2680;
import net.minecraft.class_9135;
import net.minecraft.class_9694;

// TODO add tick and power bars.

public class RollingMachineBlockEntity extends PowerAcceptorBlockEntity
	implements IToolDrop, InventoryProvider, BuiltScreenHandlerProvider {

	public int[] craftingSlots = new int[]{0, 1, 2, 3, 4, 5, 6, 7, 8};
	private class_1715 craftCache;
	public final RebornInventory<RollingMachineBlockEntity> inventory = new RebornInventory<>(12, "RollingMachineBlockEntity", 64, this);
	public boolean isRunning;
	public int tickTime = 0;
	// Only synced to the client
	public int currentRecipeTime = 0;
	public class_1799 currentRecipeOutput = class_1799.field_8037;
	public RollingMachineRecipe currentRecipe;
	private final int outputSlot;
	public boolean locked = false;
	public int balanceSlot = 0;
	RollingMachineRecipe lastRecipe = null;
	private List<class_1792> cachedInventoryStructure = null;
	public RollingMachineBlockEntity(class_2338 pos, class_2680 state) {
		super(TRBlockEntities.ROLLING_MACHINE, pos, state);
		outputSlot = 9;
	}

	@Override
	public long getBaseMaxPower() {
		return TechRebornConfig.rollingMachineMaxEnergy;
	}

	@Override
	public boolean canProvideEnergy(@Nullable class_2350 side) {
		return false;
	}

	@Override
	public long getBaseMaxOutput() {
		return 0;
	}

	@Override
	public long getBaseMaxInput() {
		return TechRebornConfig.rollingMachineMaxInput;
	}

	@Override
	public void tick(class_1937 world, class_2338 pos, class_2680 state, MachineBaseBlockEntity blockEntity) {
		super.tick(world, pos, state, blockEntity);
		if (world == null || world.method_8608()) {
			return;
		}
		charge(10);

		class_1715 craftMatrix = getCraftingMatrix(true);
		currentRecipe = findMatchingRecipe(craftMatrix, world);
		if (currentRecipe != null) {
			if (world.method_8510() % 2 == 0) {
				balanceRecipe(craftMatrix);
			}
			currentRecipeOutput = currentRecipe.getShapedRecipe().method_17727(recipeInput(craftMatrix), method_10997().method_30349());
		} else {
			currentRecipeOutput = class_1799.field_8037;
		}
		craftMatrix = getCraftingMatrix();

		if (currentRecipeOutput.method_7960() || !checkNotEmpty(craftMatrix)){
			// can't make anyway, reject.
			tickTime = 0;
			setIsActive(false);
			return;
		}
		// Now we ensured we can make something. Check energy state.
		if (getStored() > getEuPerTick(currentRecipe.power())
			&& canMake(craftMatrix)) {
			setIsActive(true);
			useEnergy(getEuPerTick(currentRecipe.power()));
			tickTime++;
		} else {
			setIsActive(false);
			return;
		}
		// Cached recipe or valid recipe exists.
		// checked if we can make at least one.
		if (tickTime >= currentRecipeTime) {
			//craft one
			if (inventory.method_5438(outputSlot).method_7960()) {
				inventory.method_5447(outputSlot, currentRecipeOutput.method_7972());
			}
			else {
				// we checked stack can fit in output slot in canMake()
				inventory.method_5438(outputSlot).method_7933(currentRecipeOutput.method_7947());
			}
			tickTime = 0;
			currentRecipeTime = Math.max((int) (currentRecipe.time() * (1.0 - getSpeedMultiplier())), 1);
			for (int i = 0; i < craftMatrix.method_5439(); i++) {
				inventory.shrinkSlot(i, 1);
			}
			if (!locked) {
				currentRecipeOutput = class_1799.field_8037;
				currentRecipe = null;
			}
		}
	}

	public void setIsActive(boolean active) {
		if (active == isRunning) {
			return;
		}
		isRunning = active;
		if (this.method_10997().method_8320(this.method_11016()).method_26204() instanceof BlockMachineBase blockMachineBase) {
			blockMachineBase.setActive(active, this.method_10997(), this.method_11016());
		}
		this.method_10997().method_8413(this.method_11016(), this.method_10997().method_8320(this.method_11016()), this.method_10997().method_8320(this.method_11016()), 3);
	}

	public Optional<class_1715> balanceRecipe(class_1715 craftCache) {
		if (currentRecipe == null) {
			return Optional.empty();
		}
		if (field_11863.method_8608()) {
			return Optional.empty();
		}
		if (!locked) {
			return Optional.empty();
		}
		if (craftCache.method_5442()) {
			return Optional.empty();
		}
		balanceSlot++;
		if (balanceSlot > craftCache.method_5439()) {
			balanceSlot = 0;
		}
		// Find the best slot for each item in a recipe, and move it if needed
		class_1799 sourceStack = inventory.method_5438(balanceSlot);
		if (sourceStack.method_7960()) {
			return Optional.empty();
		}
		List<Integer> possibleSlots = new ArrayList<>();
		for (int s = 0; s < currentRecipe.method_61671().method_64675().size(); s++) {
			class_1799 stackInSlot = inventory.method_5438(s);
			class_1856 ingredient = currentRecipe.method_61671().method_64675().get(s);
			if (ingredient != null && ingredient.method_8093(sourceStack)) {
				if (stackInSlot.method_7960()) {
					possibleSlots.add(s);
				} else if (stackInSlot.method_7909() == sourceStack.method_7909()) {
					possibleSlots.add(s);
				}
			}
		}

		if (!possibleSlots.isEmpty()) {
			int totalItems = possibleSlots.stream()
				.mapToInt(value -> inventory.method_5438(value).method_7947()).sum();
			int slots = possibleSlots.size();

			// This makes an array of ints with the best possible slot distribution
			int[] split = new int[slots];
			int remainder = totalItems % slots;
			Arrays.fill(split, totalItems / slots);
			while (remainder > 0) {
				for (int i = 0; i < split.length; i++) {
					if (remainder > 0) {
						split[i] += 1;
						remainder--;
					}
				}
			}

			List<Integer> slotEnvTyperubution = possibleSlots.stream()
				.mapToInt(value -> inventory.method_5438(value).method_7947())
				.boxed().collect(Collectors.toList());

			boolean needsBalance = false;
			for (int required : split) {
				if (slotEnvTyperubution.contains(required)) {
					// We need to remove the int, not at the int, this seems to work around that
					slotEnvTyperubution.remove(Integer.valueOf(required));
				} else {
					needsBalance = true;
				}
			}
			if (!needsBalance) {
				return Optional.empty();
			}
		} else {
			return Optional.empty();
		}

		// Slot, count
		Pair<Integer, Integer> bestSlot = null;
		for (Integer slot : possibleSlots) {
			class_1799 slotStack = inventory.method_5438(slot);
			if (slotStack.method_7960()) {
				bestSlot = Pair.of(slot, 0);
			}
			if (bestSlot == null) {
				bestSlot = Pair.of(slot, slotStack.method_7947());
			} else if (bestSlot.getRight() >= slotStack.method_7947()) {
				bestSlot = Pair.of(slot, slotStack.method_7947());
			}
		}
		if (bestSlot == null
			|| bestSlot.getLeft() == balanceSlot
			|| bestSlot.getRight() == sourceStack.method_7947()
			|| inventory.method_5438(bestSlot.getLeft()).method_7960()
			|| !ItemUtils.isItemEqual(sourceStack, inventory.method_5438(bestSlot.getLeft()), true, true)) {
			return Optional.empty();
		}
		sourceStack.method_7934(1);
		inventory.method_5438(bestSlot.getLeft()).method_7933(1);
		inventory.setHashChanged();

		return Optional.of(getCraftingMatrix());
	}

	private class_1715 getCraftingMatrix() {
		return getCraftingMatrix(false);
	}

	private class_1715 getCraftingMatrix(boolean forceRefresh) {
		if (craftCache == null) {
			craftCache = new class_1715(new RollingBEContainer(), 3, 3);
		}
		if (forceRefresh || inventory.hasChanged()) {
			for (int i = 0; i < 9; i++) {
				craftCache.method_5447(i, inventory.method_5438(i).method_7972());
			}
			inventory.resetHasChanged();
		}
		return craftCache;
	}
	private List<class_1792> fastIntlayout(){
		if (this.inventory == null) return null;
		ArrayList<class_1792> arrayList = new ArrayList<>(9);
		for (int i = 0; i < 9; i++){
			arrayList.add(this.inventory.method_5438(i).method_7909());
		}
		return arrayList;
	}

	private boolean checkNotEmpty(class_1715 craftMatrix) {
		//checks if inventory is empty or considered quasi-empty.
		if (locked) {
			boolean returnValue = false;
			// for locked condition, we need to check if inventory contains item and all slots are empty or has more than one item.
			for (int i = 0; i < craftMatrix.method_5439(); i++) {
				class_1799 stack1 = craftMatrix.method_5438(i);
				if (stack1.method_7947() == 1) {
					return false;
				}
				if (stack1.method_7947() > 1) {
					returnValue = true;
				}
			}
			return returnValue;
		}
		else {
			for (int i = 0; i < craftMatrix.method_5439(); i++) {
				class_1799 stack1 = craftMatrix.method_5438(i);
				if (!stack1.method_7960()) {
					return true;
				}
			}
		}
		return false;
	}

	public boolean canMake(class_1715 craftMatrix) {
		class_1799 stack = findMatchingRecipeOutput(craftMatrix, this.field_11863);
		if (stack.method_7960()) {
			return false;
		}
		class_1799 output = inventory.method_5438(outputSlot);
		if (output.method_7960()) {
			return true;
		}
		return ItemUtils.isItemEqual(stack, output, true, true) && output.method_7947() + stack.method_7947() <= output.method_7914();
	}

	public List<RollingMachineRecipe> getAllRecipe(class_1937 world) {
		return RecipeUtils.getRecipes(world, ModRecipes.ROLLING_MACHINE);
	}

	public class_1799 findMatchingRecipeOutput(class_1715 inv, class_1937 world) {
		RollingMachineRecipe recipe = findMatchingRecipe(inv, world);
		if (recipe == null) {
			return class_1799.field_8037;
		}
		return recipe.assemble(null, method_10997().method_30349());
	}

	public RollingMachineRecipe findMatchingRecipe(class_1715 inv, class_1937 world) {
		if (isCorrectCachedInventory()){
			return lastRecipe;
		}
		cachedInventoryStructure = fastIntlayout();
		class_9694 input = recipeInput(inv);
		for (RollingMachineRecipe recipe : getAllRecipe(world)) {
			if (recipe.getShapedRecipe().method_17728(input, world)) {
				lastRecipe = recipe;
				currentRecipeTime = Math.max((int) (recipe.time() * (1.0 - getSpeedMultiplier())), 1);
				return recipe;
			}
		}
		lastRecipe = null;
		currentRecipeTime = 0;
		return null;
	}

	private boolean isCorrectCachedInventory(){
		if (cachedInventoryStructure == null){
			return false;
		}
		List<class_1792> current = fastIntlayout();
		if (current == null || current.size() != this.cachedInventoryStructure.size()){
			return false;
		}
		for (int i = 0; i < current.size(); i++ ){
			if (current.get(i) != this.cachedInventoryStructure.get(i)){
				return false;
			}
		}
		return true;
	}

	@Override
	public class_1799 getToolDrop(final class_1657 entityPlayer) {
		return TRContent.Machine.ROLLING_MACHINE.getStack();
	}

	@Override
	public void method_11014(class_11368 view) {
		super.method_11014(view);
		this.isRunning = view.method_71433("isRunning", false);
		this.tickTime = view.method_71424("tickTime", 0);
		this.locked = view.method_71433("locked", false);
	}

	@Override
	public void method_11007(class_11372 view) {
		super.method_11007(view);
		view.method_71472("isRunning", this.isRunning);
		view.method_71465("tickTime", this.tickTime);
		view.method_71472("locked", locked);
	}

	@Override
	public RebornInventory<RollingMachineBlockEntity> getInventory() {
		return inventory;
	}

	public int getBurnTime() {
		return tickTime;
	}

	public void setBurnTime(final int burnTime) {
		this.tickTime = burnTime;
	}

	public int getBurnTimeRemainingScaled(final int scale) {
		if (tickTime == 0 || Math.max((int) (currentRecipe.time() * (1.0 - getSpeedMultiplier())), 1) == 0) {
			return 0;
		}
		return tickTime * scale / Math.max((int) (currentRecipe.time() * (1.0 - getSpeedMultiplier())), 1);
	}

	@Override
	public BuiltScreenHandler createScreenHandler(int syncID, final class_1657 player) {
		return new ScreenHandlerBuilder("rollingmachine").player(player.method_31548())
			.inventory().hotbar()
			.addInventory().blockEntity(this)
			.slot(0, 30, 22).slot(1, 48, 22).slot(2, 66, 22)
			.slot(3, 30, 40).slot(4, 48, 40).slot(5, 66, 40)
			.slot(6, 30, 58).slot(7, 48, 58).slot(8, 66, 58)
			.onCraft(inv -> this.inventory.method_5447(1, findMatchingRecipeOutput(getCraftingMatrix(), this.field_11863)))
			.outputSlot(9, 124, 40)
			.energySlot(10, 8, 70)
			.syncEnergyValue().sync(class_9135.field_49675, this::getBurnTime, this::setBurnTime).sync(class_9135.field_49675, this::getLockedInt, this::setLockedInt)
			.sync(class_9135.field_49675, this::getCurrentRecipeTime, this::setCurrentRecipeTime).addInventory().create(this, syncID);
	}

	public int getCurrentRecipeTime() {
		return currentRecipeTime;
	}

	public RollingMachineBlockEntity setCurrentRecipeTime(int currentRecipeTime) {
		this.currentRecipeTime = currentRecipeTime;
		return this;
	}

	// Easiest way to sync back to the client
	public int getLockedInt() {
		return locked ? 1 : 0;
	}

	public void setLockedInt(int lockedInt) {
		locked = lockedInt == 1;
	}

	public int getProgressScaled(final int scale) {
		if (tickTime != 0 && currentRecipeTime != 0) {
			return tickTime * scale / currentRecipeTime;
		}
		return 0;
	}

	private static class RollingBEContainer extends class_1703 {

		protected RollingBEContainer() {
			super(null, 0);
		}

		@Override
		public class_1799 method_7601(class_1657 player, int slot) {
			return class_1799.field_8037;
		}

		@Override
		public boolean method_7597(final class_1657 playerEntity) {
			return true;
		}

	}

	@Override
	public boolean canBeUpgraded() {
		return true;
	}

	private static class_9694 recipeInput(class_1715 inventory) {
		List<class_1799> stacks = new ArrayList<>(inventory.method_5439());
		for (int i = 0; i < inventory.method_5439(); i++) {
			stacks.add(inventory.method_5438(i));
		}
		return class_9694.method_59986(inventory.method_17398(), inventory.method_17397(), stacks);
	}
}
