/*
 * This file is part of TechReborn, licensed under the MIT License (MIT).
 *
 * Copyright (c) 2020 TechReborn
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package techreborn.blockentity.machine.tier1;

import net.fabricmc.fabric.api.transfer.v1.item.ItemStorage;
import net.fabricmc.fabric.api.transfer.v1.item.ItemVariant;
import net.fabricmc.fabric.api.transfer.v1.storage.Storage;
import net.fabricmc.fabric.api.transfer.v1.transaction.Transaction;
import net.minecraft.class_1657;
import net.minecraft.class_1799;
import net.minecraft.class_1937;
import net.minecraft.class_2246;
import net.minecraft.class_2338;
import net.minecraft.class_2350;
import net.minecraft.class_2680;
import net.minecraft.class_3419;
import reborncore.common.blockentity.MachineBaseBlockEntity;
import reborncore.common.util.WorldUtils;
import techreborn.blocks.machine.tier1.ResinBasinBlock;
import techreborn.blocks.misc.BlockRubberLog;
import techreborn.config.TechRebornConfig;
import techreborn.init.ModSounds;
import techreborn.init.TRBlockEntities;
import techreborn.init.TRContent;

public class ResinBasinBlockEntity extends MachineBaseBlockEntity {
	private class_2350 direction = class_2350.field_11043;

	// State
	private boolean isPouring = false;
	private boolean isFull = false;

	private int pouringTimer = 0;

	public ResinBasinBlockEntity(class_2338 pos, class_2680 state) {
		super(TRBlockEntities.RESIN_BASIN, pos, state);

		/* TODO is this the right place? */
		this.isFull = state.method_11654(ResinBasinBlock.FULL);

		if (state.method_11654(ResinBasinBlock.POURING)) {
			this.isPouring = true;
			pouringTimer = TechRebornConfig.sapTimeTicks;
		}
	}

	@Override
	public void tick(class_1937 world, class_2338 pos, class_2680 state, MachineBaseBlockEntity blockEntity) {
		super.tick(world, pos, state, blockEntity);
		if (world == null || world.method_8608()) return;

		boolean shouldUpdateState = false;

		if (isPouring) {
			pouringTimer--;

			// Play pouring audio
			if (world.method_8510() % 20 == 0) {
				world.method_8396(null, pos, ModSounds.SAP_EXTRACT, class_3419.field_15245, 1F, 1F);
			}

			if (pouringTimer <= 0) {
				isPouring = false;
				isFull = true;
				shouldUpdateState = true;
			}
		}

		// Try and deposit
		if (isFull) {
			// Get inventory
			Storage<ItemVariant> invBelow = getInventoryBelow();
			if (invBelow != null) {
				try (Transaction tx = Transaction.openOuter()) {
					int sentAmount = getSapAmount();
					if (invBelow.insert(ItemVariant.of(TRContent.Parts.SAP), sentAmount, tx) > 0) {
						tx.commit();
						isFull = false;
						shouldUpdateState = true;
					}
				}
			}
		}

		boolean readyToHarvest = !isFull && !isPouring;

		// Ensuring it's placed on a log
		if ((readyToHarvest || world.method_8510() % 20 == 0) && !validPlacement()) {
			// Not placed on log, drop on ground
			world.method_8501(pos, class_2246.field_10124.method_9564());
			WorldUtils.dropItem(TRContent.Machine.RESIN_BASIN.method_8389(), world, pos);
			return;
		}

		if (readyToHarvest) {
			// Check for rubber
			if (world.method_8510() % TechRebornConfig.checkForSapTime == 0) {
				class_2338 targetRubber = getLogWithSap();

				if (targetRubber != null) {
					// We have a valid sap log, harvest it
					world.method_8501(targetRubber, world.method_8320(targetRubber).method_11657(BlockRubberLog.HAS_SAP, false).method_11657(BlockRubberLog.SAP_SIDE, class_2350.method_10139(0)));
					isPouring = true;
					pouringTimer = TechRebornConfig.sapTimeTicks;
					shouldUpdateState = true;
				}
			}
		}

		if (shouldUpdateState) {
			setPouringState(isPouring);
			setFullState(isFull);
		}
	}

	public int getSapAmount() {
		if (!isFull)
			return 0;
		return (Math.random() <= 0.5) ? 1 : 2;
	}

	public class_1799 empty() {
		if (isFull) {
			int sapAmount = getSapAmount();

			this.isPouring = false;
			this.isFull = false;
			setFullState(false);
			setPouringState(false);

			return new class_1799(TRContent.Parts.SAP, sapAmount);
		}

		return new class_1799(TRContent.Parts.SAP, 0);
	}

	@Override
	public void onBreak(class_1937 world, class_1657 playerEntity, class_2338 blockPos, class_2680 blockState) {
		super.onBreak(world, playerEntity, blockPos, blockState);

		// Drop a sap if full
		if (this.isFull) {
			class_1799 out = new class_1799(TRContent.Parts.SAP, getSapAmount());
			WorldUtils.dropItem(out, world, field_11867);
		}
	}

	@Override
	public void onLoad() {
		super.onLoad();

		if (field_11863 == null || field_11863.method_8608()) return;

		// Set facing
		direction = field_11863.method_8320(field_11867).method_11654(ResinBasinBlock.FACING).method_10153();
	}

	private Storage<ItemVariant> getInventoryBelow() {
		return ItemStorage.SIDED.find(this.method_10997(), this.field_11867.method_10093(class_2350.field_11033), class_2350.field_11036);
	}

	private boolean validPlacement() {
		return field_11863.method_8320(this.field_11867.method_10093(direction)).method_26204() == TRContent.RUBBER_LOG;
	}


	private class_2338 getLogWithSap() {
		// Checking origin block
		class_2338 originPos = this.field_11867.method_10093(direction);
		class_2680 originState = field_11863.method_8320(originPos);

		if (originState.method_11654(BlockRubberLog.HAS_SAP)) {
			return originPos;
		}

		boolean shouldExit = false;
		class_2338 current = originPos;

		// Progress Up
		while (!shouldExit) {
			current = current.method_10093(class_2350.field_11036);

			class_2680 state = field_11863.method_8320(current);
			if (state.method_26204() == TRContent.RUBBER_LOG) {
				if (state.method_11654(BlockRubberLog.HAS_SAP)) {
					return current;
				}
			} else {
				shouldExit = true;
			}
		}

		current = originPos;
		shouldExit = false;
		// Progress Down
		while (!shouldExit) {
			current = current.method_10093(class_2350.field_11033);

			class_2680 state = field_11863.method_8320(current);
			if (state.method_26204() == TRContent.RUBBER_LOG) {
				if (state.method_11654(BlockRubberLog.HAS_SAP)) {
					return current;
				}
			} else {
				shouldExit = true;
			}
		}

		// Could not find a rubber log with sap
		return null;
	}

	private void setPouringState(boolean value) {
		if (field_11863 != null) {
			field_11863.method_8501(field_11867, field_11863.method_8320(field_11867).method_11657(ResinBasinBlock.POURING, value));
		}
	}

	private void setFullState(boolean value) {
		if (field_11863 != null) {
			field_11863.method_8501(field_11867, field_11863.method_8320(field_11867).method_11657(ResinBasinBlock.FULL, value));
		}
	}

	@Override
	public boolean hasSlotConfig() {
		return false;
	}

	@Override
	public boolean canBeUpgraded() {
		return false;
	}
}
