/*
 * This file is part of TechReborn, licensed under the MIT License (MIT).
 *
 * Copyright (c) 2020 TechReborn
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package techreborn.blockentity.machine.tier1;

import org.jetbrains.annotations.Nullable;
import reborncore.api.IToolDrop;
import reborncore.api.blockentity.InventoryProvider;
import reborncore.common.blockentity.MachineBaseBlockEntity;
import reborncore.common.blockentity.MultiblockWriter;
import reborncore.common.powerSystem.PowerAcceptorBlockEntity;
import reborncore.common.screen.BuiltScreenHandler;
import reborncore.common.screen.BuiltScreenHandlerProvider;
import reborncore.common.screen.builder.ScreenHandlerBuilder;
import reborncore.common.util.ItemUtils;
import reborncore.common.util.RebornInventory;
import techreborn.blocks.lighting.LampBlock;
import techreborn.blocks.misc.BlockRubberLog;
import techreborn.config.TechRebornConfig;
import techreborn.init.TRBlockEntities;
import techreborn.init.TRContent;

import java.util.Collections;
import java.util.List;
import net.minecraft.class_1657;
import net.minecraft.class_1799;
import net.minecraft.class_1802;
import net.minecraft.class_1937;
import net.minecraft.class_2211;
import net.minecraft.class_2246;
import net.minecraft.class_2248;
import net.minecraft.class_2256;
import net.minecraft.class_2261;
import net.minecraft.class_2266;
import net.minecraft.class_2282;
import net.minecraft.class_2302;
import net.minecraft.class_2338;
import net.minecraft.class_2350;
import net.minecraft.class_2421;
import net.minecraft.class_2445;
import net.minecraft.class_2523;
import net.minecraft.class_2680;
import net.minecraft.class_2741;
import net.minecraft.class_2758;
import net.minecraft.class_3218;
import net.minecraft.class_3830;
import net.minecraft.class_5803;

public class GreenhouseControllerBlockEntity extends PowerAcceptorBlockEntity
		implements IToolDrop, InventoryProvider, BuiltScreenHandlerProvider {

	private final RebornInventory<GreenhouseControllerBlockEntity> inventory = new RebornInventory<>(7, "GreenhouseControllerBlockEntity", 64, this);
	private class_2338 multiblockCenter;
	private int ticksToNextMultiblockCheck = 0;
	private boolean growthBoost = false;
	private int workingIndex = 0;
	// number of blocks from center
	private final int range = 4;


	public GreenhouseControllerBlockEntity(class_2338 pos, class_2680 state) {
		super(TRBlockEntities.GREENHOUSE_CONTROLLER, pos, state);
	}

	private void workCycle() {
		if (field_11863 == null){
			return;
		}

		int size = range * 2 + 1;
		int offsetX = workingIndex % size;
		int offsetZ = workingIndex / size;
		class_2338 corner = multiblockCenter.method_10069(-range, 0, -range);
		class_2338 blockPos = corner.method_10069(offsetX, 0, offsetZ);

		workingIndex = (workingIndex + 1) % (size * size);
		class_2680 blockState = field_11863.method_8320(blockPos);
		class_2248 block = blockState.method_26204();

		if (growthBoost) {
			if (block instanceof class_2256 || block instanceof class_2261
					|| block instanceof class_2523	|| block instanceof class_2266
			) {
				if (getStored() > TechRebornConfig.greenhouseControllerEnergyPerBonemeal) {
					useEnergy(TechRebornConfig.greenhouseControllerEnergyPerBonemeal);
					blockState.method_26199((class_3218) field_11863, blockPos, field_11863.field_9229);
				}
			}
		}

		if (getStored() < TechRebornConfig.greenhouseControllerEnergyPerHarvest){
			return;
		}

		if (block instanceof class_2302 cropBlock) {
			processAgedCrop(blockState, blockPos, cropBlock.method_9824(), ((class_2302) block).method_9827(), 0);
		} else if (block instanceof class_2421) {
			processAgedCrop(blockState, blockPos, class_2421.field_11306, 3, 0);
		} else if (block instanceof class_3830) {
			processAgedCrop(blockState, blockPos, class_3830.field_17000, 3, 1);
		} else if (block instanceof class_2282) {
			processAgedCrop(blockState, blockPos, class_2282.field_10779, 2, 0);
		} else if (block instanceof class_2445) {
			if (tryHarvestBlock(blockState, blockPos)) {
				field_11863.method_22352(blockPos, false);
			}
		} else if (block instanceof class_2523
				|| block instanceof class_2266
				|| block instanceof class_2211
		) {
			// If we can break bottom block we should at least remove all of them up to top, so they don't break automatically
			boolean breakBlocks = false;
			for (int y = 1; (blockState = field_11863.method_8320(blockPos.method_10086(y))).method_26204() == block; y++) {
				if (y == 1) {
					breakBlocks = tryHarvestBlock(blockState, blockPos.method_10086(y));
				} else {
					tryHarvestBlock(blockState, blockPos.method_10086(y));
				}
				if (breakBlocks) field_11863.method_22352(blockPos.method_10086(y), false);
			}
		} else if (block instanceof BlockRubberLog) {
			for (int y = 0; (blockState = field_11863.method_8320(blockPos.method_10086(y))).method_26204() == block && y < 10; y++) {
				if (blockState.method_11654(BlockRubberLog.HAS_SAP)
						&& insertIntoInv(Collections.singletonList(TRContent.Parts.SAP.getStack()))
				) {
					useEnergy(TechRebornConfig.greenhouseControllerEnergyPerHarvest);
					field_11863.method_8501(blockPos.method_10086(y), blockState.method_11657(BlockRubberLog.HAS_SAP, false).method_11657(BlockRubberLog.SAP_SIDE, class_2350.method_10139(0)));
				}
			}
		} else if (block instanceof class_5803){
			for (int y=0; (blockState = field_11863.method_8320(blockPos.method_10086(y))).method_26204() instanceof class_5803; y++){
				if (blockState.method_11654(class_2741.field_28716)
					&& insertIntoInv(Collections.singletonList(new class_1799(class_1802.field_28659, 1)))
				){
					useEnergy(TechRebornConfig.greenhouseControllerEnergyPerHarvest);
					field_11863.method_8501(blockPos.method_10086(y), blockState.method_11657(class_2741.field_28716, false));
				}
			}
		}
	}

	private void processAgedCrop(class_2680 blockState, class_2338 blockPos, class_2758 ageProperty, int maxAge, int newAge) {
		if (field_11863 == null) {
			return;
		}
		if (blockState.method_11654(ageProperty) < maxAge) {
			return;
		}
		if (tryHarvestBlock(blockState, blockPos)) {
			field_11863.method_8652(blockPos, blockState.method_11657(ageProperty, newAge), 2);
		}
	}

	private boolean tryHarvestBlock(class_2680 blockState, class_2338 blockPos) {
		if (insertIntoInv(class_2248.method_9562(blockState, (class_3218) field_11863, blockPos, null))) {
			useEnergy(TechRebornConfig.greenhouseControllerEnergyPerHarvest);
			return true;
		}
		return false;
	}

	private boolean insertIntoInv(List<class_1799> stacks) {
		boolean result = false;
		for (class_1799 stack : stacks) {
			for (int i = 0; i < 6; i++) {
				if (insertIntoInv(i, stack)) result = true;
				if (stack.method_7960()) break;
			}
		}
		return result;
	}

	private boolean insertIntoInv(int slot, class_1799 stack) {
		class_1799 targetStack = inventory.method_5438(slot);
		if (targetStack.method_7960()) {
			inventory.method_5447(slot, stack.method_7972());
			stack.method_7934(stack.method_7947());
			return true;
		} else {
			if (ItemUtils.isItemEqual(stack, targetStack, true, false)) {
				int freeStackSpace = targetStack.method_7914() - targetStack.method_7947();
				if (freeStackSpace > 0) {
					int transferAmount = Math.min(freeStackSpace, stack.method_7947());
					targetStack.method_7933(transferAmount);
					stack.method_7934(transferAmount);
					return true;
				}
			}
		}
		return false;
	}

	// PowerAcceptorBlockEntity
	@Override
	public void tick(class_1937 world, class_2338 pos, class_2680 state, MachineBaseBlockEntity blockEntity) {
		super.tick(world, pos, state, blockEntity);
		if (world == null || world.method_8608()){
			return;
		}
		if (multiblockCenter == null) {
			multiblockCenter = pos.method_10079(getFacing().method_10153(), range + 1);
		}

		charge(6);

		if (getStored() < getEuPerTick(TechRebornConfig.greenhouseControllerEnergyPerTick)) {
			return;
		}

		if (--ticksToNextMultiblockCheck < 0) {
			growthBoost = isShapeValid();
			ticksToNextMultiblockCheck = 200;
		}

		if (world.method_8510() % 20 == 0) {
			double cyclesLimit = getSpeedMultiplier() * 4 + 1;
			while (cyclesLimit-- > 0) {
				workCycle();
			}
		}
	}

	@Override
	public boolean canProvideEnergy(@Nullable class_2350 side) {
		return false;
	}

	@Override
	public long getBaseMaxPower() {
		return TechRebornConfig.greenhouseControllerMaxEnergy;
	}

	@Override
	public long getBaseMaxOutput() {
		return 0;
	}

	@Override
	public long getBaseMaxInput() {
		return TechRebornConfig.greenhouseControllerMaxInput;
	}

	// MachineBaseBlockEntity
	@Override
	public void writeMultiblock(MultiblockWriter writer) {
		class_2680 lamp = TRContent.Machine.LAMP_INCANDESCENT.block.method_9564().method_11657(class_2741.field_12525, class_2350.field_11033);
		class_2680 crop = class_2246.field_10029.method_9564();

		for (int i = 0; i < 3; i++) {
			for (int j = -1; j < 2; j++) {
				writer.add(i * 3 + 2, 3, j * 3, (world, pos) -> LampBlock.isActive(world.method_8320(pos)), lamp);
			}
		}

		for (int i = 1; i <= 9; i++) {
			for (int j = -4; j <= 4; j++) {
				writer.add(i, 0, j, (world, pos) -> true, crop);
			}
		}
	}

	// IToolDrop
	@Override
	public class_1799 getToolDrop(class_1657 entityPlayer) {
		return TRContent.Machine.GREENHOUSE_CONTROLLER.getStack();
	}

	// InventoryProvider
	@Override
	public RebornInventory<GreenhouseControllerBlockEntity> getInventory() {
		return this.inventory;
	}

	// BuiltScreenHandlerProvider
	@Override
	public BuiltScreenHandler createScreenHandler(int syncID, class_1657 player) {
		return new ScreenHandlerBuilder("greenhousecontroller").player(player.method_31548()).inventory().hotbar().addInventory()
				.blockEntity(this)
				.outputSlot(0, 30, 22).outputSlot(1, 48, 22)
				.outputSlot(2, 30, 40).outputSlot(3, 48, 40)
				.outputSlot(4, 30, 58).outputSlot(5, 48, 58)
				.energySlot(6, 8, 72).syncEnergyValue()
				.addInventory().create(this, syncID);
	}

}
