/*
 * This file is part of TechReborn, licensed under the MIT License (MIT).
 *
 * Copyright (c) 2020 TechReborn
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package techreborn.blockentity.machine.tier1;

import org.jetbrains.annotations.Nullable;
import reborncore.api.IToolDrop;
import reborncore.common.blockentity.MachineBaseBlockEntity;
import reborncore.common.blockentity.RedstoneConfiguration;
import reborncore.common.powerSystem.PowerAcceptorBlockEntity;
import reborncore.common.screen.BuiltScreenHandler;
import reborncore.common.screen.BuiltScreenHandlerProvider;
import reborncore.common.screen.builder.ScreenHandlerBuilder;
import techreborn.config.TechRebornConfig;
import techreborn.init.TRBlockEntities;
import techreborn.init.TRContent;

import java.util.List;
import java.util.Optional;
import net.minecraft.class_1657;
import net.minecraft.class_1799;
import net.minecraft.class_1937;
import net.minecraft.class_2338;
import net.minecraft.class_2350;
import net.minecraft.class_238;
import net.minecraft.class_2382;
import net.minecraft.class_243;
import net.minecraft.class_2586;
import net.minecraft.class_2680;
import net.minecraft.class_3218;
import net.minecraft.class_3417;
import net.minecraft.class_3419;
import net.minecraft.class_5454;

public class ElevatorBlockEntity extends PowerAcceptorBlockEntity implements IToolDrop, BuiltScreenHandlerProvider {

	public ElevatorBlockEntity(class_2338 pos, class_2680 state) {
		super(TRBlockEntities.ELEVATOR, pos, state);
	}

	/**
	 * @param targetPos the position of another elevator
	 */
	public boolean isRunning(final class_2338 targetPos) {
		// null-safe because of the following instanceof
		final class_2586 entity = method_10997().method_8321(targetPos);
		if (!(entity instanceof ElevatorBlockEntity)) {
			return false;
		}
		return ((ElevatorBlockEntity)entity).getStored() > 0;
	}

	/**
	 * @param targetPos the position will be checked to be an elevator or air
	 */
	public boolean isAirOrElevator(final class_2338 targetPos) {
		return method_10997().method_22347(targetPos) || method_10997().method_8321(targetPos) instanceof ElevatorBlockEntity;
	}

	/**
	 * @param targetPos the position of another elevator
	 */
	public boolean isFree(final class_2338 targetPos) {
		return method_10997().method_8320(targetPos.method_10084()).method_26215() && method_10997().method_8320(targetPos.method_10084().method_10084()).method_26215();
	}

	/**
	 * @param targetPos the position of another elevator
	 */
	public boolean isValidTarget(final class_2338 targetPos) {
		return isRunning(targetPos) && isFree(targetPos);
	}

	public Optional<class_2338> nextUpElevator() {
		class_2338 upPos = method_11016().method_10084().method_10084();
		if (!TechRebornConfig.allowElevatingThroughBlocks && (!isAirOrElevator(method_11016().method_10084()) || !isAirOrElevator(method_11016().method_10084().method_10084()))) {
			return Optional.empty();
		}
		do {
			upPos = upPos.method_10084();
			if (!TechRebornConfig.allowElevatingThroughBlocks && !isAirOrElevator(upPos)) {
				return Optional.empty();
			}
		} while (upPos.method_10264() <= method_10997().method_31600() && !isValidTarget(upPos));
		if (upPos.method_10264() < method_10997().method_31600() || isValidTarget(upPos)) {
			return Optional.of(upPos);
		}
		return Optional.empty();
	}

	public Optional<class_2338> nextDownElevator() {
		class_2338 downPos = method_11016().method_10074().method_10074();
		if (!TechRebornConfig.allowElevatingThroughBlocks && (!isAirOrElevator(method_11016().method_10074()) || !isAirOrElevator(method_11016().method_10074().method_10074()))) {
			return Optional.empty();
		}
		do {
			downPos = downPos.method_10074();
			if (!TechRebornConfig.allowElevatingThroughBlocks && !isAirOrElevator(downPos)) {
				return Optional.empty();
			}
		} while (downPos.method_10264() >= method_10997().method_31607() && !isValidTarget(downPos));
		if (downPos.method_10264() > method_10997().method_31607() || isValidTarget(downPos)) {
			return Optional.of(downPos);
		}
		return Optional.empty();
	}

	/**
	 * @param targetPos the position of another elevator
	 */
	public int energyCost(final class_2338 targetPos) {
		return Math.max(Math.abs(targetPos.method_10264()-method_11016().method_10264())*TechRebornConfig.elevatorEnergyPerBlock,0);
	}

	/**
	 * @param targetPos the position <strong>over</strong> another elevator
	 */
	protected boolean teleport(final class_1657 player, final class_2338 targetPos) {
		if (!(method_10997() instanceof class_3218)) {
			return false;
		}
		final int energy = energyCost(targetPos);
		if (getStored() < energy) {
			return false;
		}
		playTeleportSoundAt(method_11016());
		player.method_5731(new class_5454(
			(class_3218)method_10997(),
			class_243.method_24955(new class_2382(targetPos.method_10263(), targetPos.method_10264(), targetPos.method_10260())),
			class_243.field_1353,
			player.method_36454(),
			player.method_36455(),
			class_5454.field_52245
		));

		useEnergy(energy);
		playTeleportSoundAt(targetPos);
		return true;
	}

	protected void playTeleportSoundAt(final class_2338 targetPos) {
		method_10997().method_8396(null, targetPos, class_3417.field_14879, class_3419.field_15245, 1f, 1f);
	}

	public void teleportUp(final class_1657 player) {
		if (!this.field_11867.method_19769(player.method_73189(), 5) && player.method_73183() == this.field_11863) {
			// Ensure the player is close to the elevator and in the same world.
			return;
		}

		Optional<class_2338> upTarget = nextUpElevator();
		if (upTarget.isEmpty()) {
			return;
		}
		if (teleport(player, upTarget.get().method_10084())) {
			player.method_6100(false);
		}
	}

	// PowerAcceptorBlockEntity
	@Override
	public void tick(class_1937 world, class_2338 pos, class_2680 state, MachineBaseBlockEntity blockEntity) {
		super.tick(world, pos, state, blockEntity);
		if (!(world instanceof class_3218) || getStored() <= 0 || !isActive(RedstoneConfiguration.Element.POWER_IO)) {
			return;
		}

		// teleporting up must be done via mixin for now
		Optional<class_2338> downTarget = null;

		List<class_1657> players = world.method_18467(class_1657.class, new class_238(0d,1d,0d,1d,2d,1d).method_996(pos));
		if (players.size() == 0) {
			return;
		}
		for (class_1657 player : players) {
			if (player.method_5715()) {
				if (downTarget == null) {
					downTarget = nextDownElevator();
				}
				if (downTarget.isEmpty()) {
					continue;
				}
				if (teleport(player, downTarget.get().method_10084())) {
					player.method_5660(false);
				}
			}
		}
	}
	@Override
	public long getBaseMaxPower() {
		return TechRebornConfig.elevatorMaxEnergy;
	}

	@Override
	public boolean canProvideEnergy(@Nullable class_2350 side) {
		return false;
	}

	@Override
	public long getBaseMaxOutput() {
		return 0;
	}

	@Override
	public long getBaseMaxInput() {
		return TechRebornConfig.elevatorMaxInput;
	}

	// MachineBaseBlockEntity
	@Override
	public boolean hasSlotConfig() {
		return false;
	}

	@Override
	public boolean canBeUpgraded() {
		return false;
	}

	// IToolDrop
	@Override
	public class_1799 getToolDrop(class_1657 p0) {
		return TRContent.Machine.ELEVATOR.getStack();
	}

	// BuiltScreenHandlerProvider
	@Override
	public BuiltScreenHandler createScreenHandler(int syncID, class_1657 player) {
		return new ScreenHandlerBuilder("elevator")
				.player(player.method_31548())
				.inventory().hotbar().addInventory()
				.blockEntity(this)
				.syncEnergyValue()
				.addInventory().create(this, syncID);
	}
}
