/*
 * This file is part of TechReborn, licensed under the MIT License (MIT).
 *
 * Copyright (c) 2020 TechReborn
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package techreborn.blockentity.machine.tier1;

import net.minecraft.class_1657;
import net.minecraft.class_1799;
import net.minecraft.class_1937;
import net.minecraft.class_2248;
import net.minecraft.class_2338;
import net.minecraft.class_2350;
import net.minecraft.class_2680;
import net.minecraft.class_3861;
import net.minecraft.class_3956;
import net.minecraft.class_8786;
import net.minecraft.class_9135;
import net.minecraft.class_9696;
import net.minecraft.server.MinecraftServer;
import org.jetbrains.annotations.Nullable;
import reborncore.api.IToolDrop;
import reborncore.api.blockentity.InventoryProvider;
import reborncore.common.blockentity.MachineBaseBlockEntity;
import reborncore.common.blocks.BlockMachineBase;
import reborncore.common.powerSystem.PowerAcceptorBlockEntity;
import reborncore.common.recipes.RecipeCrafter;
import reborncore.common.screen.BuiltScreenHandler;
import reborncore.common.screen.BuiltScreenHandlerProvider;
import reborncore.common.screen.builder.ScreenHandlerBuilder;
import reborncore.common.util.ItemUtils;
import reborncore.common.util.RebornInventory;
import techreborn.config.TechRebornConfig;
import techreborn.init.TRBlockEntities;
import techreborn.init.TRContent;

import java.util.Optional;

public class ElectricFurnaceBlockEntity extends PowerAcceptorBlockEntity
		implements IToolDrop, InventoryProvider, BuiltScreenHandlerProvider {

	public final RebornInventory<ElectricFurnaceBlockEntity> inventory = new RebornInventory<>(3, "ElectricFurnaceBlockEntity", 64, this);
	final int inputSlot = 0;
	final int outputSlot = 1;
	int ticksSinceLastChange;
	private class_3861 currentRecipe;
	private int cookTime;
	private int cookTimeTotal;
	// Energy cost per tick of cooking
	final int EnergyPerTick = 1;

	public ElectricFurnaceBlockEntity(class_2338 pos, class_2680 state) {
		super(TRBlockEntities.ELECTRIC_FURNACE, pos, state);
	}

	private void setInvDirty(boolean isDirty) {
		inventory.setHashChanged(isDirty);
	}

	private boolean isInvDirty() {
		return inventory.hasChanged();
	}

	private void updateCurrentRecipe() {
		class_1799 stack = inventory.method_5438(inputSlot);
		if (stack.method_7960()) {
			resetCrafter();
			return;
		}
		if (field_11863 == null) return;
		MinecraftServer server = field_11863.method_8503();
		if (server == null) return;
		Optional<class_3861> testRecipe = server.method_3772().method_8132(class_3956.field_17546, new class_9696(stack), field_11863).map(class_8786::comp_1933);
		if (!testRecipe.isPresent()) {
			resetCrafter();
			return;
		}
		if (!canAcceptOutput(testRecipe.get(), outputSlot)) {
			resetCrafter();
		}
		currentRecipe = testRecipe.get();
		cookTime = 0;
		cookTimeTotal = Math.max((int) (currentRecipe.method_8167() * (1.0 - getSpeedMultiplier())), 1);
		updateState();
	}

	private boolean canAcceptOutput(class_3861 recipe, int slot) {
		class_1799 recipeOutput = recipe.method_59998(new class_9696(inventory.method_5438(slot)), method_10997().method_30349());
		if (recipeOutput.method_7960()) {
			return false;
		}
		if (inventory.method_5438(slot).method_7960()) {
			return true;
		}
		if (ItemUtils.isItemEqual(inventory.method_5438(slot), recipeOutput, true, true)) {
			return recipeOutput.method_7947() + inventory.method_5438(slot).method_7947() <= recipeOutput.method_7914();
		}
		return false;
	}

	public boolean canCraftAgain() {
		if (inventory.method_5438(inputSlot).method_7960()) {
			return false;
		}
		if (currentRecipe == null) {
			return false;
		}
		if (!canAcceptOutput(currentRecipe, outputSlot)) {
			return false;
		}
		return !(getEnergy() < currentRecipe.method_8167() * getEuPerTick(EnergyPerTick));
	}

	private void resetCrafter() {
		currentRecipe = null;
		cookTime = 0;
		cookTimeTotal = 0;
		updateState();
	}

	private void updateState() {
		class_2248 furnaceBlock = method_10997().method_8320(field_11867).method_26204();

		if (furnaceBlock instanceof BlockMachineBase blockMachineBase) {
			boolean isActive = currentRecipe != null || canCraftAgain();
			blockMachineBase.setActive(isActive, field_11863, field_11867);
		}
		field_11863.method_8413(field_11867, field_11863.method_8320(field_11867), field_11863.method_8320(field_11867), 3);
	}

	private boolean hasAllInputs(class_3861 recipe) {
		if (recipe == null) {
			return false;
		}
		class_1799 stack = inventory.method_5438(inputSlot);
		if (stack.method_7960()) {
			return false;
		}

		return recipe.method_64719(new class_9696(stack), field_11863);
	}

	private void craftRecipe(class_3861 recipe) {
		if (recipe == null) {
			return;
		}
		if (!canAcceptOutput(recipe, outputSlot)) {
			return;
		}
		class_1799 outputStack = inventory.method_5438(outputSlot);
		class_1799 result = recipe.method_59998(new class_9696(inventory.method_5438(outputSlot)), method_10997().method_30349());
		if (outputStack.method_7960()) {
			inventory.method_5447(outputSlot, result.method_7972());
		} else {
			outputStack.method_7933(result.method_7947());
		}

		inventory.method_5438(inputSlot).method_7934(1);
	}

	public int getProgressScaled(int scale) {
		if (cookTimeTotal != 0) {
			return cookTime * scale / cookTimeTotal;
		}
		return 0;
	}

	public int getCookTime() {
		return cookTime;
	}

	public void setCookTime(int cookTime) {
		this.cookTime = cookTime;
	}

	public int getCookTimeTotal() {
		return cookTimeTotal;
	}

	public void setCookTimeTotal(int cookTimeTotal) {
		this.cookTimeTotal = cookTimeTotal;
	}

	// TilePowerAcceptor
	@Override
	public void tick(class_1937 world, class_2338 pos, class_2680 state, MachineBaseBlockEntity blockEntity) {
		super.tick(world, pos, state, blockEntity);
		charge(2);

		if (world == null || world.method_8608()) {
			return;
		}

		ticksSinceLastChange++;
		// Force a has chanced every second
		if (ticksSinceLastChange == 20) {
			setInvDirty(true);
			ticksSinceLastChange = 0;
		}

		if (isInvDirty()) {
			if (currentRecipe == null) {
				updateCurrentRecipe();
			}
			if (currentRecipe != null && (!hasAllInputs(currentRecipe) || !canAcceptOutput(currentRecipe, outputSlot))) {
				resetCrafter();
			}
		}

		if (currentRecipe != null) {
			if (cookTime >= cookTimeTotal && hasAllInputs(currentRecipe)) {
				craftRecipe(currentRecipe);
				updateCurrentRecipe();
			} else if (cookTime < cookTimeTotal) {
				if (getStored() > getEuPerTick(EnergyPerTick)) {
					useEnergy(getEuPerTick(EnergyPerTick));
					cookTime++;
					if (!isMuffled() && (cookTime == 1 || cookTime % 20 == 0 && RecipeCrafter.soundHandler != null)) {
						RecipeCrafter.soundHandler.playSound(false, this);
					}
				}
			}
		}
		setInvDirty(false);
	}

	@Override
	protected boolean canProvideEnergy(@Nullable class_2350 side) {
		return false;
	}

	@Override
	public long getBaseMaxPower() {
		return TechRebornConfig.electricFurnaceMaxEnergy;
	}

	@Override
	public long getBaseMaxOutput() {
		return 0;
	}

	@Override
	public long getBaseMaxInput() {
		return TechRebornConfig.electricFurnaceMaxInput;
	}

	// IToolDrop
	@Override
	public class_1799 getToolDrop(final class_1657 entityPlayer) {
		return TRContent.Machine.ELECTRIC_FURNACE.getStack();
	}

	// InventoryProvider
	@Override
	public RebornInventory<ElectricFurnaceBlockEntity> getInventory() {
		return inventory;
	}

	// BuiltScreenHandlerProvider
	@Override
	public BuiltScreenHandler createScreenHandler(int syncID, final class_1657 player) {
		return new ScreenHandlerBuilder("electricfurnace").player(player.method_31548()).inventory().hotbar().addInventory()
				.blockEntity(this).slot(0, 55, 45).outputSlot(1, 101, 45).energySlot(2, 8, 72).syncEnergyValue()
				.sync(class_9135.field_49675, this::getCookTime, this::setCookTime).sync(class_9135.field_49675, this::getCookTimeTotal, this::setCookTimeTotal).addInventory().create(this, syncID);
	}
}
