/*
 * This file is part of TechReborn, licensed under the MIT License (MIT).
 *
 * Copyright (c) 2020 TechReborn
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package techreborn.blockentity.machine.tier1;

import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import reborncore.api.IToolDrop;
import reborncore.api.blockentity.InventoryProvider;
import reborncore.common.blockentity.MachineBaseBlockEntity;
import reborncore.common.powerSystem.PowerAcceptorBlockEntity;
import reborncore.common.screen.BuiltScreenHandler;
import reborncore.common.screen.BuiltScreenHandlerProvider;
import reborncore.common.screen.builder.ScreenHandlerBuilder;
import reborncore.common.util.ItemUtils;
import reborncore.common.util.RebornInventory;
import techreborn.config.TechRebornConfig;
import techreborn.init.ModSounds;
import techreborn.init.TRBlockEntities;
import techreborn.init.TRContent;

import java.util.*;
import net.minecraft.class_11368;
import net.minecraft.class_11372;
import net.minecraft.class_1657;
import net.minecraft.class_1703;
import net.minecraft.class_1715;
import net.minecraft.class_1792;
import net.minecraft.class_1799;
import net.minecraft.class_1802;
import net.minecraft.class_1852;
import net.minecraft.class_1937;
import net.minecraft.class_2338;
import net.minecraft.class_2350;
import net.minecraft.class_2371;
import net.minecraft.class_2680;
import net.minecraft.class_3218;
import net.minecraft.class_3419;
import net.minecraft.class_3955;
import net.minecraft.class_3956;
import net.minecraft.class_8786;
import net.minecraft.class_9135;
import net.minecraft.class_9323;
import net.minecraft.class_9694;

/**
 * Created by modmuss50 on 20/06/2017.
 */
public class AutoCraftingTableBlockEntity extends PowerAcceptorBlockEntity
	implements IToolDrop, InventoryProvider, BuiltScreenHandlerProvider {

	public static final int CRAFTING_HEIGHT = 3;
	public static final int CRAFTING_WIDTH = 3;
	public static final int CRAFTING_AREA = CRAFTING_HEIGHT * CRAFTING_WIDTH;
	public static final int RECIPE_TIME = 120;
	public static final int EU_TICK = 10;

	public final RebornInventory<AutoCraftingTableBlockEntity> inventory;
	private final BalanceTable balanceTable = new BalanceTable();
	private final int OUTPUT_SLOT = CRAFTING_AREA; // first slot is indexed by 0, so this is the last non crafting slot
	private final int EXTRA_OUTPUT_SLOT = CRAFTING_AREA + 1;

	public int progress = 0;
	public int maxProgress = RECIPE_TIME;
	public long euTick = EU_TICK;
	public long lastSoundTime = 0;

	class_1715 inventoryCrafting;
	class_3955 lastRecipe = null;
	class_1799 outputPreview = class_1799.field_8037;

	public boolean locked = false;

	public AutoCraftingTableBlockEntity(class_2338 pos, class_2680 state) {
		super(TRBlockEntities.AUTO_CRAFTING_TABLE, pos, state);
		inventoryCrafting = new class_1715(new class_1703(null, -1) {
			@Override
			public class_1799 method_7601(class_1657 player, int index) {
				return class_1799.field_8037;
			}

			@Override
			public boolean method_7597(class_1657 playerIn) {
				return false;
			}
		}, CRAFTING_WIDTH, CRAFTING_HEIGHT);
		inventory = new RebornInventory<>(CRAFTING_AREA + 2, "AutoCraftingTableBlockEntity", 64, this) {
			private void syncStack(int slot, class_1799 stack) {
				if (slot < CRAFTING_AREA) {
					inventoryCrafting.method_5447(slot, stack);
				}
			}

			@Override
			public void readData(class_11368 view) {
				super.readData(view);
				for (int i = 0; i < CRAFTING_AREA; i++) {
					inventoryCrafting.method_5447(i, inventory.method_5438(i));
				}
			}

			@Override
			public void method_5447(int slot, @NotNull class_1799 stack) {
				super.method_5447(slot, stack);
				syncStack(slot, stack);
			}

			@Override
			public class_1799 method_5441(int i) {
				syncStack(i, class_1799.field_8037);
				return super.method_5441(i);
			}

			@Override
			public class_1799 method_5434(int i, int i1) {
				class_1799 stack = super.method_5434(i, i1);
				if (this.method_5438(i).method_7960()) {
					syncStack(i, class_1799.field_8037);
				}
				return stack;
			}

			@Override
			public class_1799 shrinkSlot(int slot, int count) {
				class_1799 stack = super.shrinkSlot(slot, count);
				if (this.method_5438(slot).method_7960()) {
					syncStack(slot, class_1799.field_8037);
				}
				return stack;
			}
		};
	}

	public boolean updateCurrentRecipe(class_3218 world, class_9694 input) {
		if (lastRecipe != null && lastRecipe.method_8115(input, world)) {
			if (outputPreview == class_1799.field_8037) {
				balanceTable.updateLayout(input);
				outputPreview = lastRecipe.method_8116(input, world.method_30349());
			} else if (lastRecipe instanceof class_1852 && balanceTable.updateLayout(input)) {
				outputPreview = lastRecipe.method_8116(input, world.method_30349());
			}
			return true;
		}

		if (balanceTable.updateLayout(input)) {
			Optional<class_3955> testRecipe = world.method_64577()
				.method_8132(class_3956.field_17545, input, world).map(class_8786::comp_1933);
			if (testRecipe.isPresent()) {
				lastRecipe = testRecipe.get();
				outputPreview = lastRecipe.method_8116(input, world.method_30349());
				return true;
			} else {
				outputPreview = class_1799.field_8037;
			}
		}
		return false;
	}

	@Nullable
	private class_1799 getRecipeReminder(class_9694 input) {
		class_2371<class_1799> remainingStacks = lastRecipe.method_17704(input);
		class_1799 reminderStack, recipeReminder = class_1799.field_8037;
		for (int slot = 0, size = remainingStacks.size(); slot < size; slot++) {
			reminderStack = remainingStacks.get(slot);
			if (!reminderStack.method_7960()) {
				recipeReminder = reminderStack.method_7972();
				for (slot = slot + 1; slot < size; slot++) {
					reminderStack = remainingStacks.get(slot);
					if (!reminderStack.method_7960()) {
						if (class_1799.method_31577(recipeReminder, reminderStack)) {
							recipeReminder.method_7933(reminderStack.method_7947());
						} else {
							return null;
						}
					}
				}
				break;
			}
		}
		return recipeReminder;
	}

	private void make(class_9694.class_9765 positioned, class_1799 resultStack, class_1799 remainderStack) {
		class_9694 input = positioned.comp_2795();
		int width = input.method_59991();
		int max = (positioned.comp_2797() + input.method_59992()) * CRAFTING_WIDTH;
		int space = CRAFTING_WIDTH - width;
		for (int slot = positioned.comp_2797() * CRAFTING_WIDTH + positioned.comp_2796(); slot < max; slot += space) {
			for (int end = slot + width; slot < end; slot++) {
				if (!this.inventory.method_5438(slot).method_7960()) {
					this.inventory.shrinkSlot(slot, 1);
				}
			}
		}
		moveOutput(resultStack, OUTPUT_SLOT);
		if (!remainderStack.method_7960()) {
			moveOutput(remainderStack, EXTRA_OUTPUT_SLOT);
		}
		inventory.resetHasChanged();
	}

	private boolean hasOutputSpace(class_1799 output, int slot) {
		class_1799 stack = inventory.method_5438(slot);
		if (stack.method_7960()) {
			return true;
		}
		if (ItemUtils.isItemEqual(stack, output, true, true)) {
			return stack.method_7914() >= stack.method_7947() + output.method_7947();
		}
		return false;
	}

	private void moveOutput(class_1799 stack, int slot) {
		class_1799 currentOutput = inventory.method_5438(slot);
		if (currentOutput.method_7960()) {
			inventory.method_5447(slot, stack.method_7972());
		} else {
			currentOutput.method_7933(stack.method_7947());
		}
	}

	// PowerAcceptorBlockEntity
	@Override
	public void tick(class_1937 world, class_2338 pos, class_2680 state, MachineBaseBlockEntity blockEntity) {
		super.tick(world, pos, state, blockEntity);
		if (world == null || world.method_8608() || getStored() < euTick) {
			return;
		}
		if (inventoryCrafting.method_5442()) {
			progress = 0;
			outputPreview = class_1799.field_8037;
			return;
		}
		class_9694.class_9765 positioned = inventoryCrafting.method_60501();
		class_9694 input = positioned.comp_2795();
		if (!updateCurrentRecipe((class_3218) world, input)) {
			progress = 0;
			return;
		}
		if (!hasOutputSpace(outputPreview, OUTPUT_SLOT)) {
			return;
		}

		balanceTable.balance(inventory, input);

		// Don't allow recipe to change (Keep at least one of each slot stocked, assuming it's actually a recipe)
		if (locked) {
			for (class_1799 stack : input.method_59989()) {
				if (stack.method_7947() == 1) {
					return;
				}
			}
		}

		class_1799 recipeReminder = getRecipeReminder(input);
		if (recipeReminder == null || !hasOutputSpace(recipeReminder, EXTRA_OUTPUT_SLOT)) {
			return;
		}

		if (progress >= maxProgress) {
			progress = 0;
			make(positioned, outputPreview, recipeReminder);
			if (inventoryCrafting.method_5442()) {
				outputPreview = class_1799.field_8037;
			}
		} else {
			if (progress == 0) {
				maxProgress = Math.max((int) (RECIPE_TIME * (1.0 - getSpeedMultiplier())), 1);
				euTick = getEuPerTick(EU_TICK);
				if (getStored() < euTick) {
					return;
				}
			}
			progress++;
			if (!isMuffled()) {
				long time = world.method_8510();
				if (time - lastSoundTime > RECIPE_TIME) {
					lastSoundTime = time;
					world.method_43128(null, pos.method_10263(), pos.method_10264(), pos.method_10260(), ModSounds.AUTO_CRAFTING,
						class_3419.field_15245, 0.3F, 0.8F);
				}
			}
			useEnergy(euTick);
		}
	}

	@Override
	public long getBaseMaxPower() {
		return TechRebornConfig.autoCraftingTableMaxEnergy;
	}

	@Override
	public long getBaseMaxOutput() {
		return 0;
	}

	@Override
	public long getBaseMaxInput() {
		return TechRebornConfig.autoCraftingTableMaxInput;
	}

	@Override
	public boolean canProvideEnergy(@Nullable class_2350 side) {
		return false;
	}

	@Override
	public void method_11007(class_11372 view) {
		view.method_71472("locked", locked);
		super.method_11007(view);
	}

	@Override
	public void method_11014(class_11368 view) {
		locked = view.method_71433("locked", false);
		super.method_11014(view);
	}

	// IToolDrop
	@Override
	public class_1799 getToolDrop(class_1657 playerIn) {
		return TRContent.Machine.AUTO_CRAFTING_TABLE.getStack();
	}

	// InventoryProvider
	@Override
	public RebornInventory<AutoCraftingTableBlockEntity> getInventory() {
		return inventory;
	}

	// BuiltScreenHandlerProvider
	@Override
	public BuiltScreenHandler createScreenHandler(int syncID, class_1657 player) {
		return new ScreenHandlerBuilder("autocraftingtable").player(player.method_31548()).inventory().hotbar().addInventory()
			.blockEntity(this)
			.slot(0, 28, 25).slot(1, 46, 25).slot(2, 64, 25)
			.slot(3, 28, 43).slot(4, 46, 43).slot(5, 64, 43)
			.slot(6, 28, 61).slot(7, 46, 61).slot(8, 64, 61)
			.outputSlot(OUTPUT_SLOT, 145, 42)
			.outputSlot(EXTRA_OUTPUT_SLOT, 145, 70)
			.syncEnergyValue().sync(class_9135.field_49675, this::getProgress, this::setProgress)
			.sync(class_9135.field_49675, this::getMaxProgress, this::setMaxProgress)
			.sync(class_9135.field_49675, this::getLockedInt, this::setLockedInt)
			.sync(class_1799.field_49268, this::getOutputPreview, this::setOutputPreview)
			.addInventory().create(this, syncID);
	}

	public int getProgress() {
		return progress;
	}

	public void setProgress(int progress) {
		this.progress = progress;
	}

	public int getMaxProgress() {
		if (maxProgress == 0) {
			maxProgress = 1;
		}
		return maxProgress;
	}

	public void setMaxProgress(int maxProgress) {
		this.maxProgress = maxProgress;
	}

	public int getLockedInt() {
		return locked ? 1 : 0;
	}

	public void setLockedInt(int lockedInt) {
		locked = lockedInt == 1;
	}

	public class_1799 getOutputPreview() {
		return outputPreview;
	}

	public void setOutputPreview(class_1799 stack) {
		outputPreview = stack;
	}

	static class BalanceTable {
		private class_1792[] layout = new class_1792[0];
		private BalanceEntry entry = new BalanceEntry();
		private final Map<BalanceEntry, ArrayList<Integer>> ingredients = new HashMap<>();
		private java.util.Iterator<BalanceEntry> iterator = null;
		private boolean empty = true;
		private boolean needsMatch = false;

		public boolean updateLayout(class_9694 input) {
			int size = input.method_59983();
			class_1799[] stacks = new class_1799[size];
			class_1792[] items = new class_1792[size];

			boolean same = size == layout.length;
			for (int i = 0; i < size; i++) {
				stacks[i] = input.method_59984(i);
				items[i] = stacks[i].method_7909();
				if (same && layout[i] != items[i]) {
					same = false;
				}
			}

			if (same) {
				return false;
			}

			layout = items;
			ingredients.clear();
			entry.resetMatch();
			for (int i = 0; i < size; i++) {
				if (items[i] != class_1802.field_8162) {
					entry.setItem(items[i]);
					entry.setComponents(stacks[i].method_57353());
					ingredients.computeIfAbsent(entry, this::createSlots).add(i);
				}
			}
			ingredients.values().removeIf(this::checkInvalid);
			empty = ingredients.isEmpty();
			if (!empty) {
				iterator = ingredients.keySet().stream().toList().iterator();
				entry = iterator.next();
				needsMatch = true;
			}

			return true;
		}

		private ArrayList<Integer> createSlots(BalanceEntry _entry) {
			entry = new BalanceEntry();
			return new ArrayList<>(layout.length);
		}

		private boolean checkInvalid(ArrayList<Integer> list) {
			return list.size() == 1;
		}

		public void balance(RebornInventory<AutoCraftingTableBlockEntity> inventory, class_9694 input) {
			if (empty) return;
			if (!needsMatch) {
				if (!inventory.hasChanged()) return;
				inventory.resetHasChanged();
				needsMatch = true;
			}
			List<Integer> list = ingredients.get(entry);
			int min = Integer.MAX_VALUE, max = 0, count;
			class_1799 minStack = null, maxStack = null, itemStack;
			for (Integer slot : list) {
				itemStack = input.method_59984(slot);
				count = itemStack.method_7947();
				if (min > count) {
					min = count;
					minStack = itemStack;
				}
				if (max < count) {
					max = count;
					maxStack = itemStack;
				}
			}
			if (max > min + 1) {
				assert minStack != null && maxStack != null;
				maxStack.method_7934(1);
				minStack.method_7933(1);
				inventory.resetHasChanged();
				inventory.method_5431();
			} else {
				entry.stopMatch();
			}

			checkState(entryNext());
		}

		private boolean entryNext() {
			if (!iterator.hasNext()) {
				iterator = ingredients.keySet().stream().toList().iterator();
			}
			entry = iterator.next();
			return entry.needsMatch;
		}

		private void checkState(boolean needsMatch) {
			if (!needsMatch) {
				for (int i = ingredients.size() - 1; i > 0; i--) {
					if (entryNext()) return;
				}
				this.needsMatch = false;
				ingredients.keySet().forEach(BalanceEntry::resetMatch);
			}
		}
	}
	static class BalanceEntry {
		public class_1792 item = null;
		public class_9323 components = null;
		public boolean needsMatch = true;

		public void setItem(class_1792 item) {
			this.item = item;
		}

		public void setComponents(class_9323 components) {
			this.components = components;
		}

		public void stopMatch() {
			this.needsMatch = false;
		}

		public void resetMatch() {
			this.needsMatch = true;
		}

		@Override
		public boolean equals(Object o) {
			if (o instanceof BalanceEntry entry) {
				return item == entry.item && Objects.equals(components, entry.components);
			} else if (o instanceof class_1799 stack) {
				return item == stack.method_7909() && Objects.equals(components, stack.method_57353());
			} else {
				return false;
			}
		}

		@Override
		public int hashCode() {
			return Objects.hash(item, components);
		}
	}
}
