/*
 * This file is part of TechReborn, licensed under the MIT License (MIT).
 *
 * Copyright (c) 2022 TechReborn
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package techreborn.blockentity.machine.tier0.block.blockplacer;

import net.minecraft.class_1792;
import net.minecraft.class_1799;
import net.minecraft.class_1802;
import net.minecraft.class_1937;
import net.minecraft.class_2248;
import net.minecraft.class_2338;
import net.minecraft.class_2680;
import reborncore.common.blockentity.RedstoneConfiguration;
import techreborn.blockentity.machine.tier0.block.BlockProcessable;
import techreborn.blockentity.machine.tier0.block.BlockProcessor;
import techreborn.blockentity.machine.tier0.block.BlockProcessorUtils;
import techreborn.blockentity.machine.tier0.block.ProcessingStatus;

/**
 * <b>Class handling the process of placing a block</b>
 * <br>
 * The main purpose of this class is to implement the {@link #onTick(class_1937, class_2338)}.
 * This method defines the logic for placing a block
 *
 * @author SimonFlapse
 * @see techreborn.blockentity.machine.tier0.block.BlockPlacerBlockEntity
 */
public class BlockPlacerProcessor extends BlockPlacerNbt implements BlockProcessor {

	private final BlockProcessable processable;

	private final int inputSlot;
	private final int fakeOutputSlot;

	private final int basePlaceTime;
	private final int baseCostToPlace;

	public BlockPlacerProcessor(BlockProcessable processable, int inputSlot, int fakeOutputSlot, int basePlaceTime, int baseCostToPlace) {
		this.processable = processable;

		this.inputSlot = inputSlot;
		this.fakeOutputSlot = fakeOutputSlot;

		this.basePlaceTime = basePlaceTime;
		this.baseCostToPlace = baseCostToPlace;
	}

	public ProcessingStatus getStatusEnum() {
		return status;
	}

	@Override
	public int getCurrentTickTime() {
		return getCurrentPlaceTime();
	}

	@Override
	public int getTickTime() {
		return getPlaceTime();
	}

	public ProcessingStatus onTick(class_1937 world, class_2338 positionInFront) {
		if (!ensureRedstoneEnabled()) return status;

		if (!handleInterrupted()) return status;

		class_1799 inputItemStack = processable.getInventory().method_5438(inputSlot);

		if (!handleInputSlotEmptied(inputItemStack)) return status;

		class_2680 blockInFront = world.method_8320(positionInFront);

		if (!ensureEnoughItemsToPlace(inputItemStack)) return status;

		if (!ensureBlockCanBePlaced(blockInFront, inputItemStack, 1)) return status;

		class_1792 currentPlacingItem = processable.getInventory().method_5438(fakeOutputSlot).method_7909();

		class_1799 fakeItem = inputItemStack.method_7972();

		if (fakeItem.method_31574(class_1802.field_8162)) {
			currentPlacingItem = null;
		}

		processable.getInventory().method_5447(fakeOutputSlot, fakeItem);

		class_2680 resultingBlockState = class_2248.method_9503(inputItemStack.method_7909()).method_9564();

		float hardness = BlockProcessorUtils.getHardness(world, resultingBlockState, positionInFront);

		this.placeTime = BlockProcessorUtils.getProcessTimeWithHardness(processable, basePlaceTime, hardness);

		if (!ensureItemNotReplaced(currentPlacingItem, inputItemStack)) return status;

		if (!increasePlaceTime()) return status;

		BlockProcessorUtils.playSound(processable, currentPlaceTime);

		placeBlock(world, positionInFront, inputItemStack);

		status = BlockPlacerStatus.PROCESSING;

		return status;
	}

	private boolean ensureRedstoneEnabled() {
		if (!processable.isActive(RedstoneConfiguration.Element.RECIPE_PROCESSING)) {
			return breakControlFlow(BlockPlacerStatus.IDLE_PAUSED);
		}

		return true;
	}

	private boolean handleInterrupted() {
		//Persists the last status message until the currentPlaceTime is back to 0
		//Set the currentPlaceTime to less than 0 for as many ticks as you want a message to persist.
		//The machine processing is halted while persisting messages.
		if (currentPlaceTime < 0) {
			this.currentPlaceTime++;
			return false;
		}

		return true;
	}

	private boolean handleInputSlotEmptied(class_1799 inputStack) {
		//Makes sure that if the input slot is emptied, the processing resets
		if (inputStack.method_31574(class_1799.field_8037.method_7909())) {
			processable.getInventory().method_5447(fakeOutputSlot, class_1799.field_8037);
			resetProcessing(0);
		}

		return true;
	}

	private boolean ensureEnoughItemsToPlace(class_1799 currentStack) {
		//Ensure that we have enough items in the input to place
		if (currentStack.method_7947() <= 0) {
			return breakControlFlow(BlockPlacerStatus.IDLE);
		}

		return true;
	}

	private boolean ensureBlockCanBePlaced(class_2680 blockInFront, class_1799 fakeItem, float hardness) {
		//Checks if the block can be placed
		//Items for which the default BlockState is an air block should not be placed, that will just destroy the item
		//Blocks with hardness below 0 can not be broken, and thus should not be placed
		class_2680 blockState = class_2248.method_9503(fakeItem.method_7909()).method_9564();
		if (blockState.method_26215() || hardness < 0) {
			return breakControlFlow(BlockPlacerStatus.IDLE);
		}

		//Checks if output is blocked
		if (!blockInFront.method_26215()) {
			return breakControlFlow(BlockPlacerStatus.OUTPUT_BLOCKED);
		}

		return true;
	}

	private boolean ensureItemNotReplaced(class_1792 currentPlacingItem, class_1799 item) {
		//Ensures that a smart replace of item, will cause the processing to restart
		if (currentPlacingItem != null && !class_1799.field_8037.method_31574(currentPlacingItem) && !item.method_31574(currentPlacingItem)) {
			return breakControlFlow(BlockPlacerStatus.INTERRUPTED);
		}

		return true;
	}

	private boolean increasePlaceTime() {
		if (!processable.consumeEnergy(baseCostToPlace)) {
			return breakControlFlow(BlockPlacerStatus.NO_ENERGY);
		}
		this.currentPlaceTime++;

		return true;
	}

	private void placeBlock(class_1937 world, class_2338 positionInFront, class_1799 currentStack) {
		if (currentPlaceTime >= placeTime) {
			class_1799 itemStack = processable.getInventory().method_5438(inputSlot);

			world.method_8501(positionInFront, class_2248.method_9503(currentStack.method_7909()).method_9564());
			itemStack.method_7939(itemStack.method_7947() - 1);

			resetProcessing(0);
		}
	}

	private void resetProcessing(int tick) {
		this.currentPlaceTime = tick;
		this.placeTime = basePlaceTime;
		this.processable.getInventory().method_5447(fakeOutputSlot, class_1799.field_8037);
	}

	private boolean breakControlFlow(ProcessingStatus status) {
		resetProcessing(-20);
		this.status = status;
		return false;
	}
}
