/*
 * This file is part of TechReborn, licensed under the MIT License (MIT).
 *
 * Copyright (c) 2022 TechReborn
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package techreborn.blockentity.machine.tier0.block.blockbreaker;

import reborncore.common.blockentity.RedstoneConfiguration;
import techreborn.blockentity.machine.tier0.block.BlockProcessable;
import techreborn.blockentity.machine.tier0.block.BlockProcessor;
import techreborn.blockentity.machine.tier0.block.BlockProcessorUtils;
import techreborn.blockentity.machine.tier0.block.ProcessingStatus;
import techreborn.init.TRContent;

import java.util.Collections;
import java.util.List;
import java.util.UUID;
import net.minecraft.class_1792;
import net.minecraft.class_1799;
import net.minecraft.class_1802;
import net.minecraft.class_181;
import net.minecraft.class_1937;
import net.minecraft.class_2246;
import net.minecraft.class_2338;
import net.minecraft.class_243;
import net.minecraft.class_2680;
import net.minecraft.class_3218;
import net.minecraft.class_3481;
import net.minecraft.class_8567;

/**
 * <b>Class handling the process of breaking a block</b>
 * <br>
 * The main purpose of this class is to implement the {@link #onTick(class_1937, class_2338)}.
 * This method defines the logic for breaking a block
 *
 * @author SimonFlapse
 * @see techreborn.blockentity.machine.tier0.block.BlockBreakerBlockEntity
 */
public class BlockBreakerProcessor extends BlockBreakerNbt implements BlockProcessor {

	private final UUID processorId = UUID.randomUUID();
	private final BlockProcessable processable;

	private final int outputSlot;
	private final int fakeInputSlot;

	private final int baseBreakTime;
	private final int baseCostToBreak;

	public BlockBreakerProcessor(BlockProcessable processable, int outputSlot, int fakeInputSlot, int baseBreakTime, int baseCostToBreak) {
		this.processable = processable;

		this.outputSlot = outputSlot;
		this.fakeInputSlot = fakeInputSlot;

		this.baseBreakTime = baseBreakTime;
		this.baseCostToBreak = baseCostToBreak;
	}

	@Override
	public ProcessingStatus getStatusEnum() {
		return status;
	}

	public ProcessingStatus onTick(class_1937 world, class_2338 positionInFront) {
		handleBlockBreakingProgressReset(world, positionInFront);

		if (!ensureRedstoneEnabled()) return status;

		if (!handleInterrupted()) return status;

		class_1799 outputItemStack = processable.getInventory().method_5438(outputSlot);

		class_2680 blockInFront = world.method_8320(positionInFront);

		if (!handleBlockInFrontRemoved(blockInFront)) return status;

		class_1792 currentBreakingItem = processable.getInventory().method_5438(fakeInputSlot).method_7909();
		class_1799 item = blockInFront.method_26204().method_8389().method_7854();
		final List<class_1799> blockDrops;

		if (world instanceof class_3218 serverWorld) {
			class_8567.class_8568 builder = new class_8567.class_8568(serverWorld)
				.method_51874(class_181.field_24424, class_243.method_24953(positionInFront))
				.method_51874(class_181.field_1229, TRContent.Machine.BLOCK_BREAKER.getStack());
			blockDrops = blockInFront.method_26189(builder);
		} else {
			blockDrops = Collections.singletonList(item);
		}

		class_1799 blockDrop = blockDrops.isEmpty() ? null : blockDrops.getFirst();
		if (blockDrop != null) {
			blockDrop.method_7939(1);
		}

		class_1799 fakeItem = item.method_7972();

		if (fakeItem.method_31574(class_1802.field_8162)) {
			currentBreakingItem = null;
		}

		processable.getInventory().method_5447(fakeInputSlot, fakeItem);

		float hardness = BlockProcessorUtils.getHardness(world, blockInFront, positionInFront);

		if (!ensureBlockCanBeBroken(blockInFront, fakeItem, hardness)) return status;

		this.breakTime = BlockProcessorUtils.getProcessTimeWithHardness(processable, baseBreakTime, hardness);

		if (!ensureBlockNotReplaced(currentBreakingItem, item)) return status;

		if (!ensureBlockFitInOutput(outputItemStack, blockDrop)) return status;

		if (!increaseBreakTime(world, positionInFront)) return status;

		BlockProcessorUtils.playSound(processable, currentBreakTime);

		breakBlock(world, positionInFront, outputItemStack, blockDrop);

		status = BlockBreakerStatus.PROCESSING;

		return status;
	}

	private boolean ensureRedstoneEnabled() {
		if (!processable.isActive(RedstoneConfiguration.Element.RECIPE_PROCESSING)) {
			return breakControlFlow(BlockBreakerStatus.IDLE_PAUSED);
		}

		return true;
	}

	private void handleBlockBreakingProgressReset(class_1937 world, class_2338 pos) {
		//Resets the BlockBreakingProgress, otherwise the progress will be buggy when a new block has been placed
		if (currentBreakTime == 0) {
			setBlockBreakingProgress(world, pos, -1);
		}
	}

	private boolean handleInterrupted() {
		//Persists the last status message until the currentBreakTime is back to 0
		//Set the currentBreakTime to less than 0 for as many ticks as you want a message to persist.
		//The machine processing is halted while persisting messages.
		if (currentBreakTime < 0) {
			this.currentBreakTime++;
			return false;
		}
		return true;
	}

	private boolean handleBlockInFrontRemoved(class_2680 blockInFront) {
		//Makes sure that if the block in front is removed, the processing resets
		if (blockInFront.method_27852(class_2246.field_10124)) {
			processable.getInventory().method_5447(fakeInputSlot, class_1799.field_8037);
			resetProcessing(0);
		}

		return true;
	}

	private boolean ensureBlockCanBeBroken(class_2680 blockInFront, class_1799 fakeItem, float hardness) {
		//Resets time if there is no block
		//If breaking the block returns no output, skip breaking it
		//Blocks with a hardness below 0 are unbreakable
		//shulker boxes don't drop their content when broken, so ignore them for now
		if (blockInFront.method_26215() || fakeItem.method_7960() || hardness < 0 || blockInFront.method_26164(class_3481.field_21490)) {
			return breakControlFlow(BlockBreakerStatus.IDLE);
		}

		return true;
	}

	private boolean ensureBlockNotReplaced(class_1792 currentBreakingItem, class_1799 item) {
		//Ensures that a piston cannot be abused to push in another block without resetting the progress
		if (currentBreakingItem != null && !class_1799.field_8037.method_31574(currentBreakingItem) && !item.method_31574(currentBreakingItem)) {
			return breakControlFlow(BlockBreakerStatus.INTERRUPTED);
		}

		return true;
	}

	private boolean ensureBlockFitInOutput(class_1799 currentStack, class_1799 blockDrop) {
		if (blockDrop == null || blockDrop.method_7947() == 0) {
			return true;
		}

		//Ensures that the block is the same as the one currently in the output slot
		if (!currentStack.method_31574(class_1799.field_8037.method_7909()) && !currentStack.method_31574(blockDrop.method_7909())) {
			return breakControlFlow(BlockBreakerStatus.OUTPUT_BLOCKED);
		}

		//Ensure that output slot can fit the block
		if (currentStack.method_7914() < currentStack.method_7947() + blockDrop.method_7947()) {
			return breakControlFlow(BlockBreakerStatus.OUTPUT_FULL);
		}

		return true;
	}

	private boolean increaseBreakTime(class_1937 world, class_2338 blockPos) {
		//if (!tryUseExact(getEuPerTick(baseCostToBreak))) {
		if (!processable.consumeEnergy(baseCostToBreak)) {
			return breakControlFlow(BlockBreakerStatus.NO_ENERGY);
		}

		setBlockBreakingProgress(world, blockPos);
		this.currentBreakTime++;
		return true;
	}

	private void breakBlock(class_1937 world, class_2338 positionInFront, class_1799 currentStack, class_1799 blockDrop) {
		if (currentBreakTime >= breakTime) {

			world.method_22352(positionInFront, false);

			resetProcessing(0);

			if (blockDrop == null || blockDrop.method_7947() == 0) {
				return;
			}
			if (currentStack.method_31574(class_1799.field_8037.method_7909())) {
				processable.getInventory().method_5447(outputSlot, blockDrop);
			} else {
				int currentCount = currentStack.method_7947();
				currentStack.method_7939(currentCount + blockDrop.method_7947());
			}
		}
	}

	private void resetProcessing(int tick) {
		this.currentBreakTime = tick;
		breakTime = baseBreakTime;
	}

	private void setBlockBreakingProgress(class_1937 world, class_2338 blockPos) {
		setBlockBreakingProgress(world, blockPos, getProgress() / 10);
	}

	private void setBlockBreakingProgress(class_1937 world, class_2338 blockPos, int breakingProgress) {
		world.method_8517(processorId.hashCode(), blockPos, breakingProgress);
	}

	@Override
	public int getCurrentTickTime() {
		return this.getCurrentBreakTime();
	}

	@Override
	public int getTickTime() {
		return this.getBreakTime();
	}

	private boolean breakControlFlow(ProcessingStatus status) {
		resetProcessing(-20);
		this.status = status;
		return false;
	}
}
