/*
 * This file is part of TechReborn, licensed under the MIT License (MIT).
 *
 * Copyright (c) 2020 TechReborn
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package techreborn.blockentity.machine.multiblock;

import reborncore.common.blockentity.MultiblockWriter;
import reborncore.common.multiblock.IMultiblockPart;
import reborncore.common.recipes.RecipeCrafter;
import reborncore.common.screen.BuiltScreenHandler;
import reborncore.common.screen.BuiltScreenHandlerProvider;
import reborncore.common.screen.builder.ScreenHandlerBuilder;
import reborncore.common.util.RebornInventory;
import techreborn.blockentity.machine.GenericMachineBlockEntity;
import techreborn.blockentity.machine.multiblock.casing.MachineCasingBlockEntity;
import techreborn.blocks.misc.BlockMachineCasing;
import techreborn.config.TechRebornConfig;
import techreborn.init.ModRecipes;
import techreborn.init.TRBlockEntities;
import techreborn.init.TRContent;
import techreborn.multiblocks.MultiBlockCasing;

import java.util.function.BiPredicate;
import net.minecraft.class_1657;
import net.minecraft.class_1922;
import net.minecraft.class_2246;
import net.minecraft.class_2248;
import net.minecraft.class_2338;
import net.minecraft.class_2350;
import net.minecraft.class_2586;
import net.minecraft.class_2680;
import net.minecraft.class_9135;

public class IndustrialBlastFurnaceBlockEntity extends GenericMachineBlockEntity implements BuiltScreenHandlerProvider {

	private int cachedHeat;

	public IndustrialBlastFurnaceBlockEntity(class_2338 pos, class_2680 state) {
		super(TRBlockEntities.INDUSTRIAL_BLAST_FURNACE, pos, state, "IndustrialBlastFurnace", TechRebornConfig.industrialBlastFurnaceMaxInput, TechRebornConfig.industrialBlastFurnaceMaxEnergy, TRContent.Machine.INDUSTRIAL_BLAST_FURNACE.block, 4);
		final int[] inputs = new int[]{0, 1};
		final int[] outputs = new int[]{2, 3};
		this.inventory = new RebornInventory<>(5, "IndustrialBlastFurnaceBlockEntity", 64, this);
		this.crafter = new RecipeCrafter(ModRecipes.BLAST_FURNACE, this, 2, 2, this.inventory, inputs, outputs);
	}

	@Override
	public boolean hasMultiblock() {
		return true;
	}

	@Override
	public void writeMultiblock(MultiblockWriter writer) {
		class_2248 basic = TRContent.MachineBlocks.BASIC.getCasing();
		class_2248 advanced = TRContent.MachineBlocks.ADVANCED.getCasing();
		class_2248 industrial = TRContent.MachineBlocks.INDUSTRIAL.getCasing();
		class_2680 lava = class_2246.field_10164.method_9564();

		BiPredicate<class_1922, class_2338> casing = (view, pos) -> {
			class_2248 block = view.method_8320(pos).method_26204();
			return basic == block || advanced == block || industrial == block;
		};

		BiPredicate<class_1922, class_2338> maybeLava = (view, pos) -> {
			class_2680 state = view.method_8320(pos);
			return state == lava || state.method_26204() == class_2246.field_10124;
		};

		class_2680 state = basic.method_9564();
		writer.translate(1, 0, -1)
				.fill(0, 0, 0, 3, 1, 3, casing, state)
				.ring(class_2350.class_2351.field_11052, 3, 1, 3, casing, state, maybeLava, lava)
				.ring(class_2350.class_2351.field_11052, 3, 2, 3, casing, state, maybeLava, lava)
				.fill(0, 3, 0, 3, 4, 3, casing, state);
	}

	public int getHeat() {
		if (!isShapeValid()) {
			return 0;
		}

		// Bottom center of multiblock
		final class_2338 location = field_11867.method_10079(getFacing().method_10153(), 2);
		final class_2586 blockEntity = field_11863.method_8321(location);

		if (blockEntity instanceof MachineCasingBlockEntity) {
			if (((MachineCasingBlockEntity) blockEntity).isConnected()
					&& ((MachineCasingBlockEntity) blockEntity).getMultiblockController().isAssembled()) {
				final MultiBlockCasing casing = ((MachineCasingBlockEntity) blockEntity).getMultiblockController();

				int heat = 0;

				// Bottom center shouldn't have any blockEntity entities below it
				if (field_11863.method_8320(new class_2338(location.method_10263(), location.method_10264() - 1, location.method_10260()))
						.method_26204() == blockEntity.method_10997().method_8320(blockEntity.method_11016()).method_26204()) {
					return 0;
				}

				for (final IMultiblockPart part : casing.connectedParts) {
					heat += BlockMachineCasing.getHeatFromState(part.method_11010());
				}

				if (field_11863.method_8320(location.method_10079(class_2350.field_11036, 1)).method_26204() == class_2246.field_10164
						&& field_11863.method_8320(location.method_10079(class_2350.field_11036, 2)).method_26204() == class_2246.field_10164) {
					heat += 500;
				}
				return heat;
			}
		}

		return 0;
	}

	public void setHeat(final int heat) {
		cachedHeat = heat;
	}

	public int getCachedHeat() {
		return cachedHeat;
	}

	// IContainerProvider
	@Override
	public BuiltScreenHandler createScreenHandler(int syncID, final class_1657 player) {
		return new ScreenHandlerBuilder("blastfurnace").player(player.method_31548()).inventory().hotbar().addInventory()
				.blockEntity(this).slot(0, 50, 27).slot(1, 50, 47).outputSlot(2, 93, 37).outputSlot(3, 113, 37)
				.energySlot(4, 8, 72).syncEnergyValue().syncCrafterValue()
				.sync(class_9135.field_49675, this::getHeat, this::setHeat)
				.syncShapeValue().addInventory().create(this, syncID);
	}

}
