/*
 * This file is part of TechReborn, licensed under the MIT License (MIT).
 *
 * Copyright (c) 2020 TechReborn
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package techreborn.blockentity.machine.multiblock;

import net.minecraft.class_11368;
import net.minecraft.class_11372;
import net.minecraft.class_1657;
import net.minecraft.class_1799;
import net.minecraft.class_1937;
import net.minecraft.class_2338;
import net.minecraft.class_2350;
import net.minecraft.class_2561;
import net.minecraft.class_2586;
import net.minecraft.class_2680;
import net.minecraft.class_2960;
import net.minecraft.class_8786;
import net.minecraft.class_9135;
import org.jetbrains.annotations.Nullable;
import reborncore.common.blockentity.MachineBaseBlockEntity;
import reborncore.common.blockentity.MultiblockWriter;
import reborncore.common.crafting.RebornRecipe;
import reborncore.common.crafting.SizedIngredient;
import reborncore.common.crafting.RecipeUtils;
import reborncore.common.screen.BuiltScreenHandler;
import reborncore.common.screen.BuiltScreenHandlerProvider;
import reborncore.common.screen.builder.ScreenHandlerBuilder;
import reborncore.common.util.ItemUtils;
import reborncore.common.util.RebornInventory;
import reborncore.common.util.StringUtils;
import reborncore.common.util.Torus;
import techreborn.recipe.recipes.FusionReactorRecipe;
import techreborn.blockentity.machine.GenericMachineBlockEntity;
import techreborn.config.TechRebornConfig;
import techreborn.init.ModRecipes;
import techreborn.init.TRBlockEntities;
import techreborn.init.TRContent;

public class FusionControlComputerBlockEntity extends GenericMachineBlockEntity implements BuiltScreenHandlerProvider {

	public int craftingTickTime = 0;
	public int neededPower = 0;
	public int size = 6;
	public int state = -1;
	final int topStackSlot = 0;
	final int bottomStackSlot = 1;
	final int outputStackSlot = 2;
	class_8786<FusionReactorRecipe> currentRecipeEntry = null;
	boolean hasStartedCrafting = false;
	boolean checkNBTRecipe = false;
	long lastTick = -1;

	public FusionControlComputerBlockEntity(class_2338 pos, class_2680 state) {
		super(TRBlockEntities.FUSION_CONTROL_COMPUTER, pos, state, "FusionControlComputer", -1, -1, TRContent.Machine.FUSION_CONTROL_COMPUTER.block, -1);
		this.inventory = new RebornInventory<>(3, "FusionControlComputerBlockEntity", 64, this);
	}

	@Override
	public boolean hasMultiblock() {
		return true;
	}

	public class_2561 getStateText() {
		if (state == -1) {
			return class_2561.method_43473();
		} else if (state == 0) {
			return class_2561.method_43471("gui.techreborn.fusion.norecipe");
		} else if (state == 1) {
			if (currentRecipeEntry == null) {
				return class_2561.method_43471("gui.techreborn.fusion.charging");
			}
			int percentage = percentage(currentRecipeEntry.comp_1933().getStartEnergy(), getEnergy());
			return class_2561.method_43469("gui.techreborn.fusion.chargingdetailed", StringUtils.getPercentageText(percentage));
		} else if (state == 2) {
			return class_2561.method_43471("gui.techreborn.fusion.crafting");
		}
		return class_2561.method_43473();
	}

	/**
	 * Changes size of fusion reactor ring after GUI button click
	 *
	 * @param sizeDelta {@code int} Size increment
	 */
	public void changeSize(int sizeDelta) {
		int newSize = size + sizeDelta;
		this.size = Math.max(6, Math.min(TechRebornConfig.fusionControlComputerMaxCoilSize, newSize));
	}

	/**
	 * Resets crafter progress and recipe
	 */
	private void resetCrafter() {
		currentRecipeEntry = null;
		craftingTickTime = 0;
		neededPower = 0;
		hasStartedCrafting = false;
	}

	/**
	 * Checks that {@link class_1799} could be inserted into slot provided, including check
	 * for existing item in slot and maximum stack size
	 *
	 * @param stack {@link class_1799} Stack to insert
	 * @param slot  {@code int} Slot ID to check
	 * @param tags  {@code boolean} Should we use tags
	 * @return {@code boolean} Returns true if ItemStack will fit into slot
	 */
	public boolean canFitStack(class_1799 stack, int slot, boolean tags) {// Checks to see if it can
		// fit the stack
		if (stack.method_7960()) {
			return true;
		}
		if (inventory.method_5438(slot).method_7960()) {
			return true;
		}
		if (ItemUtils.isItemEqual(inventory.method_5438(slot), stack, true, tags)) {
			return stack.method_7947() + inventory.method_5438(slot).method_7947() <= stack.method_7914();
		}
		return false;
	}

	/**
	 * Tries to set current recipe based in inputs in reactor
	 */
	private void updateCurrentRecipe() {
		for (class_8786<FusionReactorRecipe> entry : RecipeUtils.getRecipeEntries(field_11863, ModRecipes.FUSION_REACTOR)) {
			if (validateRecipe(entry)) {
				currentRecipeEntry = entry;
				craftingTickTime = 0;
				neededPower = entry.comp_1933().getStartEnergy();
				hasStartedCrafting = false;
				break;
			}
		}
	}

	/**
	 * Validates if reactor has all inputs and can output result
	 *
	 * @param entry {@link FusionReactorRecipe} Recipe to validate
	 * @return {@code boolean} True if we have all inputs and can fit output
	 */
	private boolean validateRecipe(class_8786<FusionReactorRecipe> entry) {
		FusionReactorRecipe recipe = entry.comp_1933();
		return hasAllInputs(recipe) && canFitStack(recipe.outputs().getFirst(), outputStackSlot, true);
	}

	/**
	 * Check if {@link class_2586} has all necessary inputs for recipe provided
	 *
	 * @param recipeType {@link RebornRecipe} Recipe to check inputs
	 * @return {@code boolean} True if reactor has all inputs for recipe
	 */
	private boolean hasAllInputs(RebornRecipe recipeType) {
		for (SizedIngredient ingredient : recipeType.ingredients()) {
			boolean hasItem = false;
			if (ingredient.test(inventory.method_5438(topStackSlot))
					|| ingredient.test(inventory.method_5438(bottomStackSlot))) {
				hasItem = true;
			}
			if (!hasItem) {
				return false;
			}
		}
		return true;
	}

	/**
	 * Decrease stack size on the given slot according to recipe input
	 *
	 * @param slot {@code int} Slot number
	 */
	private void useInput(int slot) {
		if (currentRecipeEntry == null) {
			return;
		}
		for (SizedIngredient ingredient : currentRecipeEntry.comp_1933().ingredients()) {
			if (ingredient.test(inventory.method_5438(slot))) {
				inventory.shrinkSlot(slot, ingredient.count());
				break;
			}
		}
	}

	private int percentage(double MaxValue, double CurrentValue) {
		if (CurrentValue == 0) {
			return 0;
		}
		return (int) ((CurrentValue * 100.0f) / MaxValue);
	}

	// GenericMachineBlockEntity
	@Override
	public int getProgressScaled(int scale) {
		if (craftingTickTime != 0 && currentRecipeEntry != null && currentRecipeEntry.comp_1933().time() != 0) {
			return craftingTickTime * scale / currentRecipeEntry.comp_1933().time();
		}
		return 0;
	}

	@Override
	public long getBaseMaxPower() {
		return Math.min((long) (TechRebornConfig.fusionControlComputerMaxEnergy * getPowerMultiplier()), Long.MAX_VALUE);
	}

	@Override
	public long getBaseMaxOutput() {
		if (!hasStartedCrafting) {
			return 0;
		}
		return TechRebornConfig.fusionControlComputerMaxOutput;
	}

	@Override
	public long getBaseMaxInput() {
		if (hasStartedCrafting) {
			return 0;
		}
		return TechRebornConfig.fusionControlComputerMaxInput;
	}

	// PowerAcceptorBlockEntity
	@Override
	public void tick(class_1937 world, class_2338 pos, class_2680 state, MachineBaseBlockEntity blockEntity) {
		super.tick(world, pos, state, blockEntity);

		if (world == null || world.method_8608()) {
			return;
		}

		// Move this to here from the nbt read method, as it now requires the world as of 1.14
		if (checkNBTRecipe) {
			checkNBTRecipe = false;
			for (class_8786<FusionReactorRecipe> entry : RecipeUtils.getRecipeEntries(world, ModRecipes.FUSION_REACTOR)) {
				if (validateRecipe(entry)) {
					this.currentRecipeEntry = entry;
				}
			}
		}

		if (lastTick == world.method_8510()) {
			// Prevent tick accelerators, blame obstinate for this.
			return;
		}
		lastTick = world.method_8510();

		// Force check every second
		if (world.method_8510() % 20 == 0) {
			inventory.setHashChanged();
		}

		if (!isShapeValid()) {
			resetCrafter();
			return;
		}

		if (currentRecipeEntry == null && inventory.hasChanged()) {
			updateCurrentRecipe();
		}

		if (currentRecipeEntry != null) {
			final FusionReactorRecipe currentRecipe = currentRecipeEntry.comp_1933();

			if (!hasStartedCrafting && !validateRecipe(currentRecipeEntry)) {
				resetCrafter();
				inventory.resetHasChanged();
				return;
			}

			if (!hasStartedCrafting) {
				// Ignition!
				if (getStored() > currentRecipe.getStartEnergy()) {
					useEnergy(currentRecipe.getStartEnergy());
					hasStartedCrafting = true;
					useInput(topStackSlot);
					useInput(bottomStackSlot);
				}
			}
			if (hasStartedCrafting && craftingTickTime < currentRecipe.time()) {
				int recipePower = currentRecipe.power();
				// Power gen
				if (recipePower > 0) {
					// Waste power if it has nowhere to go
					long power = (long) (Math.abs(currentRecipe.power()) * getPowerMultiplier());
					addEnergy(power);
					powerChange = (power);
					craftingTickTime++;
				} else if (getStored() > -recipePower) { // Power user
					setEnergy(getEnergy() + recipePower);
					craftingTickTime++;
				}
			} else if (craftingTickTime >= currentRecipe.time()) {
				class_1799 result = currentRecipe.outputs().getFirst();
				if (canFitStack(result, outputStackSlot, true)) {
					if (inventory.method_5438(outputStackSlot).method_7960()) {
						inventory.method_5447(outputStackSlot, result.method_7972());
					} else {
						inventory.shrinkSlot(outputStackSlot, -result.method_7947());
					}
					if (validateRecipe(this.currentRecipeEntry)) {
						craftingTickTime = 0;
						useInput(topStackSlot);
						useInput(bottomStackSlot);
					} else {
						resetCrafter();
					}
				}
			}
			method_5431();
		}

		inventory.resetHasChanged();
	}

	@Override
	protected boolean canAcceptEnergy(@Nullable class_2350 side) {
		// Accept from sides
		return !(side == class_2350.field_11033 || side == class_2350.field_11036);
	}

	@Override
	public boolean canProvideEnergy(@Nullable class_2350 side) {
		// Provide from top and bottom
		return side == class_2350.field_11033 || side == class_2350.field_11036;
	}

	@Override
	public void method_11014(class_11368 view) {
		super.method_11014(view);
		this.craftingTickTime = view.method_71424("craftingTickTime", 0);
		this.neededPower = view.method_71424("neededPower", 0);
		this.hasStartedCrafting = view.method_71433("hasStartedCrafting", false);
		if (view.method_71433("hasActiveRecipe", false) && this.currentRecipeEntry == null) {
			checkNBTRecipe = true;
		}
		this.size = view.method_71424("size", 0);
		//Done here to force the smaller size, will be useful if people lag out on a large one.
		this.size = Math.min(size, TechRebornConfig.fusionControlComputerMaxCoilSize);
	}

	@Override
	public void method_11007(class_11372 view) {
		super.method_11007(view);
		view.method_71465("craftingTickTime", this.craftingTickTime);
		view.method_71465("neededPower", this.neededPower);
		view.method_71472("hasStartedCrafting", this.hasStartedCrafting);
		view.method_71472("hasActiveRecipe", this.currentRecipeEntry != null);
		view.method_71465("size", size);
	}

	// MachineBaseBlockEntity
	@Override
	public double getPowerMultiplier() {
		double calc = (1F / 2F) * Math.pow(size - 5, 1.8);
		return Math.max(Math.round(calc * 100D) / 100D, 1D);
	}

	@Override
	public void writeMultiblock(MultiblockWriter writer) {
		class_2680 coil = TRContent.Machine.FUSION_COIL.block.method_9564();
		Torus.getOriginPositions(size).forEach(pos -> writer.add(pos.method_10263(), pos.method_10264(), pos.method_10260(), coil));
	}

	@Override
	public boolean canBeUpgraded() {
		return false;
	}

	// BuiltScreenHandlerProvider
	@Override
	public BuiltScreenHandler createScreenHandler(int syncID, final class_1657 player) {
		return new ScreenHandlerBuilder("fusionreactor").player(player.method_31548()).inventory().hotbar()
				.addInventory().blockEntity(this).slot(0, 34, 47).slot(1, 126, 47).outputSlot(2, 80, 47).syncEnergyValue()
				.sync(class_9135.field_49675, this::getCraftingTickTime, this::setCraftingTickTime)
				.sync(class_9135.field_49675, this::getSize, this::setSize)
				.sync(class_9135.field_49675, this::getState, this::setState)
				.sync(class_9135.field_49675, this::getNeededPower, this::setNeededPower)
				.sync(class_2960.field_48267, this::getCurrentRecipeID, this::setCurrentRecipeID)
				.syncShapeValue()
				.addInventory()
				.create(this, syncID);
	}

	public int getCraftingTickTime() {
		return craftingTickTime;
	}

	public void setCraftingTickTime(int craftingTickTime) {
		this.craftingTickTime = craftingTickTime;
	}

	public int getSize() {
		return size;
	}

	public void setSize(int size) {
		this.size = size;
	}

	public int getState() {
		if (currentRecipeEntry == null) {
			return 0; //No Recipe
		}
		if (!hasStartedCrafting) {
			return 1; //Waiting on power
		}
		return 2; //Crafting
	}

	public void setState(int state) {
		this.state = state;
	}

	public int getNeededPower() {
		return neededPower;
	}

	public void setNeededPower(int neededPower) {
		this.neededPower = neededPower;
	}

	public class_2960 getCurrentRecipeID() {
		if (currentRecipeEntry == null) {
			return class_2960.method_60655("null", "null");
		}

		return currentRecipeEntry.comp_1932().method_29177();
	}

	public void setCurrentRecipeID(class_2960 currentRecipeID) {
		if (currentRecipeID.method_12832().equals("null")) {
			currentRecipeEntry = null;
			return;
		}

		this.currentRecipeEntry = getRecipeFromID(currentRecipeID);
	}

	private class_8786<FusionReactorRecipe> getRecipeFromID(class_2960 identifier) {
		return RecipeUtils.getRecipeEntries(field_11863, ModRecipes.FUSION_REACTOR).stream()
			.filter(recipe -> recipe.comp_1932().equals(identifier))
			.findFirst()
			.orElse(null);
	}
}
