/*
 * This file is part of TechReborn, licensed under the MIT License (MIT).
 *
 * Copyright (c) 2020 TechReborn
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package techreborn.blockentity.machine.iron;

import net.minecraft.class_11368;
import net.minecraft.class_11372;
import net.minecraft.class_1303;
import net.minecraft.class_1657;
import net.minecraft.class_1799;
import net.minecraft.class_1874;
import net.minecraft.class_2338;
import net.minecraft.class_2680;
import net.minecraft.class_3222;
import net.minecraft.class_3861;
import net.minecraft.class_3956;
import net.minecraft.class_8786;
import net.minecraft.class_9135;
import net.minecraft.class_9696;
import net.minecraft.server.MinecraftServer;
import org.jetbrains.annotations.Nullable;
import reborncore.common.screen.BuiltScreenHandler;
import reborncore.common.screen.BuiltScreenHandlerProvider;
import reborncore.common.screen.builder.ScreenHandlerBuilder;
import reborncore.common.util.RebornInventory;
import techreborn.config.TechRebornConfig;
import techreborn.init.TRBlockEntities;
import techreborn.init.TRContent;

import java.util.Optional;

public class IronFurnaceBlockEntity extends AbstractIronMachineBlockEntity implements BuiltScreenHandlerProvider {

	public final static int INPUT_SLOT = 0;
	public final static int OUTPUT_SLOT = 1;
	public final static int FUEL_SLOT = 2;

	public float experience;
	private boolean previousValid = false;
	private class_1799 previousStack = class_1799.field_8037;
	private class_8786<class_3861> lastRecipe = null;
	private int recipeCookingTime = 200;

	public IronFurnaceBlockEntity(class_2338 pos, class_2680 state) {
		super(TRBlockEntities.IRON_FURNACE, pos, state, FUEL_SLOT, TRContent.Machine.IRON_FURNACE.block);
		this.inventory = new RebornInventory<>(3, "IronFurnaceBlockEntity", 64, this);
	}

	public void handleGuiInputFromClient(class_1657 playerIn) {
		if (playerIn instanceof class_3222 player) {
			int totalExperience = (int) experience;
			while (totalExperience > 0) {
				int expToDrop = class_1303.method_5918(totalExperience);
				totalExperience -= expToDrop;
				player.method_51469().method_8649(new class_1303(player.method_51469(), player.method_23317(), player.method_23318() + 0.5D, player.method_23321() + 0.5D, expToDrop));
			}
		}
		experience = 0;
	}

	@Nullable
	private class_8786<class_3861> refreshRecipe(class_1799 stack) {
		if (field_11863 == null) return lastRecipe;
		// Check the previous recipe to see if it still applies to the current inv, saves rechecking the whole recipe list
		if (lastRecipe != null && lastRecipe.comp_1933().method_64719(new class_9696(stack), field_11863)) {
			return lastRecipe;
		} else {
			MinecraftServer server = field_11863.method_8503();
			if (server == null) return lastRecipe;

			// If the previous recipe does not apply anymore, reset the progress
			progress = 0;
			class_8786<class_3861> matchingRecipe = server.method_3772().method_8132(class_3956.field_17546, new class_9696(stack), field_11863).orElse(null);
			if (matchingRecipe != null) {
				lastRecipe = matchingRecipe;
				recipeCookingTime = matchingRecipe.comp_1933().method_8167();
			} else {
				// default value for vanilla smelting recipes is 200
				recipeCookingTime = 200;
			}
		}

		return lastRecipe;
	}


	private class_1799 getResultFor(class_1799 stack) {
		if (stack.method_7960()) {
			// Fast fail if there is no input, no point checking the recipes if the machine is empty
			return class_1799.field_8037;
		}
		if (previousStack.method_31574(stack.method_7909()) && !previousValid){
			return class_1799.field_8037;
		}

		class_8786<class_3861> matchingRecipe = refreshRecipe(stack);

		if (matchingRecipe != null) {
			return matchingRecipe.comp_1933().method_59998(new class_9696(stack), method_10997().method_30349()).method_7972();
		}

		return class_1799.field_8037;
	}

	private float getExperienceFor() {
		if (field_11863 == null) return 0F;
		MinecraftServer server = field_11863.method_8503();
		if (server == null) return 0F;
		Optional<class_3861> recipe = server.method_3772().method_8132(class_3956.field_17546, new class_9696(inventory.method_5438(0)), field_11863).map(class_8786::comp_1933);
		return recipe.map(class_1874::method_8171).orElse(0F);
	}

	// AbstractIronMachineBlockEntity
	@Override
	protected void smelt() {
		if (!canSmelt()) {
			return;
		}
		class_1799 inputStack = inventory.method_5438(INPUT_SLOT);
		class_1799 resultStack = getResultFor(inputStack);

		if (inventory.method_5438(OUTPUT_SLOT).method_7960()) {
			inventory.method_5447(OUTPUT_SLOT, resultStack.method_7972());
		} else if (inventory.method_5438(OUTPUT_SLOT).method_31574(resultStack.method_7909())) {
			inventory.method_5438(OUTPUT_SLOT).method_7933(resultStack.method_7947());
		}
		experience += getExperienceFor();
		if (inputStack.method_7947() > 1) {
			inventory.shrinkSlot(INPUT_SLOT, 1);
		} else {
			inventory.method_5447(INPUT_SLOT, class_1799.field_8037);
		}
	}

	@Override
	protected boolean canSmelt() {
		class_1799 inputStack = inventory.method_5438(INPUT_SLOT);
		if (inputStack.method_7960())
			return false;
		if (previousStack != inputStack) {
			previousStack = inputStack;
			previousValid = true;
		}
		class_1799 outputStack = getResultFor(inputStack);
		if (outputStack.method_7960()) {
			previousValid = false;
			return false;
		}
		else {
			previousValid = true;
		}
		class_1799 outputSlotStack = inventory.method_5438(OUTPUT_SLOT);
		if (outputSlotStack.method_7960())
			return true;
		if (!outputSlotStack.method_31574(outputStack.method_7909()))
			return false;
		int result = outputSlotStack.method_7947() + outputStack.method_7947();
		return result <= inventory.getStackLimit() && result <= outputStack.method_7914();
	}

	@Override
	protected int cookingTime() {
		return (int) (recipeCookingTime / TechRebornConfig.cookingScale);
	}

	@Override
	public boolean isStackValid(int slotID, class_1799 stack) {
		return !getResultFor(stack).method_7960();
	}

	@Override
	public void method_11014(class_11368 view) {
		super.method_11014(view);
		experience = view.method_71423("Experience", 0);
	}

	@Override
	public void method_11007(class_11372 view) {
		super.method_11007(view);
		view.method_71464("Experience", experience);
	}

	// IContainerProvider
	public float getExperience() {
		return experience;
	}

	public void setExperience(float experience) {
		this.experience = experience;
	}

	@Override
	public int[] getInputSlots() {
		return new int[]{INPUT_SLOT};
	}

	public int getRecipeCookingTime() {
		return recipeCookingTime;
	}

	public void setRecipeCookingTime(int recipeCookingTime) {
		this.recipeCookingTime = recipeCookingTime;
	}

	@Override
	public BuiltScreenHandler createScreenHandler(int syncID, final class_1657 player) {
		return new ScreenHandlerBuilder("ironfurnace").player(player.method_31548()).inventory().hotbar()
				.addInventory().blockEntity(this)
				.fuelSlot(2, 56, 53).slot(0, 56, 17).outputSlot(1, 116, 35)
				.sync(class_9135.field_49675, this::getBurnTime, this::setBurnTime)
				.sync(class_9135.field_49675, this::getProgress, this::setProgress)
				.sync(class_9135.field_49675, this::getTotalBurnTime, this::setTotalBurnTime)
				.sync(class_9135.field_48552, this::getExperience, this::setExperience)
				.sync(class_9135.field_49675, this::getRecipeCookingTime, this::setRecipeCookingTime)
				.addInventory().create(this, syncID);
	}
}
