/*
 * This file is part of TechReborn, licensed under the MIT License (MIT).
 *
 * Copyright (c) 2020 TechReborn
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package techreborn.blockentity.machine.iron;

import reborncore.common.crafting.RebornRecipe;
import reborncore.common.crafting.SizedIngredient;
import reborncore.common.crafting.RecipeUtils;
import reborncore.common.screen.BuiltScreenHandler;
import reborncore.common.screen.BuiltScreenHandlerProvider;
import reborncore.common.screen.builder.ScreenHandlerBuilder;
import reborncore.common.util.RebornInventory;
import techreborn.config.TechRebornConfig;
import techreborn.init.ModRecipes;
import techreborn.init.TRBlockEntities;
import techreborn.init.TRContent;

import java.util.List;
import net.minecraft.class_1657;
import net.minecraft.class_1799;
import net.minecraft.class_2338;
import net.minecraft.class_2680;
import net.minecraft.class_9135;

public class IronAlloyFurnaceBlockEntity extends AbstractIronMachineBlockEntity implements BuiltScreenHandlerProvider {

	public final static int INPUT_SLOT_1 = 0;
	public final static int INPUT_SLOT_2 = 1;
	public final static int OUTPUT_SLOT = 2;
	public final static int FUEL_SLOT = 3;
	public int recipeCookingTime = 200;

	public IronAlloyFurnaceBlockEntity(class_2338 pos, class_2680 state) {
		super(TRBlockEntities.IRON_ALLOY_FURNACE, pos, state, FUEL_SLOT, TRContent.Machine.IRON_ALLOY_FURNACE.block);
		this.inventory = new RebornInventory<>(4, "IronAlloyFurnaceBlockEntity", 64, this);
	}

	public boolean hasAllInputs(RebornRecipe recipeType) {
		if (recipeType == null) {
			return false;
		}
		for (SizedIngredient ingredient : recipeType.ingredients()) {
			boolean hasItem = false;
			for (int inputSlot = 0; inputSlot < 2; inputSlot++) {
				if (ingredient.test(inventory.method_5438(inputSlot))) {
					hasItem = true;
				}
			}
			if (!hasItem)
				return false;
		}
		return true;
	}

	private RebornRecipe getRecipe() {
		for (RebornRecipe recipeType : RecipeUtils.getRecipes(field_11863, ModRecipes.ALLOY_SMELTER)) {
			if (hasAllInputs(recipeType)) {
				return recipeType;
			}
		}

		return null;
	}

	@Override
	protected boolean canSmelt() {
		if (inventory.method_5438(INPUT_SLOT_1).method_7960() || inventory.method_5438(INPUT_SLOT_2).method_7960()) {
			return false;
		}
		class_1799 itemstack = null;
		for (RebornRecipe recipeType : RecipeUtils.getRecipes(field_11863, ModRecipes.ALLOY_SMELTER)) {
			if (hasAllInputs(recipeType)) {
				List<class_1799> outputs = recipeType.outputs();
				recipeCookingTime = recipeType.time();

				if(outputs.isEmpty()){
					continue;
				}

				itemstack = outputs.get(0);

				break;
			}
		}

		if (itemstack == null)
			return false;
		if (inventory.method_5438(OUTPUT_SLOT).method_7960())
			return true;
		if (!inventory.method_5438(OUTPUT_SLOT).method_31574(itemstack.method_7909()))
			return false;
		int result = inventory.method_5438(OUTPUT_SLOT).method_7947() + itemstack.method_7947();
		return result <= inventory.getStackLimit() && result <= inventory.method_5438(OUTPUT_SLOT).method_7914();
	}

	@Override
	protected void smelt() {
		if (!canSmelt()) {
			return;
		}

		RebornRecipe currentRecipe = getRecipe();
		if (currentRecipe == null) {
			return;
		}

		class_1799 outputStack = currentRecipe.outputs().get(0);
		if (outputStack.method_7960()) {
			return;
		}
		if (inventory.method_5438(OUTPUT_SLOT).method_7960()) {
			inventory.method_5447(OUTPUT_SLOT, outputStack.method_7972());
		} else if (inventory.method_5438(OUTPUT_SLOT).method_7909() == outputStack.method_7909()) {
			inventory.shrinkSlot(OUTPUT_SLOT, -outputStack.method_7947());
		}

		for (SizedIngredient ingredient : currentRecipe.ingredients()) {
			for (int inputSlot = 0; inputSlot < 2; inputSlot++) {
				if (ingredient.test(inventory.method_5438(inputSlot))) {
					inventory.shrinkSlot(inputSlot, ingredient.count());
					break;
				}
			}
		}
	}

	@Override
	protected int cookingTime() {
		return (int) (recipeCookingTime / TechRebornConfig.cookingScale);
	}

	public int getRecipeCookingTime() {
		return recipeCookingTime;
	}

	public void setRecipeCookingTime(int recipeCookingTime) {
		this.recipeCookingTime = recipeCookingTime;
	}

	@Override
	public BuiltScreenHandler createScreenHandler(int syncID, final class_1657 player) {
		return new ScreenHandlerBuilder("alloyfurnace").player(player.method_31548()).inventory().hotbar()
				.addInventory().blockEntity(this)
				.slot(0, 47, 17)
				.slot(1, 65, 17)
				.outputSlot(2, 116, 35).fuelSlot(3, 56, 53)
				.sync(class_9135.field_49675, this::getBurnTime, this::setBurnTime)
				.sync(class_9135.field_49675, this::getProgress, this::setProgress)
				.sync(class_9135.field_49675, this::getTotalBurnTime, this::setTotalBurnTime)
				.sync(class_9135.field_49675, this::getRecipeCookingTime, this::setRecipeCookingTime)
				.addInventory().create(this, syncID);
	}

	@Override
	public boolean isStackValid(int slotID, class_1799 stack) {
		return RecipeUtils.getRecipes(field_11863, ModRecipes.ALLOY_SMELTER).stream()
				.anyMatch(rebornRecipe -> rebornRecipe.ingredients().stream()
						.anyMatch(ingredient -> ingredient.test(stack))
				);
	}

	@Override
	public int[] getInputSlots() {
		return new int[]{INPUT_SLOT_1, INPUT_SLOT_2};
	}
}
