/*
 * This file is part of TechReborn, licensed under the MIT License (MIT).
 *
 * Copyright (c) 2020 TechReborn
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package techreborn.blockentity.machine.iron;

import net.minecraft.class_11368;
import net.minecraft.class_11372;
import net.minecraft.class_1657;
import net.minecraft.class_1799;
import net.minecraft.class_1937;
import net.minecraft.class_2248;
import net.minecraft.class_2338;
import net.minecraft.class_2591;
import net.minecraft.class_2680;
import reborncore.api.IToolDrop;
import reborncore.api.blockentity.InventoryProvider;
import reborncore.common.blockentity.MachineBaseBlockEntity;
import reborncore.common.blockentity.SlotConfiguration;
import reborncore.common.blocks.BlockMachineBase;
import reborncore.common.util.RebornInventory;
import techreborn.config.TechRebornConfig;

public abstract class AbstractIronMachineBlockEntity extends MachineBaseBlockEntity implements InventoryProvider, IToolDrop, SlotConfiguration.SlotFilter {

	public RebornInventory<?> inventory;
	public int burnTime;
	public int totalBurnTime;
	public int progress;
	final int fuelSlot;
	final class_2248 toolDrop;

	public AbstractIronMachineBlockEntity(class_2591<?> blockEntityTypeIn, class_2338 pos, class_2680 state, int fuelSlot, class_2248 toolDrop) {
		super(blockEntityTypeIn, pos, state);
		this.fuelSlot = fuelSlot;
		this.toolDrop = toolDrop;
	}

	/**
	 * Checks that we have all inputs and can put output into slot
	 *
	 */
	protected abstract boolean canSmelt();

	/**
	 * Turn ingredients into the appropriate smelted
	 * item in the output slot
	 */
	protected abstract void smelt();

	/**
	 * Get the current recipe's cooking time
	 *
	 */
	protected abstract int cookingTime();

	/**
	 * Returns the number of ticks that the supplied fuel item will keep the
	 * furnace burning, or 0 if the item isn't fuel
	 *
	 * @param stack {@link class_1799} stack of fuel
	 * @return {@code int} Number of ticks
	 */
	private int getItemBurnTime(class_1799 stack) {
		if (stack.method_7960() || field_11863 == null) {
			return 0;
		}
		return (int) (field_11863.method_61269().method_61755(stack) * TechRebornConfig.fuelScale);
	}

	/**
	 * Returns remaining fraction of fuel burn time
	 *
	 * @param scale {@code int} Scale to use for burn time
	 * @return {@code int} scaled remaining fuel burn time
	 */
	public int getBurnTimeRemainingScaled(int scale) {
		if (totalBurnTime == 0) {
			return 0;
		}

		return burnTime * scale / totalBurnTime;
	}

	/**
	 * Returns crafting progress
	 *
	 * @param scale {@code int} Scale to use for crafting progress
	 * @return {@code int} Scaled crafting progress
	 */
	public int getProgressScaled(int scale) {
		if (cookingTime() > 0) {
			return progress * scale / cookingTime();
		}
		return 0;
	}

	/**
	 * Returns true if Iron Machine is burning fuel thus can do work
	 *
	 * @return {@code boolean} True if machine is burning
	 */
	public boolean isBurning() {
		return burnTime > 0;
	}

	private void updateState() {
		class_2680 state = field_11863.method_8320(field_11867);
		if (state.method_26204() instanceof BlockMachineBase blockMachineBase) {
			if (state.method_11654(BlockMachineBase.ACTIVE) != burnTime > 0)
				blockMachineBase.setActive(burnTime > 0, field_11863, field_11867);
		}
	}

	// MachineBaseBlockEntity
	@Override
	public void method_11014(class_11368 view) {
		super.method_11014(view);
		burnTime = view.method_71424("BurnTime", 0);
		totalBurnTime = view.method_71424("TotalBurnTime", 0);
		progress = view.method_71424("Progress", 0);
	}

	@Override
	public void method_11007(class_11372 view) {
		super.method_11007(view);
		view.method_71465("BurnTime", burnTime);
		view.method_71465("TotalBurnTime", totalBurnTime);
		view.method_71465("Progress", progress);
	}

	@Override
	public void tick(class_1937 world, class_2338 pos, class_2680 state, MachineBaseBlockEntity blockEntity) {
		super.tick(world, pos, state, blockEntity);
		if (world.method_8608()) {
			return;
		}
		boolean isBurning = isBurning();
		if (isBurning) {
			--burnTime;
		}

		boolean canSmelt = canSmelt();
		if (!isBurning && canSmelt) {
			burnTime = totalBurnTime = getItemBurnTime(inventory.method_5438(fuelSlot));
			if (burnTime > 0) {
				// Fuel slot
				class_1799 fuelStack = inventory.method_5438(fuelSlot);
				class_1799 remainderStack = fuelStack.method_7909().method_7858();
				if (!remainderStack.method_7960()) {
					inventory.method_5447(fuelSlot, remainderStack);
				} else if (fuelStack.method_7947() > 1) {
					inventory.shrinkSlot(fuelSlot, 1);
				} else if (fuelStack.method_7947() == 1) {
					inventory.method_5447(fuelSlot, class_1799.field_8037);
				}
			}
		}

		if (isBurning() && canSmelt) {
			++progress;
			if (progress == cookingTime()) {
				progress = 0;
				smelt();
			}
		} else if (!canSmelt) {
			progress = 0;
		}

		if (isBurning != isBurning()) {
			inventory.setHashChanged();
			updateState();
		}
		if (inventory.hasChanged()) {
			method_5431();
		}
	}

	@Override
	public boolean canBeUpgraded() {
		return false;
	}

	// InventoryProvider
	@Override
	public RebornInventory<?> getInventory() {
		return inventory;
	}

	// IToolDrop
	@Override
	public class_1799 getToolDrop(class_1657 entityPlayer) {
		return new class_1799(toolDrop);
	}

	public int getBurnTime() {
		return this.burnTime;
	}

	public void setBurnTime(int burnTime) {
		this.burnTime = burnTime;
	}

	public int getTotalBurnTime() {
		return this.totalBurnTime;
	}

	public void setTotalBurnTime(int totalBurnTime) {
		this.totalBurnTime = totalBurnTime;
	}

	public int getProgress() {
		return progress;
	}

	public void setProgress(int progress) {
		this.progress = progress;
	}
}
