/*
 * This file is part of TechReborn, licensed under the MIT License (MIT).
 *
 * Copyright (c) 2020 TechReborn
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package techreborn.blockentity.machine;

import net.minecraft.class_1657;
import net.minecraft.class_1799;
import net.minecraft.class_1937;
import net.minecraft.class_2248;
import net.minecraft.class_2338;
import net.minecraft.class_2350;
import net.minecraft.class_2586;
import net.minecraft.class_2591;
import net.minecraft.class_2680;
import org.jetbrains.annotations.Nullable;
import reborncore.api.IToolDrop;
import reborncore.api.blockentity.InventoryProvider;
import reborncore.api.recipe.IRecipeCrafterProvider;
import reborncore.common.blockentity.MachineBaseBlockEntity;
import reborncore.common.powerSystem.PowerAcceptorBlockEntity;
import reborncore.common.recipes.RecipeCrafter;
import reborncore.common.util.RebornInventory;


/**
 * @author drcrazy
 */
public abstract class GenericMachineBlockEntity extends PowerAcceptorBlockEntity
		implements IToolDrop, InventoryProvider, IRecipeCrafterProvider {

	public final String name;
	public final int maxInput;
	public final int maxEnergy;
	public final class_2248 toolDrop;
	public final int energySlot;
	public RebornInventory<?> inventory;
	public RecipeCrafter crafter;

	/**
	 * @param name       {@link String} Name for a {@link class_2586}. Do we need it at all?
	 * @param maxInput   {@code int} Maximum energy input, value in EU
	 * @param maxEnergy  {@code int} Maximum energy buffer, value in EU
	 * @param toolDrop   {@link class_2248} Block to drop with wrench
	 * @param energySlot {@code int} Energy slot to use to charge machine from battery
	 */
	public GenericMachineBlockEntity(class_2591<?> blockEntityType, class_2338 pos, class_2680 state, String name, int maxInput, int maxEnergy, class_2248 toolDrop, int energySlot) {
		super(blockEntityType, pos, state);
		this.name = "BlockEntity" + name;
		this.maxInput = maxInput;
		this.maxEnergy = maxEnergy;
		this.toolDrop = toolDrop;
		this.energySlot = energySlot;
		checkTier();
	}

	/**
	 * Returns progress scaled to input value
	 *
	 * @param scale {@code int} Maximum value for progress
	 * @return {@code int} Scale of progress
	 */
	public int getProgressScaled(int scale) {
		if (crafter != null && crafter.currentTickTime != 0 && crafter.currentNeededTicks != 0) {
			return crafter.currentTickTime * scale / crafter.currentNeededTicks;
		}
		return 0;
	}

	// PowerAcceptorBlockEntity
	@Override
	public void tick(class_1937 world, class_2338 pos, class_2680 state, MachineBaseBlockEntity blockEntity) {
		super.tick(world, pos, state, blockEntity);
		if (world == null || world.method_8608()) {
			return;
		}
		if (energySlot != -1) {
			charge(energySlot);
		}
	}

	@Override
	public long getBaseMaxPower() {
		return maxEnergy;
	}

	@Override
	public boolean canProvideEnergy(@Nullable class_2350 side) {
		return false;
	}

	@Override
	public long getBaseMaxOutput() {
		return 0;
	}

	@Override
	public long getBaseMaxInput() {
		return maxInput;
	}

	// IToolDrop
	@Override
	public class_1799 getToolDrop(class_1657 p0) {
		return new class_1799(toolDrop, 1);
	}

	// InventoryProvider
	@Override
	public RebornInventory<?> getInventory() {
		return inventory;
	}

	// IRecipeCrafterProvider
	@Override
	public RecipeCrafter getRecipeCrafter() {
		return crafter;
	}
}
