/*
 * This file is part of TechReborn, licensed under the MIT License (MIT).
 *
 * Copyright (c) 2020 TechReborn
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package techreborn.blockentity.generator.basic;

import net.minecraft.class_11368;
import net.minecraft.class_11372;
import net.minecraft.class_1657;
import net.minecraft.class_1755;
import net.minecraft.class_1799;
import net.minecraft.class_1802;
import net.minecraft.class_1937;
import net.minecraft.class_2338;
import net.minecraft.class_2350;
import net.minecraft.class_2680;
import net.minecraft.class_9135;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import reborncore.api.IToolDrop;
import reborncore.api.blockentity.InventoryProvider;
import reborncore.common.blockentity.MachineBaseBlockEntity;
import reborncore.common.blocks.BlockMachineBase;
import reborncore.common.powerSystem.PowerAcceptorBlockEntity;
import reborncore.common.screen.BuiltScreenHandler;
import reborncore.common.screen.BuiltScreenHandlerProvider;
import reborncore.common.screen.builder.ScreenHandlerBuilder;
import reborncore.common.util.RebornInventory;
import techreborn.config.TechRebornConfig;
import techreborn.init.TRBlockEntities;
import techreborn.init.TRContent;

public class SolidFuelGeneratorBlockEntity extends PowerAcceptorBlockEntity implements IToolDrop, InventoryProvider, BuiltScreenHandlerProvider {

	public final RebornInventory<SolidFuelGeneratorBlockEntity> inventory = new RebornInventory<>(2, "SolidFuelGeneratorBlockEntity", 64, this);
	public final int fuelSlot = 0;
	public int burnTime;
	public int totalBurnTime = 0;
	public boolean isBurning;
	public boolean lastTickBurning;
	class_1799 burnItem;

	public SolidFuelGeneratorBlockEntity(class_2338 pos, class_2680 state) {
		super(TRBlockEntities.SOLID_FUEL_GENERATOR, pos, state);
	}

	public static int getItemBurnTime(class_1937 world, @NotNull class_1799 stack) {
		if (stack.method_7960()) {
			return 0;
		}
		return world.method_61269().method_61755(stack) / 4;
	}

	private void updateState() {
		assert field_11863 != null;
		final class_2680 BlockStateContainer = field_11863.method_8320(field_11867);
		if (BlockStateContainer.method_26204() instanceof final BlockMachineBase blockMachineBase) {
			boolean active = burnTime > 0 && getFreeSpace() > 0.0f;
			if (BlockStateContainer.method_11654(BlockMachineBase.ACTIVE) != active) {
				blockMachineBase.setActive(active, field_11863, field_11867);
			}
		}
	}


	// PowerAcceptorBlockEntity
	@Override
	public void tick(class_1937 world, class_2338 pos, class_2680 state, MachineBaseBlockEntity blockEntity) {
		super.tick(world, pos, state, blockEntity);
		if (world == null || world.method_8608()){
			return;
		}

		discharge(1);
		if (getFreeSpace() >= TechRebornConfig.solidFuelGeneratorOutputAmount) {
			if (burnTime > 0) {
				burnTime--;
				addEnergy(TechRebornConfig.solidFuelGeneratorOutputAmount);
				isBurning = true;
			}
		} else {
			isBurning = false;
			updateState();
		}

		if (burnTime == 0) {
			updateState();
			burnTime = totalBurnTime = SolidFuelGeneratorBlockEntity.getItemBurnTime(world, inventory.method_5438(fuelSlot));
			if (burnTime > 0) {
				updateState();
				burnItem = inventory.method_5438(fuelSlot);
				if (inventory.method_5438(fuelSlot).method_7947() == 1) {
					if (inventory.method_5438(fuelSlot).method_7909() == class_1802.field_8187 || inventory.method_5438(fuelSlot).method_7909() instanceof class_1755) {
						inventory.method_5447(fuelSlot, new class_1799(class_1802.field_8550));
					} else {
						inventory.method_5447(fuelSlot, class_1799.field_8037);
					}

				} else {
					inventory.shrinkSlot(fuelSlot, 1);
				}
			}
		}

		lastTickBurning = isBurning;
	}

	@Override
	public long getBaseMaxPower() {
		return TechRebornConfig.solidFuelGeneratorMaxEnergy;
	}

	@Override
	public boolean canAcceptEnergy(@Nullable class_2350 side) {
		return false;
	}

	@Override
	public long getBaseMaxOutput() {
		return TechRebornConfig.solidFuelGeneratorMaxOutput;
	}

	@Override
	public long getBaseMaxInput() {
		return 0;
	}

	@Override
	public void method_11014(class_11368 view) {
		super.method_11014(view);
		burnTime = view.method_71424("BurnTime", 0);
		totalBurnTime = view.method_71424("TotalBurnTime", 0);
	}

	@Override
	public void method_11007(class_11372 view) {
		super.method_11007(view);
		view.method_71465("BurnTime", burnTime);
		view.method_71465("TotalBurnTime", totalBurnTime);
	}

	// MachineBaseBlockEntity
	@Override
	public boolean canBeUpgraded() {
		return false;
	}

	// IToolDrop
	@Override
	public class_1799 getToolDrop(class_1657 playerIn) {
		return TRContent.Machine.SOLID_FUEL_GENERATOR.getStack();
	}

	// InventoryProvider
	@Override
	public RebornInventory<SolidFuelGeneratorBlockEntity> getInventory() {
		return inventory;
	}

	// BuiltScreenHandlerProvider
	public int getBurnTime() {
		return burnTime;
	}

	public void setBurnTime(final int burnTime) {
		this.burnTime = burnTime;
	}

	public int getTotalBurnTime() {
		return totalBurnTime;
	}

	public void setTotalBurnTime(final int totalBurnTime) {
		this.totalBurnTime = totalBurnTime;
	}

	public int getScaledBurnTime(final int i) {
		return (int) ((float) burnTime / (float) totalBurnTime * i);
	}

	@Override
	public BuiltScreenHandler createScreenHandler(int syncID, final class_1657 player) {
		return new ScreenHandlerBuilder("generator").player(player.method_31548()).inventory().hotbar().addInventory()
				.blockEntity(this).fuelSlot(0, 80, 54).energySlot(1, 8, 72).syncEnergyValue()
				.sync(class_9135.field_49675, this::getBurnTime, this::setBurnTime)
				.sync(class_9135.field_49675, this::getTotalBurnTime, this::setTotalBurnTime).addInventory().create(this, syncID);
	}
}
