/*
 * This file is part of TechReborn, licensed under the MIT License (MIT).
 *
 * Copyright (c) 2020 TechReborn
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package techreborn.blockentity.generator;

import org.jetbrains.annotations.Nullable;
import reborncore.api.IToolDrop;
import reborncore.common.blockentity.MachineBaseBlockEntity;
import reborncore.common.blocks.BlockMachineBase;
import reborncore.common.powerSystem.PowerAcceptorBlockEntity;
import reborncore.common.powerSystem.PowerSystem;
import reborncore.common.powerSystem.RcEnergyTier;
import reborncore.common.screen.BuiltScreenHandler;
import reborncore.common.screen.BuiltScreenHandlerProvider;
import reborncore.common.screen.builder.ScreenHandlerBuilder;
import reborncore.common.util.StringUtils;
import techreborn.blocks.generator.BlockSolarPanel;
import techreborn.init.TRBlockEntities;
import techreborn.init.TRContent;
import techreborn.init.TRContent.SolarPanels;

import java.util.List;
import java.util.Objects;
import net.minecraft.class_11368;
import net.minecraft.class_124;
import net.minecraft.class_1657;
import net.minecraft.class_1799;
import net.minecraft.class_1937;
import net.minecraft.class_2248;
import net.minecraft.class_2338;
import net.minecraft.class_2350;
import net.minecraft.class_2561;
import net.minecraft.class_2586;
import net.minecraft.class_2680;
import net.minecraft.class_9135;

public class SolarPanelBlockEntity extends PowerAcceptorBlockEntity implements IToolDrop, BuiltScreenHandlerProvider {

	private boolean generating = false;

	// Range of panel between day/night production; we calculate this only when panel is updated
	private int dayNightRange = 0;

	private SolarPanels panel;

	public SolarPanelBlockEntity(class_2338 pos, class_2680 state) {
		super(TRBlockEntities.SOLAR_PANEL, pos, state);
	}

	public SolarPanelBlockEntity(class_2338 pos, class_2680 state, SolarPanels panel) {
		super(TRBlockEntities.SOLAR_PANEL, pos, state);
		this.panel = panel;
	}

	private void updatePanel() {
		Objects.requireNonNull(field_11863, "World may not be null.");

		class_2248 panelBlock = field_11863.method_8320(field_11867).method_26204();
		if (panelBlock instanceof BlockSolarPanel solarPanelBlock) {
			panel = solarPanelBlock.panelType;
		}

		dayNightRange = getPanel().generationRateD - getPanel().generationRateN;
	}

	// Setters/getters that provide boolean interface to underlying generating int; something about
	// screen auto-sync REQUIRES an integer value (booleans don't get transmitted?!), so resorted to
	// this ugly approach
	public boolean isGenerating() { return generating; }
	private void setIsGenerating(boolean isGenerating) {
		Objects.requireNonNull(field_11863, "World may not be null.");

		if (isGenerating != isGenerating()) {
			// Update block state if necessary
			field_11863.method_8501(field_11867, field_11863.method_8320(field_11867).method_11657(BlockMachineBase.ACTIVE, isGenerating));
		}
		this.generating = isGenerating;
	}

	SolarPanels getPanel() {
		if (panel == null) {
			updatePanel();
		}
		return panel;
	}

	private void updateState() {
		Objects.requireNonNull(field_11863, "World may not be null.");

		// Generation is only possible if sky is visible above us
		setIsGenerating(field_11863.method_8311(field_11867.method_10084()));
	}

	public int getGenerationRate() {
		Objects.requireNonNull(field_11863, "World may not be null.");

		if (!isGenerating()) {
			return 0;
		}

		float skyAngle = field_11863.method_30274(0);

		// Ok, we are actively generating power, but check for a few conditions that would restrict
		// the generation to minimal production...
		if (!field_11863.method_8597().comp_642() || // No light source in dimension (e.g. nether or end)
			(skyAngle > 0.25 && skyAngle < 0.75) || // Light source is below horizon
			(field_11863.method_8419() || field_11863.method_8546())) { // Weather is present
			return getPanel().generationRateN;
		}

		// At this point, we know a light source is present, and it's clear weather. We need to determine
		// the level of generation based on % of time through the day, with peak production at noon and
		// a smooth transition to night production as sun rises/sets
		float multiplier;
		if (skyAngle > 0.75) {
			// Morning to noon
			multiplier = (0.25f - (1 - skyAngle)) / 0.25f;
		} else {
			// Noon to sunset
			multiplier = (0.25f - skyAngle) / 0.25f;
		}

		return (int)Math.ceil(getPanel().generationRateN + (dayNightRange * multiplier));
	}


	// Overrides

	@Override
	public void tick(class_1937 world, class_2338 pos, class_2680 state, MachineBaseBlockEntity blockEntity) {
		super.tick(world, pos, state, blockEntity);
		if (world == null || world.method_8608()) {
			return;
		}

		if (getPanel() == TRContent.SolarPanels.CREATIVE) {
			checkOverfill = false;
			setEnergy(Integer.MAX_VALUE);
			for (class_2350 side : class_2350.values()) {
				class_2586 to = world.method_8321(pos.method_10093(side));
				if (to instanceof PowerAcceptorBlockEntity receiver) {
					if (receiver.getMaxInput(side.method_10153()) > 0){
						receiver.setStored(receiver.getMaxStoredPower());
					}
				}
			}
			return;
		}

		// State checking and updating
		if (world.method_8510() % 20 == 0) {
			checkOverfill = true;
			updateState();
		}

		// Power generation calculations
		addEnergy(getGenerationRate());
	}

	@Override
	public long getBaseMaxPower() {
		return getPanel().internalCapacity;
	}

	@Override
	protected boolean canAcceptEnergy(@Nullable class_2350 side) { return false; }

	@Override
	public long getBaseMaxOutput() {
		if (getPanel() == TRContent.SolarPanels.CREATIVE) {
			return RcEnergyTier.INSANE.getMaxOutput();
		}
		// Solar panel output will only be limited by the cables the users use
		return RcEnergyTier.EXTREME.getMaxOutput();
	}

	@Override
	public long getBaseMaxInput() {
		return 0;
	}

	@Override
	public boolean canBeUpgraded() {
		return false;
	}

	@Override
	public boolean hasSlotConfig() {
		return false;
	}

	@Override
	public RcEnergyTier getTier() {
		return getPanel().powerTier;
	}

	@Override
	public void checkTier() {
		// Nope
	}

	@Override
	public void addInfo(List<class_2561> info, boolean isReal, boolean hasData) {
		if (panel == SolarPanels.CREATIVE) {
			return;
		}

		info.add(
				class_2561.method_43471("reborncore.tooltip.energy.maxEnergy")
						.method_27692(class_124.field_1080)
						.method_27693(": ")
						.method_10852(
								class_2561.method_43470(PowerSystem.getLocalizedPower(getMaxStoredPower()))
										.method_27692(class_124.field_1065)
						)
		);

		info.add(
				class_2561.method_43471("techreborn.tooltip.generationRate.day")
						.method_27692(class_124.field_1080)
						.method_27693(": ")
						.method_10852(
								class_2561.method_43470(PowerSystem.getLocalizedPower(panel.generationRateD))
										.method_27692(class_124.field_1065)
						)
		);

		info.add(
				class_2561.method_43471("techreborn.tooltip.generationRate.night")
						.method_27692(class_124.field_1080)
						.method_27693(": ")
						.method_10852(
								class_2561.method_43470(PowerSystem.getLocalizedPower(panel.generationRateN))
										.method_27692(class_124.field_1065)
						)
		);

		info.add(
				class_2561.method_43471("reborncore.tooltip.energy.tier")
						.method_27692(class_124.field_1080)
						.method_27693(": ")
						.method_10852(
								class_2561.method_43470(StringUtils.toFirstCapitalAllLowercase(getTier().toString()))
										.method_27692(class_124.field_1065)
						)
		);
	}

	@Override
	public void method_11014(class_11368 view) {
		if (field_11863 == null) {
			// We are in BlockEntity.create method during chunk load.
			this.checkOverfill = false;
			return;
		}
		updatePanel();
		super.method_11014(view);
	}

	// MachineBaseBlockEntity
	@Override
	public void onLoad() {
		super.onLoad();
		updatePanel();
	}

	// IToolDrop
	@Override
	public class_1799 getToolDrop(final class_1657 playerIn) {
		return new class_1799(getBlockType());
	}

	@Override
	public BuiltScreenHandler createScreenHandler(int syncID, final class_1657 player) {
		return new ScreenHandlerBuilder("solar_panel").player(player.method_31548()).inventory().hotbar().addInventory()
				.blockEntity(this).syncEnergyValue()
				.sync(class_9135.field_48547, this::isGenerating, this::setIsGenerating)
				.addInventory().create(this, syncID);
	}
}
