/*
 * This file is part of TechReborn, licensed under the MIT License (MIT).
 *
 * Copyright (c) 2020 TechReborn
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package techreborn.blockentity.cable;

import net.fabricmc.fabric.api.lookup.v1.block.BlockApiCache;
import net.minecraft.class_11368;
import net.minecraft.class_11372;
import net.minecraft.class_124;
import net.minecraft.class_1657;
import net.minecraft.class_1799;
import net.minecraft.class_1937;
import net.minecraft.class_2248;
import net.minecraft.class_2338;
import net.minecraft.class_2350;
import net.minecraft.class_2487;
import net.minecraft.class_2561;
import net.minecraft.class_2586;
import net.minecraft.class_2622;
import net.minecraft.class_2680;
import net.minecraft.class_3218;
import net.minecraft.class_5558;
import net.minecraft.class_7225;
import org.jetbrains.annotations.Nullable;
import reborncore.api.IListInfoProvider;
import reborncore.api.IToolDrop;
import reborncore.common.network.NetworkManager;
import reborncore.common.network.clientbound.CustomDescriptionPayload;
import reborncore.common.powerSystem.PowerSystem;
import reborncore.common.util.StringUtils;
import team.reborn.energy.api.EnergyStorage;
import team.reborn.energy.api.base.SimpleSidedEnergyContainer;
import techreborn.blocks.cable.CableBlock;
import techreborn.init.TRBlockEntities;
import techreborn.init.TRContent;

import java.util.ArrayList;
import java.util.List;

public class CableBlockEntity extends class_2586
	implements class_5558<CableBlockEntity>, IListInfoProvider, IToolDrop {
	// Can't use SimpleEnergyStorage because the cable type is not available when the BE is constructed.
	final SimpleSidedEnergyContainer energyContainer = new SimpleSidedEnergyContainer() {
		@Override
		public long getCapacity() {
			return getCableType().transferRate * 4L;
		}

		@Override
		public long getMaxInsert(class_2350 side) {
			if (allowTransfer(side)) return getCableType().transferRate;
			else return 0;
		}

		@Override
		public long getMaxExtract(class_2350 side) {
			if (allowTransfer(side)) return getCableType().transferRate;
			else return 0;
		}
	};
	private TRContent.Cables cableType = null;
	@Nullable
	private class_2680 cover = null;
	long lastTick = 0;
	// null means that it needs to be re-queried
	List<CableTarget> targets = null;
	/**
	 * Adjacent caches, used to quickly query adjacent cable block entities.
	 */
	@SuppressWarnings("unchecked")
	private final BlockApiCache<EnergyStorage, class_2350>[] adjacentCaches = new BlockApiCache[6];
	/**
	 * Bitmask to prevent input or output into/from the cable when the cable already transferred in the target direction.
	 * This prevents double transfer rates, and back and forth between two cables.
	 */
	int blockedSides = 0;

	/**
	 * This is only used during the cable tick, whereas {@link #blockedSides} is used between ticks.
	 */
	boolean ioBlocked = false;

	public CableBlockEntity(class_2338 pos, class_2680 state) {
		super(TRBlockEntities.CABLE, pos, state);
	}

	public CableBlockEntity(class_2338 pos, class_2680 state, TRContent.Cables type) {
		super(TRBlockEntities.CABLE, pos, state);
		this.cableType = type;
	}

	TRContent.Cables getCableType() {
		if (cableType != null) {
			return cableType;
		}
		if (field_11863 == null) {
			return TRContent.Cables.COPPER;
		}
		class_2248 block = field_11863.method_8320(field_11867).method_26204();
		if (block instanceof CableBlock) {
			return ((CableBlock) block).type;
		}
		//Something has gone wrong if this happens
		return TRContent.Cables.COPPER;
	}

	private boolean allowTransfer(class_2350 side) {
		if (side == null) {
			return true;
		}

		return !ioBlocked && (blockedSides & (1 << side.ordinal())) == 0;
	}

	public EnergyStorage getSideEnergyStorage(@Nullable class_2350 side) {
		return energyContainer.getSideStorage(side);
	}

	public @Nullable class_2680 getCover() {
		return cover;
	}

	public void setCover(class_2680 cover) {
		this.cover = cover;
		if (field_11863 != null && !field_11863.method_8608()) {
			NetworkManager.sendToTracking(new CustomDescriptionPayload(method_11016(), this.method_38244(field_11863.method_30349())), this);
		}
	}

	public long getEnergy() {
		return energyContainer.amount;
	}

	public void setEnergy(long energy) {
		energyContainer.amount = energy;
	}

	private BlockApiCache<EnergyStorage, class_2350> getAdjacentCache(class_2350 direction) {
		if (adjacentCaches[direction.method_10146()] == null) {
			adjacentCaches[direction.method_10146()] = BlockApiCache.create(EnergyStorage.SIDED, (class_3218) field_11863, field_11867.method_10093(direction));
		}
		return adjacentCaches[direction.method_10146()];
	}

	@Nullable
	class_2586 getAdjacentBlockEntity(class_2350 direction) {
		return getAdjacentCache(direction).getBlockEntity();
	}

	void appendTargets(List<OfferedEnergyStorage> targetStorages) {
		class_3218 serverWorld = (class_3218) field_11863;
		if (serverWorld == null) {
			return;
		}

		// Update our targets if necessary.
		if (targets == null) {
			class_2680 newBlockState = method_11010();

			targets = new ArrayList<>();
			for (class_2350 direction : class_2350.values()) {
				boolean foundSomething = false;

				BlockApiCache<EnergyStorage, class_2350> adjCache = getAdjacentCache(direction);

				if (adjCache.getBlockEntity() instanceof CableBlockEntity adjCable) {
					if (adjCable.getCableType().transferRate == getCableType().transferRate) {
						// Make sure cables are not used as regular targets.
						foundSomething = true;
					}
				} else if (adjCache.find(direction.method_10153()) != null) {
					foundSomething = true;
					targets.add(new CableTarget(direction, adjCache));
				}

				newBlockState = newBlockState.method_11657(CableBlock.PROPERTY_MAP.get(direction), foundSomething);
			}

			serverWorld.method_8501(method_11016(), newBlockState);
		}

		// Fill the list.
		for (CableTarget target : targets) {
			EnergyStorage storage = target.find();

			if (storage == null) {
				// Schedule a rebuild next tick.
				// This is just a reference change, the iterator remains valid.
				targets = null;
			} else {
				targetStorages.add(new OfferedEnergyStorage(this, target.directionTo, storage));
			}
		}

		// Reset blocked sides.
		blockedSides = 0;
	}

	// BlockEntity
	@Override
	public class_2487 method_16887(class_7225.class_7874 registryLookup) {
		return method_38244(registryLookup);
	}

	@Override
	public class_2622 method_38235() {
		class_2487 nbtTag = new class_2487();
		// writeNbt(nbtTag);
		return class_2622.method_38585(this);
	}

	@Override
	public void method_11014(class_11368 view) {
		super.method_11014(view);
		energyContainer.amount = view.method_71425("energy", 0);
		cover = view.method_71426("cover", class_2680.field_24734).orElse(null);
	}

	@Override
	public void method_11007(class_11372 view) {
		super.method_11007(view);
		view.method_71466("energy", energyContainer.amount);
		if (cover != null) {
			view.method_71468("cover", class_2680.field_24734, cover);
		}
	}

	public void neighborUpdate() {
		targets = null;
	}

	// BlockEntityTicker
	@Override
	public void tick(class_1937 world, class_2338 pos, class_2680 state, CableBlockEntity blockEntity2) {
		if (world == null || world.method_8608()) {
			return;
		}

		CableTickManager.handleCableTick(this);
	}

	// IListInfoProvider
	@Override
	public void addInfo(List<class_2561> info, boolean isReal, boolean hasData) {
		info.add(
			class_2561.method_43471("techreborn.tooltip.transferRate")
				.method_27692(class_124.field_1080)
				.method_27693(": ")
				.method_27693(PowerSystem.getLocalizedPower(getCableType().transferRate))
				.method_27692(class_124.field_1065)
				.method_27693("/t")
		);

		info.add(
			class_2561.method_43471("techreborn.tooltip.tier")
				.method_27692(class_124.field_1080)
				.method_27693(": ")
				.method_10852(
					class_2561.method_43470(StringUtils.toFirstCapitalAllLowercase(getCableType().tier.toString()))
						.method_27692(class_124.field_1065)
				)
		);

		if (!getCableType().canKill) {
			info.add(class_2561.method_43471("techreborn.tooltip.cable.can_cover").method_27692(class_124.field_1080));
		}
	}

	// IToolDrop
	@Override
	public class_1799 getToolDrop(class_1657 playerIn) {
		return new class_1799(getCableType().block);
	}

	@Override
	public @Nullable class_2680 getRenderData() {
		return cover;
	}

	private record CableTarget(class_2350 directionTo, BlockApiCache<EnergyStorage, class_2350> cache) {

		@Nullable
		EnergyStorage find() {
			return cache.find(directionTo.method_10153());
		}
	}
}
