/*
 * This file is part of RebornCore, licensed under the MIT License (MIT).
 *
 * Copyright (c) 2021 TeamReborn
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package reborncore.common.util;

import net.fabricmc.fabric.api.transfer.v1.context.ContainerItemContext;
import net.fabricmc.fabric.api.transfer.v1.item.ItemVariant;
import net.fabricmc.fabric.api.transfer.v1.item.PlayerInventoryStorage;
import net.fabricmc.fabric.api.transfer.v1.storage.base.SingleSlotStorage;
import net.minecraft.class_1657;
import net.minecraft.class_1799;
import net.minecraft.class_2371;
import net.minecraft.class_3545;
import net.minecraft.class_9288;
import net.minecraft.class_9323;
import net.minecraft.class_9334;
import reborncore.common.powerSystem.RcEnergyItem;
import reborncore.common.recipes.IRecipeInput;
import team.reborn.energy.api.EnergyStorage;
import team.reborn.energy.api.EnergyStorageUtil;

import java.util.List;
import java.util.function.Predicate;

/**
 * Created by mark on 12/04/15.
 */
public class ItemUtils {

	public static boolean isItemEqual(class_1799 a, class_1799 b, boolean matchComponent, boolean useTags) {
		if (a.method_7960() && b.method_7960()) {
			return true;
		}
		if (a.method_7960() || b.method_7960()) {
			return false;
		}
		if (matchComponent && class_1799.method_31577(a, b)) {
			return true;
		}
		if (useTags) {

			//TODO tags
		}
		return false;
	}

	public static boolean canExtractAnyFromShulker(class_1799 shulkerStack, class_1799 targetStack) {
		//bundle method
		List<class_1799> stacks = getBlockEntityStacks(shulkerStack);

		for (class_1799 stack : stacks) {
			if (class_1799.method_31577(targetStack, stack)) {
				return true;
			}
		}
		return false;
	}
	public static int canExtractFromCachedShulker(List<class_1799> stacks, class_1799 targetStack) {
		//bundle method
		if (stacks == null) return 0;
		int defaultValue = 0;
		for (class_1799 stack : stacks) {
			if (class_1799.method_31577(targetStack, stack)) {
				defaultValue += stack.method_7947();
			}
		}
		return defaultValue;
	}

	public static boolean isStackListEmpty(List<class_1799> stacks) {
		for (class_1799 stack : stacks) {
			if (!stack.method_7960()) {
				return false;
			}
		}
		return true;
	}

	public static int extractableFromCachedShulker(List<class_1799> stacks, class_1799 targetStack, int maxAmount) {
		int extracted = 0;
		for (class_1799 stack : stacks) {
			if (stack.method_7960()) continue;
			if (class_1799.method_31577(targetStack, stack)) {
				int count = stack.method_7947();
				int toExtract = Math.min(maxAmount, count);
				stack.method_7934(toExtract);
				maxAmount -= toExtract;
				extracted += toExtract;
			}
			if (maxAmount == 0) break;
			if (maxAmount < 0) throw new AssertionError("Extracted more than required amount!");
		}
		return extracted;
	}

	public static class_3545<Integer, class_1799> extractFromShulker(class_1799 shulkerStack, class_2371<class_1799> entityStack, class_1799 targetStack, int capacity) {
		class_1799 newStack = shulkerStack.method_7972();
		if (entityStack == null) {
			return new class_3545<>(0, shulkerStack);
		}

		int extracted = extractableFromCachedShulker(entityStack, targetStack, capacity);
		if (extracted == 0) {
			return new class_3545<>(0, shulkerStack);
		}

		if (isStackListEmpty(entityStack)) {
			newStack.method_57379(class_9334.field_49622, class_9288.field_49334);
			return new class_3545<>(extracted, newStack);
		}
		newStack.method_57379(class_9334.field_49622, class_9288.method_57493(entityStack));
		return new class_3545<>(extracted, newStack);
	}

	public static class_2371<class_1799> getBlockEntityStacks(class_1799 targetStack) {
		int maxSize = 128; // theorical max is 255
		class_2371<class_1799> returnStacks = class_2371.method_10213(maxSize, class_1799.field_8037);
		targetStack.method_58695(class_9334.field_49622, class_9288.field_49334).method_57492(returnStacks);

		return returnStacks;
	}

	public static boolean isEqualIgnoreEnergy(class_1799 stack1, class_1799 stack2) {
		if (stack1 == stack2) {
			return true;
		}
		if (stack1.method_7947() != stack2.method_7947()) {
			return false;
		}
		if (class_1799.method_31577(stack1, stack2)) {
			return true;
		}
		if (stack1.method_57353() == class_9323.field_49584 || stack2.method_57353() == class_9323.field_49584) {
			return false;
		}
		class_1799 stack1Copy = stack1.method_7972();
		stack1Copy.method_57381(EnergyStorage.ENERGY_COMPONENT);
		class_1799 stack2Copy = stack2.method_7972();
		stack2Copy.method_57381(EnergyStorage.ENERGY_COMPONENT);

		return class_1799.method_31577(stack1Copy, stack2Copy);
	}

	//TODO tags
	public static boolean isInputEqual(Object input, class_1799 other, boolean matchNBT,
									boolean useTags) {
		if (input instanceof class_1799) {
			return isItemEqual((class_1799) input, other, matchNBT, useTags);
		} else if (input instanceof String) {
			//TODO tags
		} else if (input instanceof IRecipeInput) {
			List<class_1799> inputs = ((IRecipeInput) input).getAllStacks();
			for (class_1799 stack : inputs) {
				if (isItemEqual(stack, other, matchNBT, false)) {
					return true;
				}
			}
		}
		return false;
	}

	public static int getPowerForDurabilityBar(class_1799 stack) {
		if (!(stack.method_7909() instanceof RcEnergyItem energyItem)) {
			throw new UnsupportedOperationException();
		}

		return Math.round((energyItem.getStoredEnergy(stack) * 100f / energyItem.getEnergyCapacity(stack)) * 13) / 100;
	}

	public static int getColorForDurabilityBar(class_1799 stack) {
		return 0xff8006;
	}



	/**
	 * Output energy from item to other items in inventory
	 *
	 * @param player    {@link class_1657} Player having powered item
	 * @param itemStack {@link class_1799} Powered item
	 * @param maxOutput {@code int} Maximum output rate of powered item
	 */
	public static void distributePowerToInventory(class_1657 player, class_1799 itemStack, long maxOutput) {
		distributePowerToInventory(player, itemStack, maxOutput, (stack) -> true);
	}

	/**
	 * Output energy from item to other items in inventory
	 *
	 * @param player    {@link class_1657} Player having powered item
	 * @param itemStack {@link class_1799} Powered item
	 * @param maxOutput {@code int} Maximum output rate of powered item
	 * @param filter    {@link Predicate} Filter for items to output to
	 * @throws IllegalArgumentException If failed to locate stack in players inventory
	 */
	public static void distributePowerToInventory(class_1657 player, class_1799 itemStack, long maxOutput, Predicate<class_1799> filter) {
		// Locate the current stack in the player inventory.
		PlayerInventoryStorage playerInv = PlayerInventoryStorage.of(player);
		SingleSlotStorage<ItemVariant> sourceSlot = null;

		for (int i = 0; i < player.method_31548().method_5439(); i++) {
			if (player.method_31548().method_5438(i) == itemStack) {
				sourceSlot = playerInv.getSlots().get(i);
				break;
			}
		}

		if (sourceSlot == null) {
			throw new IllegalArgumentException("Failed to locate current stack in the player inventory.");
		}

		EnergyStorage sourceStorage = ContainerItemContext.ofPlayerSlot(player, sourceSlot).find(EnergyStorage.ITEM);

		if (sourceStorage == null) {
			return;
		}

		for (int i = 0; i < player.method_31548().method_5439(); i++) {
			class_1799 invStack = player.method_31548().method_5438(i);

			if (invStack.method_7960() || !filter.test(invStack)) {
				continue;
			}

			EnergyStorageUtil.move(
					sourceStorage,
					ContainerItemContext.ofPlayerSlot(player, playerInv.getSlots().get(i)).find(EnergyStorage.ITEM),
					maxOutput,
					null
			);
		}
	}
}
