/*
 * This file is part of RebornCore, licensed under the MIT License (MIT).
 *
 * Copyright (c) 2021 TeamReborn
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package reborncore.common.screen.builder;

import com.mojang.serialization.Codec;
import com.mojang.serialization.DataResult;
import org.apache.commons.lang3.Range;
import reborncore.RebornCore;
import reborncore.api.blockentity.IUpgrade;
import reborncore.api.blockentity.IUpgradeable;
import reborncore.api.recipe.IRecipeCrafterProvider;
import reborncore.common.blockentity.MachineBaseBlockEntity;
import reborncore.common.blockentity.RedstoneConfiguration;
import reborncore.common.fluid.FluidUtils;
import reborncore.common.powerSystem.PowerAcceptorBlockEntity;
import reborncore.common.screen.Syncable;
import reborncore.common.screen.slot.*;
import team.reborn.energy.api.EnergyStorageUtil;

import java.util.function.Consumer;
import java.util.function.Predicate;
import java.util.function.Supplier;
import net.minecraft.class_1263;
import net.minecraft.class_1715;
import net.minecraft.class_1799;
import net.minecraft.class_1937;
import net.minecraft.class_2487;
import net.minecraft.class_2509;
import net.minecraft.class_2520;
import net.minecraft.class_2586;
import net.minecraft.class_9129;
import net.minecraft.class_9135;
import net.minecraft.class_9139;

public class BlockEntityScreenHandlerBuilder {

	private final class_1263 inventory;
	private final class_2586 blockEntity;
	private final ScreenHandlerBuilder parent;
	private final int rangeStart;
	private final class_1937 world;

	BlockEntityScreenHandlerBuilder(final ScreenHandlerBuilder parent, final class_2586 blockEntity) {
		if (blockEntity instanceof class_1263) {
			this.inventory = (class_1263) blockEntity;
		} else {
			throw new RuntimeException(blockEntity.getClass().getName() + " is not an inventory");
		}
		this.blockEntity = blockEntity;
		this.world = blockEntity.method_10997();
		this.parent = parent;
		this.rangeStart = parent.slots.size();
		if (inventory instanceof IUpgradeable) {
			upgradeSlots((IUpgradeable) inventory);
		}
		if (blockEntity instanceof MachineBaseBlockEntity machineBaseBlockEntity) {
			sync(RedstoneConfiguration.PACKET_CODEC, machineBaseBlockEntity::getRedstoneConfiguration, machineBaseBlockEntity::setRedstoneConfiguration);
		}
	}

	public BlockEntityScreenHandlerBuilder slot(final int index, final int x, final int y) {
		this.parent.slots.add(new BaseSlot(this.inventory, index, x, y));
		return this;
	}

	public BlockEntityScreenHandlerBuilder slot(final int index, final int x, final int y, Predicate<class_1799> filter) {
		this.parent.slots.add(new BaseSlot(this.inventory, index, x, y, filter));
		return this;
	}

	public BlockEntityScreenHandlerBuilder outputSlot(final int index, final int x, final int y) {
		this.parent.slots.add(new SlotOutput(this.inventory, index, x, y));
		return this;
	}

	public BlockEntityScreenHandlerBuilder fakeSlot(final int index, final int x, final int y) {
		this.parent.slots.add(new SlotFake(this.inventory, index, x, y, false, false, Integer.MAX_VALUE));
		return this;
	}

	public BlockEntityScreenHandlerBuilder filterSlot(final int index, final int x, final int y,
													final Predicate<class_1799> filter) {
		this.parent.slots.add(new FilteredSlot(this.inventory, index, x, y).setFilter(filter));
		return this;
	}

	public BlockEntityScreenHandlerBuilder energySlot(final int index, final int x, final int y) {
		this.parent.slots.add(new FilteredSlot(this.inventory, index, x, y)
				.setFilter(EnergyStorageUtil::isEnergyStorage));
		return this;
	}

	public BlockEntityScreenHandlerBuilder fluidSlot(final int index, final int x, final int y) {
		this.parent.slots.add(new FilteredSlot(this.inventory, index, x, y).setFilter(FluidUtils::isContainer));
		return this;
	}

	public BlockEntityScreenHandlerBuilder fuelSlot(final int index, final int x, final int y) {
		this.parent.slots.add(new FilteredSlot(this.inventory, index, x, y).setFilter(
			(stack) -> this.world.method_61269().method_61752(stack)
		));
		return this;
	}

	@Deprecated
	public BlockEntityScreenHandlerBuilder upgradeSlot(final int index, final int x, final int y) {
		this.parent.slots.add(new FilteredSlot(this.inventory, index, x, y)
				.setFilter(stack -> stack.method_7909() instanceof IUpgrade));
		return this;
	}

	private BlockEntityScreenHandlerBuilder upgradeSlots(IUpgradeable upgradeable) {
		if (upgradeable.canBeUpgraded()) {
			for (int i = 0; i < upgradeable.getUpgradeSlotCount(); i++) {
				this.parent.slots.add(new UpgradeSlot(upgradeable.getUpgradeInventory(), i, -18, i * 18 + 12));
			}
		}
		return this;
	}

	/**
	 * @param <T>      The type of the block entity
	 * @param supplier {@link Supplier<T>} The supplier it can supply a variable holding in an Object.
	 *                 it will be synced with a custom packet
	 * @param setter   {@link Consumer<T>} The setter to call when the variable has been updated.
	 * @return {@link BlockEntityScreenHandlerBuilder} Inventory which will do the sync
	 */
	public <T> BlockEntityScreenHandlerBuilder sync(class_9139<? super class_9129, T> codec, Supplier<T> supplier, Consumer<T> setter) {
		this.parent.objectValues.add(new SyncedObject<>(codec, supplier, setter));
		return this;
	}

	public BlockEntityScreenHandlerBuilder sync(Syncable syncable) {
		syncable.configureSync(this::sync);
		return this;
	}

	public <T> BlockEntityScreenHandlerBuilder sync(Codec<T> codec) {
		return sync(class_9135.field_48556, () -> {
			DataResult<class_2520> dataResult = codec.encodeStart(class_2509.field_11560, (T) blockEntity);
			if (dataResult.error().isPresent()) {
				throw new RuntimeException("Failed to encode: " + dataResult.error().get().message() + " " + blockEntity);
			} else {
				return (class_2487) dataResult.result().get();
			}
		}, compoundTag -> {
			DataResult<T> dataResult = codec.parse(class_2509.field_11560, compoundTag);
			if (dataResult.error().isPresent()) {
				throw new RuntimeException("Failed to encode: " + dataResult.error().get().message() + " " + blockEntity);
			}
		});
	}

	public BlockEntityScreenHandlerBuilder syncEnergyValue() {
		if (this.blockEntity instanceof PowerAcceptorBlockEntity powerAcceptor) {
			return this.sync(class_9135.field_48551, powerAcceptor::getEnergy, powerAcceptor::setEnergy)
					.sync(class_9135.field_48551, powerAcceptor::getExtraPowerStorage, powerAcceptor::setExtraPowerStorage)
					.sync(class_9135.field_48551, powerAcceptor::getPowerChange, powerAcceptor::setPowerChange);
		}

		RebornCore.LOGGER.error(this.inventory + " is not an instance of TilePowerAcceptor! Energy cannot be synced.");
		return this;
	}

	public BlockEntityScreenHandlerBuilder syncShapeValue() {
		if (this.blockEntity instanceof MachineBaseBlockEntity baseBlockEntity) {
			return this.sync(class_9135.field_48547, baseBlockEntity::isShapeValid, baseBlockEntity::setShapeValid);
		}

		throw new IllegalStateException(this.inventory + " is not an instance of MachineBaseBlockEntity! Shape cannot be synced.");
	}

	public BlockEntityScreenHandlerBuilder syncCrafterValue() {
		if (this.blockEntity instanceof IRecipeCrafterProvider recipeCrafter) {
			return this
					.sync(class_9135.field_49675, () -> recipeCrafter.getRecipeCrafter().currentTickTime, (time) -> recipeCrafter.getRecipeCrafter().currentTickTime = time)
					.sync(class_9135.field_49675, () -> recipeCrafter.getRecipeCrafter().currentNeededTicks, (ticks) -> recipeCrafter.getRecipeCrafter().currentNeededTicks = ticks);
		}

		RebornCore.LOGGER.error(this.inventory + " is not an instance of IRecipeCrafterProvider! Craft progress cannot be synced.");
		return this;
	}

	public BlockEntityScreenHandlerBuilder onCraft(final Consumer<class_1715> onCraft) {
		this.parent.craftEvents.add(onCraft);
		return this;
	}

	public ScreenHandlerBuilder addInventory() {
		this.parent.blockEntityInventoryRanges.add(Range.of(this.rangeStart, this.parent.slots.size() - 1));
		return this.parent;
	}
}
