/*
 * This file is part of RebornCore, licensed under the MIT License (MIT).
 *
 * Copyright (c) 2021 TeamReborn
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package reborncore.common.screen;

import io.netty.buffer.Unpooled;
import it.unimi.dsi.fastutil.ints.Int2ObjectMap;
import it.unimi.dsi.fastutil.ints.Int2ObjectOpenHashMap;
import net.fabricmc.fabric.api.networking.v1.PacketByteBufs;
import net.minecraft.class_1263;
import net.minecraft.class_1657;
import net.minecraft.class_1703;
import net.minecraft.class_1712;
import net.minecraft.class_1715;
import net.minecraft.class_1735;
import net.minecraft.class_1799;
import net.minecraft.class_2338;
import net.minecraft.class_2540;
import net.minecraft.class_3917;
import net.minecraft.class_9129;
import net.minecraft.class_9139;
import org.apache.commons.lang3.Range;
import reborncore.common.blockentity.MachineBaseBlockEntity;
import reborncore.common.network.NetworkManager;
import reborncore.common.network.clientbound.ScreenHandlerUpdatePayload;
import reborncore.common.screen.builder.SyncedObject;
import reborncore.common.util.ItemUtils;
import reborncore.common.util.RangeUtil;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Consumer;
import java.util.function.Predicate;

public class BuiltScreenHandler extends class_1703 {
	private final String name;

	private final Predicate<class_1657> canInteract;
	private final List<Range<Integer>> playerSlotRanges;
	private final List<Range<Integer>> blockEntitySlotRanges;

	// Holds the SyncPair along with the last value
	private final Map<IdentifiedSyncedObject<?>, Object> syncPairCache = new HashMap<>();
	private final Int2ObjectMap<IdentifiedSyncedObject<?>> syncPairIdLookup = new Int2ObjectOpenHashMap<>();

	private List<Consumer<class_1715>> craftEvents;

	private final MachineBaseBlockEntity blockEntity;

	public BuiltScreenHandler(int syncID, final String name, final Predicate<class_1657> canInteract,
							final List<Range<Integer>> playerSlotRange,
							final List<Range<Integer>> blockEntitySlotRange, MachineBaseBlockEntity blockEntity) {
		super(null, syncID);
		this.name = name;

		this.canInteract = canInteract;

		this.playerSlotRanges = RangeUtil.joinAdjacent(playerSlotRange);
		this.blockEntitySlotRanges = RangeUtil.joinAdjacent(blockEntitySlotRange);

		this.blockEntity = blockEntity;
	}

	public void addObjectSync(final List<SyncedObject<?>> syncedObjects) {
		for (final SyncedObject<?> syncedObject : syncedObjects) {
			// Add a new sync pair to the cache with a null value
			int id = syncPairCache.size() + 1;
			var syncPair = new IdentifiedSyncedObject(syncedObject, id);
			this.syncPairCache.put(syncPair, null);
			this.syncPairIdLookup.put(id, syncPair);
		}
	}

	public void addCraftEvents(final List<Consumer<class_1715>> craftEvents) {
		this.craftEvents = craftEvents;
	}

	@Override
	public boolean method_7597(final class_1657 playerIn) {
		return this.canInteract.test(playerIn);
	}

	@Override
	public final void method_7609(final class_1263 inv) {
		if (!this.craftEvents.isEmpty()) {
			this.craftEvents.forEach(consumer -> consumer.accept((class_1715) inv));
		}
	}

	@Override
	public void method_7623() {
		super.method_7623();

		for (final class_1712 listener : field_7765) {
			sendContentUpdatePacketToListener(listener);
		}
	}

	@Override
	public void method_7596(final class_1712 listener) {
		super.method_7596(listener);

		sendContentUpdatePacketToListener(listener);
	}

	private void sendContentUpdatePacketToListener(final class_1712 listener) {
		Map<IdentifiedSyncedObject<?>, Object> updatedValues = new HashMap<>();

		this.syncPairCache.replaceAll((identifiedSyncedObject, cached) -> {
			final Object value = identifiedSyncedObject.get();

			if (!value.equals(cached)) {
				updatedValues.put(identifiedSyncedObject, value);
				return value;
			}
			return null;
		});

		if (updatedValues.isEmpty()) {
			return;
		}

		byte[] data = writeScreenHandlerData(updatedValues);
		ServerPlayerEntityScreenHandlerHelper.getServerPlayerEntity(listener)
			.ifPresent(serverPlayerEntity -> NetworkManager.sendToPlayer(new ScreenHandlerUpdatePayload(data), serverPlayerEntity));
	}

	@SuppressWarnings({"rawtypes", "unchecked"})
	private byte[] writeScreenHandlerData(Map<IdentifiedSyncedObject<?>, Object> updatedValues) {
		class_9129 byteBuf = new class_9129(PacketByteBufs.create(), blockEntity.method_10997().method_30349());

		byteBuf.method_53002(updatedValues.size());
		for (Map.Entry<IdentifiedSyncedObject<?>, Object> entry : updatedValues.entrySet()) {
			class_9139 codec = entry.getKey().object().codec();
			byteBuf.method_53002(entry.getKey().id());
			codec.encode(byteBuf, entry.getValue());
		}

		return byteBuf.array();
	}

	@SuppressWarnings({"rawtypes", "unchecked"})
	public void applyScreenHandlerData(byte[] data) {
		class_9129 byteBuf = new class_9129(new class_2540(Unpooled.wrappedBuffer(data)), blockEntity.method_10997().method_30349());
		int size = byteBuf.readInt();

		for (int i = 0; i < size; i++) {
			int id = byteBuf.readInt();
			IdentifiedSyncedObject syncedObject = syncPairIdLookup.get(id);
			Object value = syncedObject.object().codec().decode(byteBuf);
			syncedObject.set(value);
		}
	}

	@Override
	public class_1799 method_7601(final class_1657 player, final int index) {

		class_1799 originalStack = class_1799.field_8037;

		final class_1735 slot = this.field_7761.get(index);

		if (slot != null && slot.method_7681()) {

			final class_1799 stackInSlot = slot.method_7677();
			originalStack = stackInSlot.method_7972();

			boolean shifted = false;

			for (final Range<Integer> range : this.playerSlotRanges) {
				if (range.contains(index)) {

					if (this.shiftToBlockEntity(stackInSlot)) {
						shifted = true;
					}
					break;
				}
			}

			if (!shifted) {
				for (final Range<Integer> range : this.blockEntitySlotRanges) {
					if (range.contains(index)) {
						if (this.shiftToPlayer(stackInSlot)) {
							shifted = true;
						}
						break;
					}
				}
			}

			slot.method_7670(stackInSlot, originalStack);
			if (stackInSlot.method_7947() <= 0) {
				slot.method_53512(class_1799.field_8037);
			} else {
				slot.method_7668();
			}
			if (stackInSlot.method_7947() == originalStack.method_7947()) {
				return class_1799.field_8037;
			}
			slot.method_7667(player, stackInSlot);
		}
		return originalStack;

	}

	protected boolean shiftItemStack(final class_1799 stackToShift, final int start, final int end) {
		if (stackToShift.method_7960()) {
			return false;
		}
		int inCount = stackToShift.method_7947();

		// First lets see if we have the same item in a slot to merge with
		for (int slotIndex = start; stackToShift.method_7947() > 0 && slotIndex < end; slotIndex++) {
			final class_1735 slot = this.field_7761.get(slotIndex);
			final class_1799 stackInSlot = slot.method_7677();
			int maxCount = Math.min(stackToShift.method_7914(), slot.method_7675());

			if (!stackToShift.method_7960() && slot.method_7680(stackToShift)) {
				if (ItemUtils.isItemEqual(stackInSlot, stackToShift, true, false)) {
					// Got 2 stacks that need merging
					int freeStackSpace = maxCount - stackInSlot.method_7947();
					if (freeStackSpace > 0) {
						int transferAmount = Math.min(freeStackSpace, stackToShift.method_7947());
						stackInSlot.method_7933(transferAmount);
						stackToShift.method_7934(transferAmount);
					}
				}
			}
		}

		// If not lets go find the next free slot to insert our remaining stack
		for (int slotIndex = start; stackToShift.method_7947() > 0 && slotIndex < end; slotIndex++) {
			final class_1735 slot = this.field_7761.get(slotIndex);
			final class_1799 stackInSlot = slot.method_7677();

			if (stackInSlot.method_7960() && slot.method_7680(stackToShift)) {
				int maxCount = Math.min(stackToShift.method_7914(), slot.method_7675());

				int moveCount = Math.min(maxCount, stackToShift.method_7947());
				class_1799 moveStack = stackToShift.method_7972();
				moveStack.method_7939(moveCount);
				slot.method_53512(moveStack);
				stackToShift.method_7934(moveCount);
			}
		}

		// If we moved some, but still have more left over lets try again
		if (!stackToShift.method_7960() && stackToShift.method_7947() != inCount) {
			shiftItemStack(stackToShift, start, end);
		}

		return stackToShift.method_7947() != inCount;
	}

	private boolean shiftToBlockEntity(final class_1799 stackToShift) {
		if (!blockEntity.getOptionalInventory().isPresent()) {
			return false;
		}
		for (final Range<Integer> range : this.blockEntitySlotRanges) {
			if (this.shiftItemStack(stackToShift, range.getMinimum(), range.getMaximum() + 1)) {
				return true;
			}
		}
		return false;
	}

	private boolean shiftToPlayer(final class_1799 stackToShift) {
		for (final Range<Integer> range : this.playerSlotRanges) {
			if (this.shiftItemStack(stackToShift, range.getMinimum(), range.getMaximum() + 1)) {
				return true;
			}
		}
		return false;
	}

	public String getName() {
		return this.name;
	}

	@Override
	public class_1735 method_7621(class_1735 slotIn) {
		return super.method_7621(slotIn);
	}

	public MachineBaseBlockEntity getBlockEntity() {
		return blockEntity;
	}

	public class_2338 getPos() {
		return getBlockEntity().method_11016();
	}

	class_3917<BuiltScreenHandler> type = null;

	public void setType(class_3917<BuiltScreenHandler> type) {
		this.type = type;
	}

	@Override
	public class_3917<BuiltScreenHandler> method_17358() {
		return type;
	}

	private record IdentifiedSyncedObject<T>(SyncedObject<T> object, int id) {
		public T get() {
			return object.getter().get();
		}

		public void set(T value) {
			object.setter().accept(value);
		}
	}
}
