/*
 * This file is part of RebornCore, licensed under the MIT License (MIT).
 *
 * Copyright (c) 2021 TeamReborn
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package reborncore.common.recipes;

import org.jetbrains.annotations.Nullable;
import reborncore.RebornCore;
import reborncore.api.recipe.IRecipeCrafterProvider;
import reborncore.common.blocks.BlockMachineBase;
import reborncore.common.crafting.RebornRecipe;
import reborncore.common.crafting.SizedIngredient;
import reborncore.common.crafting.RecipeUtils;
import reborncore.common.powerSystem.PowerAcceptorBlockEntity;
import reborncore.common.util.ItemUtils;
import reborncore.common.util.RebornInventory;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import net.minecraft.class_11368;
import net.minecraft.class_11372;
import net.minecraft.class_1799;
import net.minecraft.class_2338;
import net.minecraft.class_2586;
import net.minecraft.class_2680;
import net.minecraft.class_3956;
import net.minecraft.class_5455;

/**
 * Use this in your blockEntity entity to craft things
 */
public class RecipeCrafter implements IUpgradeHandler {

	/**
	 * This is the recipe type to use
	 */
	public class_3956<? extends RebornRecipe> recipeType;

	/**
	 * This is the parent blockEntity
	 */
	public class_2586 blockEntity;

	/**
	 * This is the place to use the power from
	 */
	public PowerAcceptorBlockEntity energy;

	public Optional<IUpgradeHandler> parentUpgradeHandler = Optional.empty();

	/**
	 * This is the amount of inputs that the setRecipe has
	 */
	public int inputs;

	/**
	 * This is the amount of outputs that the recipe has
	 */
	public int outputs;

	/**
	 * This is the inventory to use for the crafting
	 */
	public RebornInventory<?> inventory;

	/**
	 * This is the list of the slots that the crafting logic should look for the
	 * input item stacks.
	 */
	public int[] inputSlots;

	/**
	 * This is the list for the slots that the crafting logic should look fot
	 * the output item stacks.
	 */
	public int[] outputSlots;
	public RebornRecipe currentRecipe;
	public long lastSoundTime = 0;
	private long cachedWorldTime = 0;
	public int currentTickTime = 0;
	public int currentNeededTicks = 1;// Set to 1 to stop rare crashes

	int ticksSinceLastChange;

	@Nullable
	public static ICrafterSoundHandler soundHandler = (firstRun, blockEntity) -> {
	};

	public RecipeCrafter(class_3956<? extends RebornRecipe> recipeType, class_2586 blockEntity, int inputs, int outputs, RebornInventory<?> inventory,
						int[] inputSlots, int[] outputSlots) {
		this.recipeType = recipeType;
		this.blockEntity = blockEntity;
		if (blockEntity instanceof PowerAcceptorBlockEntity powerAcceptor) {
			energy = powerAcceptor;
		}
		if (blockEntity instanceof IUpgradeHandler) {
			parentUpgradeHandler = Optional.of((IUpgradeHandler) blockEntity);
		}
		this.inputs = inputs;
		this.outputs = outputs;
		this.inventory = inventory;
		this.inputSlots = inputSlots;
		this.outputSlots = outputSlots;
		if (!(blockEntity instanceof IRecipeCrafterProvider)) {
			RebornCore.LOGGER.error(blockEntity.getClass().getName() + " does not use IRecipeCrafterProvider report this to the issue tracker!");
		}
	}

	/**
	 * Call this on the blockEntity tick
	 */
	public void updateEntity() {
		if (blockEntity.method_10997() == null || blockEntity.method_10997().method_8608()) {
			return;
		}
		ticksSinceLastChange++;
		if (cachedWorldTime == 0){
			cachedWorldTime = blockEntity.method_10997().method_8510();
		}
		cachedWorldTime++;
		// Force a has chanced every second
		if (ticksSinceLastChange == 20) {
			setInvDirty(true);
			ticksSinceLastChange = 0;
			setIsActive();
		}
		// It will now look for new recipes.
		if (currentRecipe == null && isInvDirty()) {
			updateCurrentRecipe();
		}
		if (currentRecipe != null) {
			// If it doesn't have all the inputs reset
			if (isInvDirty() && !hasAllInputs() || energy.hasMultiblock() && !energy.isShapeValid()) {
				currentRecipe = null;
				currentTickTime = 0;
				setIsActive();
			}
			// If it has reached the recipe tick time
			if (currentRecipe != null && currentTickTime >= currentNeededTicks && hasAllInputs()) {
				final List<class_1799> outputs = currentRecipe.outputs();

				boolean canGiveInvAll = true;
				// Checks to see if it can fit the output
				for (int i = 0; i < outputs.size(); i++) {
					if (!canFitOutput(outputs.get(i), outputSlots[i])) {
						canGiveInvAll = false;
					}
				}
				// The slots that have been filled
				ArrayList<Integer> filledSlots = new ArrayList<>();
				if (canGiveInvAll && currentRecipe.onCraft(blockEntity)) {
					for (int i = 0; i < outputs.size(); i++) {
						// Checks it has not been filled
						if (!filledSlots.contains(outputSlots[i])) {
							// Fills the slot with the output stack
							fitStack(outputs.get(i).method_7972(), outputSlots[i]);
							filledSlots.add(outputSlots[i]);
						}
					}
					// This uses all the inputs
					useAllInputs();
					// Reset
					currentRecipe = null;
					currentTickTime = 0;
					updateCurrentRecipe();
					// Update active state if the blockEntity isn't going to start crafting again
					if (currentRecipe == null) {
						setIsActive();
					}
				}
			} else if (currentRecipe != null && currentTickTime < currentNeededTicks) {
				long useRequirement = getEuPerTick(currentRecipe.power());
				if (energy.tryUseExact(useRequirement)) {
					currentTickTime++;
					if ((currentTickTime == 1 || currentTickTime % 20 == 0 && cachedWorldTime > lastSoundTime+ 10) && soundHandler != null && !isMuffled()) {
						lastSoundTime = cachedWorldTime;
						soundHandler.playSound(false, blockEntity);
					}
				}
			}
		}
		setInvDirty(false);
	}

	/**
	 * Checks that we have all inputs, can fit output and update max tick time and current tick time
	 */
	public void updateCurrentRecipe() {
		currentTickTime = 0;
		for (RebornRecipe recipe : RecipeUtils.getRecipes(blockEntity.method_10997(), recipeType)) {
			// This checks to see if it has all the inputs
			if (!hasAllInputs(recipe)) continue;
			if (!recipe.canCraft(blockEntity)) continue;

			final List<class_1799> outputs = recipe.outputs();

			// This checks to see if it can fit all the outputs
			boolean hasOutputSpace = true;
			for (int i = 0; i < outputs.size(); i++) {
				if (!canFitOutput(outputs.get(i), outputSlots[i])) {
					hasOutputSpace = false;
				}
			}
			if (!hasOutputSpace) continue;
			// Sets the current recipe then syncs
			setCurrentRecipe(recipe);
			this.currentNeededTicks = Math.max((int) (currentRecipe.time() * (1.0 - getSpeedMultiplier())), 1);
			setIsActive();
			return;
		}
		setCurrentRecipe(null);
		currentNeededTicks = 0;
		setIsActive();
	}

	public boolean hasAllInputs() {
		return hasAllInputs(currentRecipe);
	}

	public boolean hasAllInputs(RebornRecipe recipeType) {
		if (recipeType == null) {
			return false;
		}
		for (SizedIngredient ingredient : recipeType.ingredients()) {
			boolean hasItem = false;
			for (int slot : inputSlots) {
				if (ingredient.test(inventory.method_5438(slot))) {
					hasItem = true;
				}
			}
			if (!hasItem) {
				return false;
			}
		}
		return true;
	}

	public void useAllInputs() {
		if (currentRecipe == null) {
			return;
		}
		for (SizedIngredient ingredient : currentRecipe.ingredients()) {
			for (int inputSlot : inputSlots) {// Uses all the inputs
				if (ingredient.test(inventory.method_5438(inputSlot))) {
					inventory.shrinkSlot(inputSlot, ingredient.count());
					break;
				}
			}
		}
	}

	public boolean canFitOutput(class_1799 stack, int slot) {// Checks to see if it can fit the stack
		if (stack.method_7960()) {
			return true;
		}
		if (inventory.method_5438(slot).method_7960()) {
			return true;
		}
		if (ItemUtils.isItemEqual(inventory.method_5438(slot), stack, true, true)) {
			return stack.method_7947() + inventory.method_5438(slot).method_7947() <= stack.method_7914();
		}
		return false;
	}

	public void fitStack(class_1799 stack, int slot) {// This fits a stack into a slot
		if (stack.method_7960()) {
			return;
		}
		if (inventory.method_5438(slot).method_7960()) {// If the slot is empty set the contents
			inventory.method_5447(slot, stack);
			return;
		}
		if (class_1799.method_31577(inventory.method_5438(slot), stack)) {// If the slot has stuff in
			if (stack.method_7947() + inventory.method_5438(slot).method_7947() <= stack.method_7914()) {// Check to see if it fits
				class_1799 newStack = stack.method_7972();
				// Sets the new stack size
				newStack.method_7939(inventory.method_5438(slot).method_7947() + stack.method_7947());
				inventory.method_5447(slot, newStack);
			}
		}
	}

	public void read(class_11368 view) {
		view.method_71420("Crater").ifPresent(data -> {
			currentTickTime = data.method_71424("currentTickTime", 0);
		});

		if (blockEntity != null && blockEntity.method_10997() != null && blockEntity.method_10997().method_8608()) {
			blockEntity.method_10997().method_8413(blockEntity.method_11016(),
					blockEntity.method_10997().method_8320(blockEntity.method_11016()),
					blockEntity.method_10997().method_8320(blockEntity.method_11016()), 3);
		}
	}

	public void write(class_11372 view) {
		view.method_71461("Crater").method_71463("currentTickTime", currentTickTime);
	}

	private boolean isActive() {
		return currentRecipe != null && energy.getEnergy() >= currentRecipe.power();
	}

	public boolean canCraftAgain() {
		for (RebornRecipe recipe : RecipeUtils.getRecipes(blockEntity.method_10997(), recipeType)) {
			if (recipe.canCraft(blockEntity) && hasAllInputs(recipe)) {
				final List<class_1799> outputs = recipe.outputs();

				for (int i = 0; i < outputs.size(); i++) {
					if (!canFitOutput(outputs.get(i), outputSlots[i])) {
						return false;
					}
				}
				return !(energy.getEnergy() < recipe.power());
			}
		}
		return false;
	}

	public void setIsActive() {
		class_2338 pos = blockEntity.method_11016();
		if (blockEntity.method_10997() == null) return;
		class_2680 oldState  = blockEntity.method_10997().method_8320(pos);
		if (oldState.method_26204() instanceof BlockMachineBase blockMachineBase) {
			boolean isActive = isActive() || canCraftAgain();

			if (isActive == oldState.method_11654(BlockMachineBase.ACTIVE)) {
				return;
			}

			blockMachineBase.setActive(isActive, blockEntity.method_10997(), pos);
			blockEntity.method_10997().method_8413(pos, oldState, blockEntity.method_10997().method_8320(pos), 3);
		}
	}

	public void setCurrentRecipe(RebornRecipe recipe) {
		this.currentRecipe = recipe;
	}

	public boolean isInvDirty() {
		return inventory.hasChanged();
	}

	public void setInvDirty(boolean isDirty) {
		inventory.setHashChanged(isDirty);
	}

	public boolean isStackValidInput(class_1799 stack) {
		if (stack.method_7960()) {
			return false;
		}

		// Test with a stack with the max stack size as some independents will check the stack size.
		// A bit of a hack but should work.
		class_1799 largeStack = stack.method_7972();
		largeStack.method_7939(largeStack.method_7914());
		for (RebornRecipe recipe : RecipeUtils.getRecipes(blockEntity.method_10997(), recipeType)) {
			for (SizedIngredient ingredient : recipe.ingredients()) {
				if (ingredient.test(largeStack)) {
					return true;
				}
			}
		}
		return false;
	}

	@Override
	public void resetSpeedMultiplier() {
		parentUpgradeHandler.ifPresent(IUpgradeHandler::resetSpeedMultiplier);
	}

	@Override
	public double getSpeedMultiplier() {
		return Math.min(parentUpgradeHandler.map(IUpgradeHandler::getSpeedMultiplier).orElse(0D), 0.99);
	}

	@Override
	public void addPowerMultiplier(double amount) {
		parentUpgradeHandler.ifPresent(iUpgradeHandler -> iUpgradeHandler.addPowerMultiplier(amount));
	}

	@Override
	public void resetPowerMultiplier() {
		parentUpgradeHandler.ifPresent(IUpgradeHandler::resetPowerMultiplier);
	}

	@Override
	public double getPowerMultiplier() {
		return parentUpgradeHandler.map(IUpgradeHandler::getPowerMultiplier).orElse(1D);
	}

	@Override
	public long getEuPerTick(long baseEu) {
		long power = parentUpgradeHandler.map(iUpgradeHandler -> iUpgradeHandler.getEuPerTick(baseEu)).orElse(1L);
		return Math.min(power, energy.getMaxStoredPower());
	}

	@Override
	public void addSpeedMultiplier(double amount) {
		parentUpgradeHandler.ifPresent(iUpgradeHandler -> iUpgradeHandler.addSpeedMultiplier(amount));
	}

	@Override
	public void muffle() {
		parentUpgradeHandler.ifPresent(IUpgradeHandler::muffle);
	}

	@Override
	public void resetMuffler() {
		parentUpgradeHandler.ifPresent(IUpgradeHandler::resetMuffler);
	}

	@Override
	public boolean isMuffled() {
		return parentUpgradeHandler.map(IUpgradeHandler::isMuffled).orElse(false);
	}

	@Nullable
	private class_5455 getDynamicRegistryManager() {
		return blockEntity.method_10997().method_30349();
	}
}
