/*
 * This file is part of RebornCore, licensed under the MIT License (MIT).
 *
 * Copyright (c) 2022 TeamReborn
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package reborncore.common.powerSystem;

import net.fabricmc.fabric.api.item.v1.FabricItem;
import net.minecraft.class_1268;
import net.minecraft.class_1657;
import net.minecraft.class_1799;
import net.minecraft.class_1887;
import net.minecraft.class_1893;
import net.minecraft.class_5819;
import net.minecraft.class_6880;
import net.minecraft.class_9304;
import net.minecraft.class_9334;
import reborncore.common.util.ItemUtils;
import team.reborn.energy.api.EnergyStorage;
import team.reborn.energy.api.base.SimpleEnergyItem;


/**
 * Implement on simple energy-containing items and (on top of what {@link SimpleEnergyItem} does):
 * <ul>
 *     <li>A tooltip will be added for the item, indicating the stored power,
 *     the max power and the extraction rates.</li>
 *     <li>Any {@link RcEnergyItem} input in a crafting recipe input will automatically
 *     give its energy to the output if the output implements {@link RcEnergyItem}.</li>
 * </ul>
 * TODO: consider moving this functionality to the energy API?
 */
public interface RcEnergyItem extends SimpleEnergyItem, FabricItem {
	long getEnergyCapacity(class_1799 stack);

	/**
	 * @return {@link RcEnergyTier} the tier of this {@link EnergyStorage}, used to have standard I/O rates.
	 */
	RcEnergyTier getTier();

	default long getEnergyMaxInput(class_1799 stack) {
		return getTier().getMaxInput();
	}

	default long getEnergyMaxOutput(class_1799 stack) {
		return getTier().getMaxOutput();
	}

	@Override
	default boolean allowComponentsUpdateAnimation(class_1657 player, class_1268 hand, class_1799 oldStack, class_1799 newStack) {
		return !ItemUtils.isEqualIgnoreEnergy(oldStack, newStack);
	}

	@Override
	default boolean allowContinuingBlockBreaking(class_1657 player, class_1799 oldStack, class_1799 newStack) {
		return ItemUtils.isEqualIgnoreEnergy(oldStack, newStack);
	}

	/**
	 * Tries to use energy with honor to Unbreaking enchantment
	 *
	 * @param stack ItemStack representing item with Energy
	 * @param amount Initial amount of energy to use. This will be further decreased according to level of Unbreaking
	 *               enchantment
	 * @return Returns true if was able to use that energy from that item
	 */
	@Override
	default boolean tryUseEnergy(class_1799 stack, long amount){
		class_5819 random = class_5819.method_43047();

		int unbreakingLevel = getUnbreakingLevel(stack);
		if (unbreakingLevel > 0) {
			amount = amount / (random.method_43048(unbreakingLevel) + 1);
		}
		return SimpleEnergyItem.super.tryUseEnergy(stack, amount);
	}

	// A hack to do this without context of the DRM
	private int getUnbreakingLevel(class_1799 stack) {
		class_9304 enchantments = stack.method_58695(class_9334.field_49633, class_9304.field_49385);
		for (class_6880<class_1887> entry : enchantments.method_57534()) {
			if (entry.method_40230().equals(class_1893.field_9119)) {
				return enchantments.method_57536(entry);
			}
		}
		return 0;
	}
}
